/*====================================+=====================================+
! File CFileList_Sel.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <wx/msgdlg.h>
#include "common/sr_lib.h"
#include "CFileList.h"
#include "CDlgFilter.h"
#include "CApp.h"
/*-------------------------------------------------------------------------*/



/*--( Selection of a file )------------------------------------------------*/
int CFileList::sel( int i_item )
{
   /*----------------------------------------------------------------------*/
   if( m_dir.sel( i_item ) != 0 ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   RefreshItem( i_item ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Unselection of a file )----------------------------------------------*/
int CFileList::unsel( int i_item )
{
   /*----------------------------------------------------------------------*/
   int i_unsel_num = m_dir[ i_item ].get_sel_num() ;
   int i_sel_num ;
   /*----------------------------------------------------------------------*/
   if( m_dir.unsel( i_item ) != 0 ) { return( -1 ) ; }

   /*--( The unselected item has to be refreshed )-------------------------*/
   RefreshItem( i_item ) ;

   /*--( The selection numbers of the following items have changed )-------*/
   for( i_sel_num = i_unsel_num; i_sel_num < m_dir.get_nb_sel(); ++i_sel_num)
   {
      /*-------------------------------------------------------------------*/
      RefreshItem( m_dir.get_sel_file_num( i_sel_num ) ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Full selection (keep the current selection ) )-----------------------*/
void CFileList::sel_all()
{
   /*----------------------------------------------------------------------*/
   int i_num ;
   /*--( Something to do ? )-----------------------------------------------*/
   if( m_dir.get_nb_sel() == m_dir.size() )
   {  return ; }
   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < m_dir.size() ; ++i_num )
   {
      /*--( For the already selected files this won't do anything )--------*/
      sel( i_num ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   compute_new_names() ;
   /*----------------------------------------------------------------------*/
   wxGetApp().m_frame->gui_update_oper() ;
   /*----------------------------------------------------------------------*/
}

/*--( Full unselection )---------------------------------------------------*/
void CFileList::unsel_all()
{
   /*----------------------------------------------------------------------*/
   int i_sel_num ;
   /*--( Something to do ? )-----------------------------------------------*/
   if( m_dir.get_nb_sel() == 0 )
   {  return ; }
   /*-----------------------------------------------------------------------+
   ! The unselection is done from the end of the list to avoid useless      !
   ! computations (the unselection of a file changes the selection numbers  !
   ! of the following ones).                                                !
   +-----------------------------------------------------------------------*/
   for( i_sel_num = m_dir.get_nb_sel() - 1 ; i_sel_num >= 0  ; --i_sel_num )
   {
      /*-------------------------------------------------------------------*/
      unsel( m_dir.get_sel_file_num( i_sel_num ) ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   wxGetApp().m_frame->gui_update_oper() ;
   /*----------------------------------------------------------------------*/
}

/*--( Invert selection )---------------------------------------------------*/
void CFileList::inv_sel()
{
   /*----------------------------------------------------------------------*/
   sr::t_vec_int m_vec_i_inv_sel ;
   int           i_num ;

   /*-----------------------------------------------------------------------+
   ! The solution which seems the best solution for the inversion ...       !
   ! - keep the list of the files that will finally be selected             !
   ! - unselect all                                                         !
   ! - reselect what was planned at the fisrt step                          !
   +-----------------------------------------------------------------------*/
   for( i_num = m_dir.size() - 1 ; i_num >= 0  ; --i_num )
   {
      /*-------------------------------------------------------------------*/
      if( !m_dir[ i_num ].is_selected() && m_dir[ i_num ].is_selectable() )
      {  m_vec_i_inv_sel.push_back( i_num ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*--( Full unselection. It is done directly to avoid useless actions )--*/
   for( i_num = m_dir.get_nb_sel() - 1 ; i_num >= 0  ; --i_num )
   {  unsel( m_dir.get_sel_file_num( i_num ) ) ; }

   /*--( And reselect ... )------------------------------------------------*/
   for( i_num = m_vec_i_inv_sel.size() - 1 ; i_num >= 0 ; --i_num )
   {  sel( m_vec_i_inv_sel[ i_num ] ) ; }

   /*----------------------------------------------------------------------*/
   compute_new_names() ;
   /*----------------------------------------------------------------------*/
   wxGetApp().m_frame->gui_update_oper() ;
   /*----------------------------------------------------------------------*/
}

/*--( Selection/Unselection of highlighted files )------------------------*/
void CFileList::sel_unsel_hil( bool boo_sel )
{
   /*----------------------------------------------------------------------*/
   long l_num = -1 ;
   /*----------------------------------------------------------------------*/
   while( ( l_num = get_next_hil_item( l_num ) ) != -1 )
   {
      /*-------------------------------------------------------------------*/
      if( boo_sel )
      {  sel( l_num ) ; }
      else
      {  unsel( l_num ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   compute_new_names() ;
   /*----------------------------------------------------------------------*/
   wxGetApp().m_frame->gui_update_oper() ;
   /*----------------------------------------------------------------------*/
}

/*--( Selection based on a filter )----------------------------------------*/
int CFileList::sel_fil( const wxString &s_fil, int i_flags )
{
   /*----------------------------------------------------------------------*/
   bool boo_match = ( ( i_flags & SEARCH_FILTER_MATCH ) != 0 ) ;
   bool boo_dir   = ( ( i_flags & SEARCH_FILTER_DIR   ) != 0 ) ;
   int  i_num ;
   int  i_first_sel = -1 ;

   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < m_dir.size() ; ++i_num )
   {
      /*-------------------------------------------------------------------*/
      if(    m_dir[ i_num ].is_selectable()
          && !m_dir[ i_num ].is_selected()
          && (    !m_dir[ i_num ].is_dir()
               ||  m_dir[ i_num ].is_dir() == boo_dir
             )
          &&    boo_match
             == sr::filename_match_filter( m_dir[ i_num ].get_full_name(),
                                           s_fil
                                         )
        )
      {  /*----------------------------------------------------------------*/
         if( i_first_sel < 0 ) { i_first_sel = i_num ; }
         /*----------------------------------------------------------------*/
         sel( i_num ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( i_first_sel >= 0 ) { set_item_active( i_first_sel ) ; }
   /*----------------------------------------------------------------------*/
   compute_new_names() ;
   /*----------------------------------------------------------------------*/
   wxGetApp().m_frame->gui_update_oper() ;
   /*----------------------------------------------------------------------*/
   return( i_first_sel ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Unselection based on a filter )--------------------------------------*/
void CFileList::unsel_fil( const wxString &s_fil, int i_flags )
{
   /*----------------------------------------------------------------------*/
   bool boo_match = ( ( i_flags & SEARCH_FILTER_MATCH ) != 0 ) ;
   bool boo_dir   = ( ( i_flags & SEARCH_FILTER_DIR   ) != 0 ) ;
   int  i_num ;
   int  i_sel_num ;
   int  i_first_unsel = -1 ;

   /*-----------------------------------------------------------------------+
   ! Do it from the end to limit recomputations                             !
   +-----------------------------------------------------------------------*/
   for( i_sel_num = m_dir.get_nb_sel() - 1 ; i_sel_num >= 0 ; --i_sel_num )
   {
      /*-------------------------------------------------------------------*/
      i_num = m_dir.get_sel_file_num( i_sel_num ) ;
      /*-------------------------------------------------------------------*/
      if(    (    !m_dir[ i_num ].is_dir()
               ||  m_dir[ i_num ].is_dir() == boo_dir
             )
          &&    boo_match
             == sr::filename_match_filter( m_dir[ i_num ].get_full_name(),
                                           s_fil
                                         )
        )
      {  /*----------------------------------------------------------------*/
         i_first_unsel = i_num ;
         /*----------------------------------------------------------------*/
         unsel( i_num ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   compute_new_names() ;
   /*----------------------------------------------------------------------*/
   if( i_first_unsel >= 0 ) { set_item_active( i_first_unsel ) ; }
   /*----------------------------------------------------------------------*/
   wxGetApp().m_frame->gui_update_oper() ;
   /*----------------------------------------------------------------------*/
}

/*--( Highlight files based on a filter )----------------------------------*/
void CFileList::hil_fil( const wxString &s_fil, int i_flags )
{
   /*----------------------------------------------------------------------*/
   bool boo_match = ( ( i_flags & SEARCH_FILTER_MATCH ) != 0 ) ;
   bool boo_dir   = ( ( i_flags & SEARCH_FILTER_DIR   ) != 0 ) ;
   int  i_num            ;
   int  i_first_hil = -1 ;

   /*--( Unhighlight all )-------------------------------------------------*/
   set_item_hil( -1, false ) ;
   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < m_dir.size() ; ++i_num )
   {
      /*-------------------------------------------------------------------*/
      if(    (    !m_dir[ i_num ].is_dir()
               ||  m_dir[ i_num ].is_dir() == boo_dir
             )
          &&    boo_match
             == sr::filename_match_filter( m_dir[ i_num ].get_full_name(),
                                           s_fil
                                         )
        )
      {  /*----------------------------------------------------------------*/
         if( i_first_hil < 0 ) { i_first_hil = i_num ; }
         /*----------------------------------------------------------------*/
         set_item_hil( i_num, true ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( i_first_hil >= 0 )
   {  set_item_visible( i_first_hil ) ; set_item_focus( i_first_hil ) ; }
   /*----------------------------------------------------------------------*/
}

/*--( Highlight selected files )-------------------------------------------*/
void CFileList::hil_sel()
{
   /*----------------------------------------------------------------------*/
   if( m_dir.get_nb_sel() == 0 )
   {  return ; }
   /*--( Unhighlight all )-------------------------------------------------*/
   set_item_hil( -1, false ) ;
   /*----------------------------------------------------------------------*/
   int i_sel_num ;
   /*--( Highlight only the selected files )-------------------------------*/
   for( i_sel_num = 0 ; i_sel_num < m_dir.get_nb_sel() ; ++i_sel_num )
   {  set_item_hil( m_dir.get_sel_file_num( i_sel_num ), true ) ; }
   /*----------------------------------------------------------------------*/
   int i_num = m_dir.get_sel_file_num( 0 ) ;
   set_item_visible( i_num ) ;
   set_item_focus( i_num ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::info_search_not_found()
{
   /*----------------------------------------------------------------------*/
   wxMessageBox( wxString::Format( _( "No \"%s\" found%s"
                                      "%s \"%s\""
                                    ),
                                   get_subitem_name( COL_NONE_NAME ),
                                   SR_EOL,
                                   CDlgFilter::st_get_match_dialog_text(
                                           wxGetApp().M_boo_opfil_match.get()
                                                                       ),
                                   wxGetApp().M_s_opfil.get()
                                 ),

                 _( "Find" ),
                 wxOK | wxICON_INFORMATION
               ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Search the next file based on a filter )-----------------------------*/
void CFileList::search_fil_next( e_foc foc )
{
   /*--( Useful ? )--------------------------------------------------------*/
   if(    wxGetApp().M_s_opfil.empty()
       || !wxGetApp().m_frame->get_search_dialog_done()
     )
   {  return ; }
   /*----------------------------------------------------------------------*/
   int i_num = ( foc == FOC_FIRST_OCC ? 0 : get_item_focus() + 1 ) ;
   /*----------------------------------------------------------------------*/
   while(    i_num < m_dir.size()
          && (    (    m_dir[ i_num ].is_dir()
                    &&    m_dir[ i_num ].is_dir()
                       != wxGetApp().M_boo_opfil_dir.get()
                  )
               || (    wxGetApp().M_boo_opfil_match.get()
                    != sr::filename_match_filter(
                                              m_dir[ i_num ].get_full_name(),
                                              wxGetApp().M_s_opfil.get()
                                                )
                  )
             )
        )
   {  ++i_num ; }
   /*----------------------------------------------------------------------*/
   if( i_num < m_dir.size() )
   {  set_item_active( i_num ) ; }
   else
   {  info_search_not_found() ; }
   /*----------------------------------------------------------------------*/
}

/*--( Search the previous file based on a filter )-------------------------*/
void CFileList::search_fil_prev( e_foc foc )
{
   /*--( Useful ? )--------------------------------------------------------*/
   if(    wxGetApp().M_s_opfil.empty()
       || !wxGetApp().m_frame->get_search_dialog_done()
     )
   {  return ; }
   /*----------------------------------------------------------------------*/
   int i_num = ( foc == FOC_FIRST_OCC ? m_dir.size() : get_item_focus() ) -1;
   /*----------------------------------------------------------------------*/
   while(    i_num >= 0
          && (    (    m_dir[ i_num ].is_dir()
                    &&    m_dir[ i_num ].is_dir()
                       != wxGetApp().M_boo_opfil_dir.get()
                  )
               || (    sr::filename_match_filter(
                                              m_dir[ i_num ].get_full_name(),
                                              wxGetApp().M_s_opfil.get()
                                                )
                    != wxGetApp().M_boo_opfil_match.get()
                  )
             )
        )
   {  --i_num ; }
   /*----------------------------------------------------------------------*/
   if( i_num >= 0 )
   {  set_item_active( i_num ) ; }
   else
   {  info_search_not_found() ; }
   /*----------------------------------------------------------------------*/
}

/*--( Search the next selected file without following the selection order )*/
void CFileList::search_sel_next( e_foc foc )
{
   /*--( Useless to go further if there are no files selected )------------*/
   if( m_dir.get_nb_sel() == 0 ) { return ; }
   /*----------------------------------------------------------------------*/
   int i_num = ( foc == FOC_FIRST_OCC ? 0 : get_item_focus() + 1 ) ;
   /*----------------------------------------------------------------------*/
   while( i_num < m_dir.size() && !m_dir[ i_num ].is_selected() )
   {  ++i_num ; }
   /*----------------------------------------------------------------------*/
   if( i_num < m_dir.size() )
   {  set_item_active( i_num ) ; }
   else
   {  info_search_not_found() ; }
   /*----------------------------------------------------------------------*/
}

/*--( Search the previous selected file without following the sel order )--*/
void CFileList::search_sel_prev( e_foc foc )
{
   /*----------------------------------------------------------------------*/
   if( m_dir.get_nb_sel() == 0 ) { return ; }
   /*----------------------------------------------------------------------*/
   int i_num = ( foc == FOC_FIRST_OCC ? m_dir.size() : get_item_focus() ) -1;
   /*----------------------------------------------------------------------*/
   while( i_num >= 0 && !m_dir[ i_num ].is_selected() )
   {  --i_num ; }
   /*----------------------------------------------------------------------*/
   if( i_num >= 0 )
   {  set_item_active( i_num ) ; }
   else
   {  info_search_not_found() ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::sel_unsel_hil_following_focus()
{
   /*----------------------------------------------------------------------*/
   const CFile *p_file = get_p_file_focus() ;
   if( p_file == NULL ) { return ; }
   /*----------------------------------------------------------------------*/
   sel_unsel_hil( !p_file->is_selected() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::sel_name_array( const wxArrayString &as_file )
{
   /*----------------------------------------------------------------------*/
   int i_num       ;
   int i_num_found ;
   /*----------------------------------------------------------------------*/
   Freeze() ;
   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < ( int )as_file.GetCount() ; ++i_num )
   {
      /*-------------------------------------------------------------------*/
      if( ( i_num_found = m_dir.file_with_name( as_file[ i_num ] ) ) < 0 )
      {  continue ; }
      /*-------------------------------------------------------------------*/
      sel( i_num_found ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   compute_new_names() ;
   sort_file_list_after_selection_change() ;
   /*----------------------------------------------------------------------*/
   wxGetApp().m_frame->gui_update_oper() ;
   /*----------------------------------------------------------------------*/
   Thaw() ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Try to optimize the visibility of selected files and set the focus on the !
! first selected one                                                        !
+--------------------------------------------------------------------------*/
void CFileList::init_view_sel()
{
   /*----------------------------------------------------------------------*/
   if( m_dir.get_nb_sel() == 0 )
   {  return ; }

   /*----------------------------------------------------------------------*/
   int  i_sel_num  ;
   int  i_num      ;
   long l_max_item ;
   long l_min_item ;
   /*----------------------------------------------------------------------*/
   l_max_item = l_min_item = m_dir.get_sel_file_num( 0 ) ;
   /*----------------------------------------------------------------------*/
   for( i_sel_num = 1 ; i_sel_num < m_dir.get_nb_sel() ; ++i_sel_num )
   {
      /*-------------------------------------------------------------------*/
      i_num = m_dir.get_sel_file_num( i_sel_num ) ;
      /*-------------------------------------------------------------------*/
      if( l_min_item > i_num ) { l_min_item = i_num ; }
      if( l_max_item < i_num ) { l_max_item = i_num ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   set_item_visible( l_min_item ) ;
   set_item_visible( l_max_item ) ;
   /*--( The first selected file will be the focused one )-----------------*/
   set_item_active( m_dir.get_sel_file_num( 0 ) ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                    End of file CFileList_Sel.cpp                          !
+==========================================================================*/
