/*====================================+=====================================+
! File CFileList_Compute.cpp          ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "CFileList.h"
#include "CApp.h"
#include "CRunInfoMessage.h"
/*-------------------------------------------------------------------------*/



/*--------------------------------------------------------------------------+
! This recursive function will substitute parts of the expression given     !
! as parameter :                                                            !
! - environment variables (they can be represented as: ${...})              !
! - favourites expressions (they can be represented as: #{"fav"})           !
!                                                                           !
! Unknown elements are replaced by empty strings.                           !
!                                                                           !
! "p_as_inc_stack" will hold (if any) the favourite expr inclusion stack    !
+---------------------------------------------------------------------------+
! The substitution is done on "level 1" of the expression. This means that  !
! it won't be done inside a constant string: %f("$HOME","mytext")           !
! To be clearer, the expression associated to an element is copied as it    !
! is. This is right except for favourite expression "recursion": favourite  !
! "included" in another one ...                                             !
+--------------------------------------------------------------------------*/
int CFileList::st_preprocess_exp( const wxString &s_exp,
                                  wxString &s_res,
                                  wxArrayString *p_as_inc_stack
                                )
{
   /*----------------------------------------------------------------------*/
   t_exp_comp_info eci        ;
   wxString        s_var      ;
   wxString        s_val      ;
   CDir            dir        ;
   CFile           dummy_file ;

   /*--( Fake data )-------------------------------------------------------*/
   dir.set_fake_dir( wxGetApp().get_exe_path() ) ;
   dummy_file.set_cdir( &dir ) ;
   dir.push_back( dummy_file ) ;
   dir.sel( 0 ) ;
   /*----------------------------------------------------------------------*/
   eci.boo_first_sub_exp = true ;
   eci.it_exp = s_exp.begin()   ;
   eci.it_exp_end = s_exp.end() ;
   /*----------------------------------------------------------------------*/
   s_res.clear() ;
   s_res.reserve( s_exp.size() ) ;

   /*----------------------------------------------------------------------*/
   for( ; eci.it_exp < eci.it_exp_end ; ++eci.it_exp )
   {
      /*-------------------------------------------------------------------*/
      switch( ( *eci.it_exp ).GetValue() )
      {
         /*--( The escape character and the escaped one are copied )-------*/
         case '\\' :
            /*-------------------------------------------------------------*/
            s_res += *eci.it_exp ;
            if( eci.it_exp + 1 < eci.it_exp_end )
            {  s_res += *++eci.it_exp ; }
            /*-------------------------------------------------------------*/
            break ;

         /*--( Variable with modifiers ... etc ... )-----------------------*/
         case '%' :
         {
            /*-------------------------------------------------------------*/
            sr::wxString_cit it_exp_sav = eci.it_exp ;
            /*--------------------------------------------------------------+
            ! The result of the evaluation is not what is interesting here. !
            ! The main goal is to find out the end of the expression        !
            ! associated to the variable.                                   !
            +--------------------------------------------------------------*/
            dummy_file.compute_exp_variable( eci, s_val ) ;
            s_res.append( it_exp_sav,
                          it_exp_sav + ( eci.it_exp - it_exp_sav + 1 )
                        ) ;
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }

         /*--( An environment variable ? )---------------------------------*/
         case '$' :
         {
            /*--( The '$' is added if it is not prefixing a param string )-*/
            if( *++eci.it_exp != '{' || eci.it_exp == eci.it_exp_end )
            {  s_res += *--eci.it_exp ;
               break ;
            }
            /*-------------------------------------------------------------*/
            dummy_file.extr_param_string( eci, s_var ) ;
            /*--------------------------------------------------------------+
            ! Under Linux a value can contain unprintable chars.            !
            ! All characters are backslashed to suppress their potential    !
            ! syntactic meaning.                                            !
            +--------------------------------------------------------------*/
            if( wxGetEnv( s_var, &s_val ) && sr::clean_string( s_val ) > 0 )
            {  /*----------------------------------------------------------*/
               sr::wxString_cit it ;
               /*----------------------------------------------------------*/
               for( it = s_val.begin() ;
                       it != s_val.end()
                    && s_res.size() < ( size_t )g_co_i_string_sz_max ;
                    ++it
                  )
               {  s_res.append( '\\' ) ; s_res.append( *it ) ; }
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }

         /*--( A favourite expression ? )----------------------------------*/
         case '#' :
         {
            /*--( The '#' is added if it is not prefixing a param string )-*/
            if( *++eci.it_exp != '{' || eci.it_exp == eci.it_exp_end )
            {  s_res += *--eci.it_exp ;
               break ;
            }
            /*-------------------------------------------------------------*/
            dummy_file.extr_param_quoted_string( eci, s_var ) ;
            /*--( A favourite name has been extracted ? )------------------*/
            if( !s_var.empty() )
            {
               /*----------------------------------------------------------*/
               wxArrayString as_inc_stack ;
               /*-----------------------------------------------------------+
               ! To avoid infinite recursion, the "include stack" is kept   !
               ! and checked                                                !
               +-----------------------------------------------------------*/
               if( p_as_inc_stack == NULL )
               {  p_as_inc_stack = &as_inc_stack ; }
               /*--( Favourite expression already treated ? )--------------*/
               if( p_as_inc_stack->Index( s_var ) != wxNOT_FOUND )
               {  sr::error_message( wxString::Format(
                                          _( "Infinite include of favourite "
                                             "expression detected (\"%s\")"
                                           ),
                                          s_var
                                                     )
                                   ) ;
                  return( -1 ) ;
               }
               /*--( The favourite expression has to be preprocessed too )-*/
               p_as_inc_stack->Add( s_var ) ;
               if( st_preprocess_exp( wxGetApp().m_favourite.get_exp(s_var),
                                      s_val, p_as_inc_stack
                                    ) != 0
                 )
               {  return( -2 ) ; }
               p_as_inc_stack->RemoveAt( p_as_inc_stack->GetCount() - 1 ) ;
               /*----------------------------------------------------------*/
               s_res += s_val ;
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }

         /*--( Normal character )------------------------------------------*/
         default :
            /*-------------------------------------------------------------*/
            s_res += *eci.it_exp ;
            /*-------------------------------------------------------------*/
            break ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Some initialisations have to be done before treating files                !
+--------------------------------------------------------------------------*/
int CFileList::st_compute_init( const wxString &s_exp, wxString &s_res )
{
   /*--( External data. Done before preprocess to avoid useless reload )---*/
   CFile::st_m_ed_st.reset() ;
   CFile::st_m_ed_cp.reset() ;
   CFile::st_m_ed_fi.reset() ;

   /*--( Expand environment variables, include ... )-----------------------*/
   if( st_preprocess_exp( s_exp, s_res ) != 0 )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileList::compute_new_names()
{
   /*--( The last version of the combo transferred )-----------------------*/
   wxGetApp().m_frame->trsf_exp_combo_var() ;
   /*----------------------------------------------------------------------*/
   if( m_dir.get_nb_sel() == 0 ) { return ; }

   /*-----------------------------------------------------------------------+
   ! To try to limit screen "glitches", only the new name column is         !
   ! refreshed instead of the full item ...                                 !
   +-----------------------------------------------------------------------*/
   const int co_i_col_disp_new_name
           = get_subitem_col_disp( COL_NONE_NEW_NAME ) ;

   /*----------------------------------------------------------------------*/
   CRunInfoMessage msg( _( "Computing ..." ), _( "Computing (%d) ..." ) ) ;

   /*----------------------------------------------------------------------*/
   t_exp_comp_info  eci              ;
   wxString         s_exp            ;
   sr::wxString_cit it_sub_exp_start ;
   int              i_sel_num        ;
   int              i_num            ;
   wxRect           rect_new_name    ;

   /*--( Computation initialisations )-------------------------------------*/
   if( st_compute_init( wxGetApp().M_s_exp.get(), s_exp ) != 0 )
   {  return ; }
   eci.boo_first_sub_exp = true ;
   eci.it_exp = s_exp.begin()   ;
   eci.it_exp_end = s_exp.end() ;

   /*-----------------------------------------------------------------------+
   ! Even if the expression is empty the recomputation is done to clear the !
   ! result.
   ! It is made one sub-expression after another on all selected files.     !
   +-----------------------------------------------------------------------*/
   do
   {
      /*-------------------------------------------------------------------*/
      it_sub_exp_start = eci.it_exp ;
      /*--------------------------------------------------------------------+
      ! It is done following the selection number to manage the             !
      ! dependencies like the ones for "%nc"                                !
      +--------------------------------------------------------------------*/
      for( i_sel_num = 0 ; i_sel_num < m_dir.get_nb_sel() ; ++i_sel_num )
      {
         /*----------------------------------------------------------------*/
         msg.display( i_sel_num ) ;
         /*----------------------------------------------------------------*/
         i_num = m_dir.get_sel_file_num( i_sel_num ) ;
         /*----------------------------------------------------------------*/
         eci.it_exp = it_sub_exp_start ;
         m_dir[ i_num ].compute_sub_exp( eci ) ;
         m_dir[ i_num ].set_new_name( eci.s_current_value ) ;

         /*-----------------------------------------------------------------+
         ! The "global" modifications are only applied at the end of the    !
         ! last sub-expression.                                             !
         +-----------------------------------------------------------------*/
         if( eci.it_exp == eci.it_exp_end )
         {
            /*-------------------------------------------------------------*/
            m_dir[ i_num ].new_name_global_modification() ;
            /*-------------------------------------------------------------*/
            if(    co_i_col_disp_new_name >= 0
                && GetSubItemRect( i_num, co_i_col_disp_new_name,
                                   rect_new_name
                                 )
              )
            {  RefreshRect( rect_new_name, true ) ; }
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      eci.boo_first_sub_exp = false ;
      /*-------------------------------------------------------------------*/
   } while( eci.it_exp < eci.it_exp_end ) ;

   /*----------------------------------------------------------------------*/
   msg.reset_display() ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Before starting a global recompute this function reinitialize some data   !
! which value depend on some parameters: numbers, padding, language ...     !
+--------------------------------------------------------------------------*/
void CFileList::compute_data_and_new_names()
{
   /*----------------------------------------------------------------------*/
   if( m_dir.empty() ) { return ; }
   /*----------------------------------------------------------------------*/
   CRunInfoMessage msg( _( "Computing ..." ), _( "Computing (%d) ..." ) ) ;
   int i_num ;

   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < m_dir.size() ; ++i_num )
   {
      /*-------------------------------------------------------------------*/
      msg.display( i_num ) ;
      /*-------------------------------------------------------------------*/
      if( !m_dir[ i_num ].is_dir() ) { m_dir[ i_num ].init_info_s_track() ; }
      if( m_dir[ i_num ].is_selected() ) { m_dir[ i_num ].init_s_sel_num() ;}
      /*-------------------------------------------------------------------*/
      RefreshItem( i_num ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   msg.reset_display() ;

   /*-----------------------------------------------------------------------+
   ! Recompute because the exp can refer to what has just been modified     !
   +-----------------------------------------------------------------------*/
   compute_new_names() ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                   End of file CFileList_Compute.cpp                       !
+==========================================================================*/
