/*====================================+=====================================+
! File CFileInit_x3f.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!          Image files ".x3f": Sigma RAW (known also as "FOVb")             !
!                                                                           !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! Even if they are not really EXIF, the metadata are stored in this group   !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
struct str_x3f_header_common ;
struct str_x3f_directory_section ;
struct str_x3f_directory_entry ;
struct str_x3f_properties_header ;
struct str_x3f_properties_offset ;

/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_x3f : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_x3f( CFileInit &parent ) : CFileInit_type_base( parent )
      {  ; }
      /*-------------------------------------------------------------------*/
      int  read_le_data( str_x3f_header_common &xhc ) ;
      int  read_le_data( str_x3f_directory_section &xds ) ;
      int  read_le_data( str_x3f_directory_entry &xde ) ;
      int  read_le_data( str_x3f_properties_header &xph ) ;
      int  read_le_data( str_x3f_properties_offset &xpo ) ;
      void convert_to_col_value( int i_col, const wxString &s_val ) ;
      int  read_properties( wxFileOffset fo_offset ) ;
      int  read_directory_section( wxFileOffset fo_offset ) ;
      int  x3f_read() ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*-------------------------------------------------------------------------*/
struct str_x3f_header_common
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_file_type_id         ; // Contains "FOVb"
   wxUint32 dw_file_format_version  ;
   // cppcheck-suppress unusedStructMember
   char     ___tb_c_unique_id[ 16 ] ;
   wxUint32 ___dw_mark_bits         ;
   wxUint32 dw_width                ;
   wxUint32 dw_height               ;
   wxUint32 dw_rotation             ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_x3f_directory_section
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_id         ; // Contains "SECd"
   wxUint32 ___dw_version ;
   wxUint32 dw_nb_entries ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_x3f_directory_entry
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_offset ;
   wxUint32 __dw_size ;
   wxUint32 dw_type   ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_x3f_properties_header
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_id           ; // Contains "SECp"
   wxUint32 ___dw_version   ;
   wxUint32 dw_nb_entries   ;
   wxUint32 dw_char_format  ; // 0 = CHAR16 Unicode
   wxUint32 ___dw_reserved  ;
   wxUint32 dw_total_length ; // Total len of name/value data in characters
                              // (not bytes) => x 2 for unicode
   /*----------------------------------------------------------------------*/
} ;

/*--( From start of character data )---------------------------------------*/
struct str_x3f_properties_offset
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_offset_name  ;
   wxUint32 dw_offset_value ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*-------------------------------------------------------------------------*/
int CFileInit_x3f::read_le_data( str_x3f_header_common &xhc )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( xhc ), &xhc ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   xhc.dw_file_type_id = wxUINT32_SWAP_ALWAYS( xhc.dw_file_type_id ) ;
     xhc.dw_file_format_version
   = wxUINT32_SWAP_ALWAYS( xhc.dw_file_format_version ) ;
   xhc.dw_width    = wxUINT32_SWAP_ALWAYS( xhc.dw_width ) ;
   xhc.dw_height   = wxUINT32_SWAP_ALWAYS( xhc.dw_height ) ;
   xhc.dw_rotation = wxUINT32_SWAP_ALWAYS( xhc.dw_rotation ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_x3f::read_le_data( str_x3f_directory_section &xds )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( xds ), &xds ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   xds.dw_id         = wxUINT32_SWAP_ALWAYS( xds.dw_id ) ;
   xds.dw_nb_entries = wxUINT32_SWAP_ALWAYS( xds.dw_nb_entries ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_x3f::read_le_data( str_x3f_directory_entry &xde )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( xde ), &xde ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   xde.dw_offset = wxUINT32_SWAP_ALWAYS( xde.dw_offset ) ;
   xde.dw_type   = wxUINT32_SWAP_ALWAYS( xde.dw_type ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_x3f::read_le_data( str_x3f_properties_header &xph )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( xph ), &xph ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   xph.dw_id           = wxUINT32_SWAP_ALWAYS( xph.dw_id ) ;
   xph.dw_nb_entries   = wxUINT32_SWAP_ALWAYS( xph.dw_nb_entries ) ;
   xph.dw_char_format  = wxUINT32_SWAP_ALWAYS( xph.dw_char_format ) ;
   xph.dw_total_length = wxUINT32_SWAP_ALWAYS( xph.dw_total_length ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_x3f::read_le_data( str_x3f_properties_offset &xpo )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( xpo ), &xpo ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   xpo.dw_offset_name  = wxUINT32_SWAP_ALWAYS( xpo.dw_offset_name  ) ;
   xpo.dw_offset_value = wxUINT32_SWAP_ALWAYS( xpo.dw_offset_value ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static int st_x3f_conv_tag_col( const wxString &s_var )
{

   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "AEMODE" ) == 0 )
   {  return( COL_EXIF_EXPOSUREMODE ) ; }
   /*-----------------------------------------------------------------------+
   AFMODE AF-S Auto-focus mode ("AF-S" = single, "AF-C" =
   continuous, "MF" = manual focus)
   AP_DESC 5.6 Aperture shown on LCD at time of shot.
   +-----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "APERTURE" ) == 0 )
   {  return( COL_EXIF_APERTUREVALUE ) ; }
   /*-----------------------------------------------------------------------+
   BRACKET 3 of 9 If camera is shooting in auto-bracket mode,
   indicates the shot number in the bracket (starting
   with "1 of ___"
   BURST 3 If camera is shooting in multishot mode, indicates
   the shot number within the burst; the first shot is
   "1".
   +-----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "CAMMANUF" ) == 0 )
   {  return( COL_EXIF_MAKE ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "CAMMODEL" ) == 0 )
   {  return( COL_EXIF_MODEL ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "CAMNAME" ) == 0 )
   {  return( COL_EXIF_ARTIST ) ; }
   /*-----------------------------------------------------------------------+
   CAMSERIAL FOV3- Camera serial number.
   12345678
   DRIVE SINGLE Drive dial position ("SINGLE" = single,
   "MULTI" = multishot, "2S" = 2-sec timer, "10S"
   = 10-sec timer, "UP" = mirror-up mode, "AB" =
   auto-bracket, "OFF" = off.
   +-----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "EXPCOMP" ) == 0 )
   {  return( COL_EXIF_EXPOSUREBIASVALUE ) ; }
   /*-----------------------------------------------------------------------+
   EXPNET -1.50 Net exposure compensation for this shot (sum of
   exposure compensation and bracketing
   compensation).
   +-----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "EXPTIME" ) == 0 )
   {  return( COL_EXIF_EXPOSURETIME ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "FIRMVERS" ) == 0 )
   {  return( COL_EXIF_SOFTWARE ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "FLASH" ) == 0 )
   {  return( COL_EXIF_FLASH ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "FLENGTH" ) == 0 )
   {  return( COL_EXIF_FOCALLENGTH ) ; }
   if( s_var.CmpNoCase( "FLEQ35MM" ) == 0 )
   {  return( COL_EXIF_FOCALLENGTHIN35MMFILM ) ; }
   /*-----------------------------------------------------------------------+
   FOCUS AF Focus status ("AF" = auto-focus locked, "NO
   LOCK" = auto-focus didn't lock, "M" = manual;
   user did focusing).
   +-----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "IMAGERBOARDID" ) == 0 )
   {  return( COL_EXIF_IMAGEUNIQUEID ) ; }
   /*-----------------------------------------------------------------------+
   IMAGERTEMP I 27 28 D Temperature (Celsius) of  sensor / header board
   28 29 before and after image and before and after the
   last dark frame.  If no dark frame is present, the D
   and following numbers will be absent.
   +-----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "ISO" ) == 0 )
   {  return( COL_EXIF_ISOSPEEDRATINGS ) ; }
   /*-----------------------------------------------------------------------+
   LENSARANGE 2.8 to 32 Lens aperture range (in exact value numbers or
   LCD display numbers?  at current zoom or ?)
   LENSFRANGE 80 to 120 Lens focal length range in mm
   LENSMODEL 123 Lens identifier byte or "NONE" if no lens present.
   +-----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "PMODE" ) == 0 )
   {  return( COL_EXIF_EXPOSUREPROGRAM ) ; }
   /*-----------------------------------------------------------------------+
   RESOLUTION HI Image resolution shown on the LCD ("LOW","MED", or "HI")
   SENSORID FOV19 Rev0 : Sensor type and revision.
   SH_DESC 1/125 Shutter speed shown on LCD at time of shot.
   +-----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "SHUTTER" ) == 0 )
   {  return( COL_EXIF_SHUTTERSPEEDVALUE ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "TIME" ) == 0 )
   {  return( COL_EXIF_DATETIME ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "WB_DESC" ) == 0 )
   {  return( COL_EXIF_WHITEBALANCE ) ; }

   /*----------------------------------------------------------------------*/
   return( COL_NB ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit_x3f::convert_to_col_value( int i_col, const wxString &s_val )
{
   /*----------------------------------------------------------------------*/
   double do_val ;
   long   l_val  ;
   /*----------------------------------------------------------------------*/
   switch( i_col )
   {
      /*-------------------------------------------------------------------*/
      case COL_EXIF_EXPOSUREMODE :
         if( s_val.CmpNoCase( "8" ) == 0 )
         {  m_f.val_s( i_col ) = "8-segment" ; }
         else if( s_val.CmpNoCase( "C" ) == 0 )
         {  m_f.val_s( i_col ) = "center-weighted" ; }
         else if( s_val.CmpNoCase( "A" ) == 0 )
         {  m_f.val_s( i_col ) = "average-weighted" ; }
         else
         {  m_f.val_s( i_col ) = s_val ; }
         break ;
      /*-------------------------------------------------------------------*/
      case COL_EXIF_APERTUREVALUE :
         if( !s_val.ToCDouble( &do_val ) )
         {  m_f.val_s( i_col ) = s_val ; }
         else
         {  m_f.val_do( i_col ) = do_val ;
            m_f.val_s( i_col ).Printf( "F%.1lf", sr::round_1( do_val ) ) ;
         }
         break ;
      /*-------------------------------------------------------------------*/
      case COL_EXIF_EXPOSUREBIASVALUE :
         if( !s_val.ToCDouble( &do_val ) )
         {  m_f.val_s( i_col ) = s_val ; }
         else
         {  m_f.val_do( i_col ) = do_val ;
            m_f.val_s( i_col ).Printf( "EV%.1lf", sr::round_1( do_val ) ) ;
         }
         break ;
      /*--( Value expressed in microseconds )------------------------------*/
      case COL_EXIF_EXPOSURETIME :
         if( !s_val.ToLong( &l_val, 10 ) )
         {  m_f.val_s( i_col ) = s_val ; }
         else
         {  m_fi.init_exif_exposuretime( l_val, 1000000 ) ; }
         break ;
      /*-------------------------------------------------------------------*/
      case COL_EXIF_FLASH :
         if( s_val.CmpNoCase( "ON" ) == 0 )
         {  m_f.val_s( i_col ) = "flash" ; }
         else if( s_val.CmpNoCase( "OFF" ) == 0 )
         {  m_f.val_s( i_col ) = "no flash" ; }
         else if( s_val.CmpNoCase( "REDEYE" ) == 0 )
         {  m_f.val_s( i_col ) = "red-eye reduction mode" ; }
         else
         {  m_f.val_s( i_col ) = s_val ; }
         break ;
      /*-------------------------------------------------------------------*/
      case COL_EXIF_FOCALLENGTH :
         if( !s_val.ToLong( &l_val, 10 ) )
         {  m_f.val_s( i_col ) = s_val ; }
         else
         {  m_fi.init_exif_focallength( ( double )l_val ) ; }
         break ;
      /*-------------------------------------------------------------------*/
      case COL_EXIF_FOCALLENGTHIN35MMFILM :
         if( !s_val.ToLong( &l_val, 10 ) )
         {  m_f.val_s( i_col ) = s_val ; }
         else
         {  m_fi.init_exif_focallength35mmfilm( l_val ) ; }
         break ;
      /*-------------------------------------------------------------------*/
      case COL_EXIF_ISOSPEEDRATINGS :
         if( !s_val.ToLong( &l_val, 10 ) )
         {  m_f.val_s( i_col ) = s_val ; }
         else
         {  m_f.val_ll( i_col ) = l_val ;
            m_f.val_s( i_col ) << l_val ;
         }
         break ;
      /*-------------------------------------------------------------------*/
      case COL_EXIF_EXPOSUREPROGRAM :
         /*----------------------------------------------------------------*/
         if( s_val.CmpNoCase( "P" ) == 0 )
         {  m_f.val_s( i_col ) = "program" ; }
         else if( s_val.CmpNoCase( "A" ) == 0 )
         {  m_f.val_s( i_col ) = "aperture-priority" ; }
         else if( s_val.CmpNoCase( "S" ) == 0 )
         {  m_f.val_s( i_col ) = "shutter-priority" ; }
         else if( s_val.CmpNoCase( "M" ) == 0 )
         {  m_f.val_s( i_col ) = "manual" ; }
         else
         {  m_f.val_s( i_col ) = s_val ; }
         break ;
      /*--( Value expressed in seconds )-----------------------------------*/
      case COL_EXIF_SHUTTERSPEEDVALUE :
         if( !s_val.ToCDouble( &do_val ) )
         {  m_f.val_s( i_col ) = s_val ; }
         else
         {  m_fi.init_exif_shutterspeedvalue( do_val ) ; }
         break ;
      /*-------------------------------------------------------------------*/
      case COL_EXIF_DATETIME :
      {
         /*----------------------------------------------------------------*/
         wxDateTime dt ;
         /*----------------------------------------------------------------*/
         if(    !s_val.ToLong( &l_val, 10 )
             || sr::init_datetime( ( time_t )l_val, dt ) != 0
           )
         {  m_f.val_s( i_col ) = s_val ; }
         else
         {  m_fi.init_date( COL_EXIF_DATETIME, dt ) ; }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      default:
         m_f.val_s( i_col ) = s_val ;
         break ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_x3f::read_properties( wxFileOffset fo_offset )
{
   /*----------------------------------------------------------------------*/
   str_x3f_properties_header prop_header ;
   int                       i_num_entry ;
   str_x3f_properties_offset *p_offset   ;
   wxString                  s_var       ;
   wxString                  s_val       ;

   /*--( Read the properties header )--------------------------------------*/
   if(    m_fa.set_offset( fo_offset ) != 0
       || read_le_data( prop_header ) != 0
     )
   {  return( -1 ) ; }

   /*-----------------------------------------------------------------------+
   ! 0 is Unicode 16. Don't know the other possible values ...              !
   +-----------------------------------------------------------------------*/
   if( prop_header.dw_char_format != 0 )
   {  return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   m_fi.set_conv_from_utf16le() ;

   /*--( Some basic limitations )------------------------------------------*/
   if( prop_header.dw_nb_entries > 500 )
   {  wxFAIL ; return( -2 ) ; }
   /*--( This value is expressed in characters => length * 2 for unicode )-*/
   if( prop_header.dw_total_length > 20 * 1024 )
   {  wxFAIL ; return( -3 ) ; }

   /*----------------------------------------------------------------------*/
   wxMemoryBuffer mb_buffer_offset ;
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer(   prop_header.dw_nb_entries
                         * sizeof( str_x3f_properties_offset ),
                         mb_buffer_offset
                       ) != 0
     )
   {  return( -4 ) ; }

   /*-----------------------------------------------------------------------+
   ! Load the text block: size expressed in chars => x 2                    !
   ! The "+ 2" is here to add a last '\0' to avoid read problems during     !
   ! wxString constructions/analyse.                                        !
   +-----------------------------------------------------------------------*/
   int i_name_value_nb_bytes = prop_header.dw_total_length * 2 ;
   wxMemoryBuffer mb_buffer_name_value( i_name_value_nb_bytes + 2 ) ;
   if( m_fa.read_buffer( i_name_value_nb_bytes, mb_buffer_name_value ) != 0 )
   {  return( -5 ) ; }

   /*--( Last '\0' )-------------------------------------------------------*/
   memset(   ( wxUint8 * )mb_buffer_name_value.GetData()
           + i_name_value_nb_bytes,
           '\0', 2
         ) ;

   /*--( And parse the couples name/value )--------------------------------*/
   for( i_num_entry = 0,
        p_offset = ( str_x3f_properties_offset * )mb_buffer_offset.GetData();
        i_num_entry < ( long )prop_header.dw_nb_entries ;
        ++i_num_entry, ++p_offset
      )
   {
      /*--( Minimal control )----------------------------------------------*/
      if(    p_offset->dw_offset_name  >= prop_header.dw_total_length
          || p_offset->dw_offset_value >= prop_header.dw_total_length
        )
      {  continue ; }

      /*--------------------------------------------------------------------+
      ! As written in the x3f doc, a couple can appear many times and the   !
      ! "right" one is the last.                                            !
      ! That's why there is no "reservation" here.                          !
      +--------------------------------------------------------------------*/

      /*--( Name extraction )----------------------------------------------*/
      if( m_fi.prepare_string(   (const char *)mb_buffer_name_value.GetData()
                               + 2 * p_offset->dw_offset_name, 64, s_var
                             ) <= 0
        )
      {  continue ; }

      /*--( And try to find a correspondence to internal col ? )-----------*/
      int i_col = st_x3f_conv_tag_col( s_var ) ;
      if( i_col == COL_NB ) { continue ; }

      /*--------------------------------------------------------------------+
      ! Name is ok, the value can now be extracted and convert/store it     !
      +--------------------------------------------------------------------*/
      if( m_fi.prepare_string(   (const char *)mb_buffer_name_value.GetData()
                               + 2 * p_offset->dw_offset_value,
                               g_co_i_string_sz_max * 2,
                               s_val
                             ) > 0
        )
      {  convert_to_col_value( i_col, s_val ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_x3f::read_directory_section( wxFileOffset fo_offset )
{
   /*----------------------------------------------------------------------*/
   str_x3f_directory_section dir_section       ;
   str_x3f_directory_entry   dir_entry = { 0 } ;  // Init for compil warning
   bool                      boo_found = false ;

   /*----------------------------------------------------------------------*/
   if(    m_fa.set_offset( fo_offset ) != 0
       || read_le_data( dir_section ) != 0
     )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   if( dir_section.dw_id != SR_FOURCC( 'S','E','C','d' ) )
   {  return( -2 ) ; }

   /*--( Looking for the "properties" )------------------------------------*/
   while( !boo_found && dir_section.dw_nb_entries > 0 )
   {
      /*-------------------------------------------------------------------*/
      if( read_le_data( dir_entry ) != 0 )
      {  return( -3 ) ; }

      /*-------------------------------------------------------------------*/
      boo_found = ( dir_entry.dw_type == SR_FOURCC( 'P','R','O','P' ) ) ;
      /*-------------------------------------------------------------------*/
      --dir_section.dw_nb_entries ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( boo_found )
   {  return( read_properties( dir_entry.dw_offset ) ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_x3f::x3f_read()
{
   /*----------------------------------------------------------------------*/
   str_x3f_header_common header_common ;
   wxUint32              dw_dir_offset ;
   wxString              s_val         ;

   /*----------------------------------------------------------------------*/
   if( read_le_data( header_common ) != 0 )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   if( header_common.dw_file_type_id != SR_FOURCC( 'F','O','V','b' ) )
   {  return( -2 ) ; }

   /*----------------------------------------------------------------------*/
   m_fi.init_img_x_y( header_common.dw_width, header_common.dw_height ) ;

   /*----------------------------------------------------------------------*/
   switch( header_common.dw_rotation )
   {
      /*-------------------------------------------------------------------*/
      case   0 : m_fi.init_exif_rotation( IV_ROTATION_0 )   ; break ;
      case  90 : m_fi.init_exif_rotation( IV_ROTATION_90 )  ; break ;
      case 180 : m_fi.init_exif_rotation( IV_ROTATION_180 ) ; break ;
      case   6 : m_fi.init_exif_rotation( IV_ROTATION_270 ) ; break ;
      /*-------------------------------------------------------------------*/
      default : m_fi.init_exif_rotation( IV_ROTATION_NB )   ; break ;
      /*-------------------------------------------------------------------*/
   }

   /*--( Load the final part of the header. It depends on the version )----*/
   if(  header_common.dw_file_format_version >= 0x00020001 )
   {
      /*-------------------------------------------------------------------*/
      char tb_c_header_210_white_balance_string[ 32 ] ;
      /*-------------------------------------------------------------------*/
      if( m_fa.read_buffer( sizeof( tb_c_header_210_white_balance_string ),
                            tb_c_header_210_white_balance_string
                          ) != 0
        )
      {  return( -3 ) ; }
      /*-------------------------------------------------------------------*/
      if( m_fi.prepare_string( tb_c_header_210_white_balance_string,
                               sizeof( tb_c_header_210_white_balance_string),
                               s_val
                             ) > 0
        )
      {  m_f.val_s( COL_EXIF_WHITEBALANCE ) = s_val ; }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if(    m_fa.set_offset_from_end( sizeof( dw_dir_offset ) ) != 0
       || m_fa.read_le_data( dw_dir_offset ) != 0
     )
   {  return( -3 ) ; }

   /*----------------------------------------------------------------------*/
   if( read_directory_section( dw_dir_offset ) != 0 )
   {  return( -4 ) ; }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_x3f()
{
   /*----------------------------------------------------------------------*/
   m_s_type_det = "x3f" ;
   /*----------------------------------------------------------------------*/
   return( CFileInit_x3f( *this ).x3f_read() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                     End of file CFileInit_x3f.cpp                         !
+==========================================================================*/
