/*====================================+=====================================+
! File CFileInit_tiff.cpp             ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                      TIFF Annotation of image files                       !
!                                                                           !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! Used in standard files: tif, nef ...                                      !
! but also the exif (tif in jpg)                                            !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
#include "CApp.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
struct str_tiff_ifd_entry ;

/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_tiff : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_tiff( CFileInit &parent )
                    : CFileInit_type_base( parent ),
                      m_p_c_type_det( NULL ),
                      m_p_fnc_sw( NULL ),     // for cppcheck
                      m_p_fnc_uw( NULL ),     // for cppcheck
                      m_p_fnc_sdw( NULL ),    // for cppcheck
                      m_p_fnc_udw( NULL )     // for cppcheck
      {  m_fi.set_conv_from_8bit() ; }

   /*----------------------------------------------------------------------*/
   public :
      /*--( Data specific to the tif/exif analyse )------------------------*/
      wxFileOffset m_fo_offset ;
      wxULongLong  m_ull_size  ;
      /*-------------------------------------------------------------------*/
      const char *m_p_c_type_det ;
      /*-------------------------------------------------------------------*/
      wxInt16  ( *m_p_fnc_sw  )( wxInt16  sw  ) ;
      wxUint16 ( *m_p_fnc_uw  )( wxUint16 uw  ) ;
      wxInt32  ( *m_p_fnc_sdw )( wxInt32  sdw ) ;
      wxUint32 ( *m_p_fnc_udw )( wxUint32 udw ) ;
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      bool ok_to_read( wxFileOffset fo_offset, wxULongLong ull_size ) const
      {  return( fo_offset >= 0 && fo_offset + ull_size <= m_ull_size ) ; }
      /*-------------------------------------------------------------------*/
      void string_tab_urat( int i_nb, wxUint32 *p_urat, wxString &s_val
                          ) const ;
      void string_tab_pos_gps( int i_nb, wxUint32 *p_urat, wxString &s_val
                             ) const ;
      void string_tab_tms( int i_nb, wxUint32 *p_urat, wxString &s_val
                         ) const ;
      /*-------------------------------------------------------------------*/
      int read_tag( const int ifd_tag, const int i_col,
                    const str_tiff_ifd_entry &ifd_entry
                  ) ;
      int read_ifd( wxFileOffset fo_offset, int i_tcg ) ;
      int tiff_init( wxFileOffset fo_offset, wxULongLong ull_size ) ;
      int read_offset_ifd( wxFileOffset &fo_offset_ifd ) ;
      /*-------------------------------------------------------------------*/
      int read_exif( wxFileOffset fo_offset, wxULongLong ull_size ) ;
      int read_tiff( wxFileOffset fo_offset, wxULongLong ull_size ) ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*--------------------------------------------------------------------------+
! As the file can be BE or LE, the "read_??_data" method are not used.      !
! The conversion is directly done here.                                     !
+--------------------------------------------------------------------------*/
struct str_tiff_header
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_align ;
   wxUint16 w_mark  ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_tiff_offset_ifd
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_offset ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_tiff_ifd_header
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_nb ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_tiff_ifd_entry
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_tag      ;
   wxUint16 w_type     ;
   wxUint32 dw_nb_elmt ;
   wxUint32 dw_off_val ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*-------------------------------------------------------------------------*/
enum enu_tiff_tag_type
{
   /*----------------------------------------------------------------------*/
   TT_UBYTE  =  1, TT_SBYTE  =  6, TT_STRING = 2, TT_UNDEF  = 7,
   TT_USHORT =  3, TT_SSHORT =  8,
   TT_ULONG  =  4, TT_SLONG  =  9,
   TT_URAT   =  5, TT_SRAT   = 10,
   TT_SFLOAT = 11, TT_DFLOAT = 12
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
enum enu_ifd_tag
{
   /*--( Bit 0: Thumbnail => 1, "full" image => 0 )------------------------*/
   IT_NewSubFileType             = 0x00FE,
   /*--( Array of eventual sub-IFDs )--------------------------------------*/
   IT_SubIFDs                    = 0x014a,
   /*--( IPTC info )-------------------------------------------------------*/
   IT_Iptc1                      = 0x83bb,
   IT_Iptc2                      = 0x8568,

   /*----------------------------------------------------------------------*/
   IT_ImageWidth                 = 0x0100, // SHORT or LONG
   IT_ImageLength                = 0x0101, // SHORT or LONG
   IT_BitsPerSample              = 0x0102, // SHORT 8,8,8
   /*----------------------------------------------------------------------*/
   IT_ImageDescription           = 0x010e, // "de"
   IT_Make                       = 0x010f, // "mk"
   IT_Model                      = 0x0110, // "mo"
   IT_Orientation                = 0x0112, // "or"
   IT_XResolution                = 0x011a, // "xr"
   IT_YResolution                = 0x011b, // "yr"
   IT_ResolutionUnit             = 0x0128, // "ru"
   IT_Software                   = 0x0131, // "so"
   IT_Artist                     = 0x013b, // "ar"
   IT_DateTime                   = 0x0132, // "dt"
   IT_WhitePoint                 = 0x013e, // "wp"
   IT_PrimaryChromaticities      = 0x013f, // "pc"
   IT_YCbCrCoefficients          = 0x0211, // "yc"
   IT_YCbCrPositioning           = 0x0213, // "yp"
   IT_ReferenceBlackWhite        = 0x0214, // "bw"
   IT_Copyright                  = 0x8298, // "co"
   IT_ExifOffset                 = 0x8769, // "eo"
   IT_GPSOffset                  = 0x8825, // "go"
   IT_JpegOffsetSOI              = 0x0201, // "to"
   IT_JpegDataSize               = 0x0202, // "ts"
   /*----------------------------------------------------------------------*/
   IT_ExposureTime               = 0x829a, // "et"
   IT_FNumber                    = 0x829d, // "fn"
   IT_ExposureProgram            = 0x8822, // "ep"
   IT_SpectralSensitivity        = 0x8824, // "sy" // 2
   IT_ISOSpeedRatings            = 0x8827, // "sr"
   IT_ExifVersion                = 0x9000, // "ve"
   IT_SubsecTime                 = 0x9290, // "St"
   IT_DateTimeOriginal           = 0x9003, // "do"
   IT_SubsecTimeOriginal         = 0x9291, // "So"
   IT_DateTimeDigitized          = 0x9004, // "dd"
   IT_SubsecTimeDigitized        = 0x9292, // "Sd"
   IT_ComponentConfiguration     = 0x9101, // "cc"
   IT_CompressedBitsPerPixel     = 0x9102, // "bp"
   IT_ShutterSpeedValue          = 0x9201, // "ss"
   IT_ApertureValue              = 0x9202, // "av"
   IT_BrightnessValue            = 0x9203, // "bv"
   IT_ExposureBiasValue          = 0x9204, // "ev"
   IT_MaxApertureValue           = 0x9205, // "ma"
   IT_SubjectDistance            = 0x9206, // "sd"
   IT_MeteringMode               = 0x9207, // "mm"
   IT_LightSource                = 0x9208, // "ls"
   IT_Flash                      = 0x9209, // "fa"
   IT_FocalLength                = 0x920a, // "fl"
   IT_FlashEnergy                = 0xa20b, // "fe" // 2
   IT_UserComment                = 0x9286, // "uc"
   IT_FlashPixVersion            = 0xa000, // "fv"
   IT_ColorSpace                 = 0xa001, // "cs"
   IT_ExifImageWidth             = 0xa002, // "iw"
   IT_ExifImageHeight            = 0xa003, // "ih"
   IT_RelatedSoundFile           = 0xa004, // "sf"
   IT_FocalPlaneXResolution      = 0xa20e, // "fx"
   IT_FocalPlaneYResolution      = 0xa20f, // "fy"
   IT_FocalPlaneResolutionUnit   = 0xa210, // "fu"
   IT_ExposureIndex              = 0xa215, // "ei" // 2
   IT_SensingMethod              = 0xa217, // "sm"
   IT_FileSource                 = 0xa300, // "fs"
   IT_SceneType                  = 0xa301, // "st"
   IT_CustomRendered             = 0xa401, // "cr" // 2
   IT_ExposureMode               = 0xa402, // "em" // 2
   IT_WhiteBalance               = 0xa403, // "wb" // 2
   IT_DigitalZoomRatio           = 0xa404, // "dz" // 2
   IT_FocalLengthIn35mmFilm      = 0xa405, // "fL" // 2
   IT_SceneCaptureType           = 0xa406, // "sc" // 2
   IT_GainControl                = 0xa407, // "gc" // 2
   IT_Contrast                   = 0xa408, // "ct" // 2
   IT_Saturation                 = 0xa409, // "su" // 2
   IT_Sharpness                  = 0xa40a, // "sh" // 2
   IT_SubjectDistanceRange       = 0xa40c, // "sD" // 2
   IT_ImageUniqueID              = 0xa420, // "iu"
   /*--( GPS )-------------------------------------------------------------*/
   IT_GPSVersionID               = 0x0000, // "gv"
   IT_GPSLatitudeRef             = 0x0001, // "lA"
   IT_GPSLatitude                = 0x0002, // "la"
   IT_GPSLongitudeRef            = 0x0003, // "lO"
   IT_GPSLongitude               = 0x0004, // "lo"
   IT_GPSAltitudeRef             = 0x0005, // "aL"
   IT_GPSAltitude                = 0x0006, // "al"
   IT_GPSTimeStamp               = 0x0007, // "TS"
   IT_GPSSatellites              = 0x0008, // "sa"
   IT_GPSStatus                  = 0x0009, // "us"
   IT_GPSMeasureMode             = 0x000A, // "mM"
   IT_GPSDOP                     = 0x000B, // "dp"
   IT_GPSSpeedRef                = 0x000C, // "sP"
   IT_GPSSpeed                   = 0x000D, // "sp"
   IT_GPSTrackRef                = 0x000E, // "tR"
   IT_GPSTrack                   = 0x000F, // "tr"
   IT_GPSImgDirectionRef         = 0x0010, // "iD"
   IT_GPSImgDirection            = 0x0011, // "id"
   IT_GPSMapDatum                = 0x0012, // "md"
   IT_GPSDestLatitudeRef         = 0x0013, // "LA"
   IT_GPSDestLatitude            = 0x0014, // "La"
   IT_GPSDestLongitudeRef        = 0x0015, // "LO"
   IT_GPSDestLongitude           = 0x0016, // "Lo"
   IT_GPSDestBearingRef          = 0x0017, // "bE"
   IT_GPSDestBearing             = 0x0018, // "be"
   IT_GPSDestDistanceRef         = 0x0019, // "DR"
   IT_GPSDestDistance            = 0x001A  // "Dr"
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
enum enu_tiff_tag_col_group
{
   /*----------------------------------------------------------------------*/
   TIFF_TCG_BASE      = 1 << 1 ,
   TIFF_TCG_THUMBNAIL = 1 << 2 ,
   TIFF_TCG_STANDARD  = 1 << 3 ,
   TIFF_TCG_GPS       = 1 << 4 ,
   /*-----------------------------------------------------------------------+
   ! As the GPS tags id (0x00 -> 0x1A) can be use for other matters, the    !
   ! GPS tags will only be searched in the IFDs located at the offset       !
   ! associated to the tag TT_GPSOffset.                                    !
   ! Look in the code for: TIFF_TCG_GPS                                     !
   +-----------------------------------------------------------------------*/
   TIFF_TCG_ALL_EXCEPT_GPS = ~TIFF_TCG_GPS
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
static int st_tiff_conv_tiff_tag_col( int i_tag, int i_tcg )
{
   /*--( Image "base" data )-----------------------------------------------*/
   if( ( i_tcg & TIFF_TCG_BASE ) != 0 )
   {
      /*-------------------------------------------------------------------*/
      switch( i_tag )
      {
         /*----------------------------------------------------------------*/
         case IT_ImageWidth           : return( COL_IMG_X    ) ;
         case IT_ImageLength          : return( COL_IMG_Y    ) ;
         case IT_BitsPerSample        : return( COL_IMG_BPP  ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( ( i_tcg & TIFF_TCG_THUMBNAIL ) != 0 )
   {
      /*-------------------------------------------------------------------*/
      switch( i_tag )
      {
         /*----------------------------------------------------------------*/
         case IT_JpegOffsetSOI        : return( COL_EXIF_THUMBNAIL_OFFSET ) ;
         case IT_JpegDataSize         : return( COL_EXIF_THUMBNAIL_SIZE   ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( ( i_tcg & TIFF_TCG_STANDARD ) != 0 )
   {
      /*-------------------------------------------------------------------*/
      switch( i_tag )
      {
         /*----------------------------------------------------------------*/
         case IT_Orientation          : return( COL_EXIF_ORIENTATION )      ;
         case IT_XResolution          : return( COL_EXIF_XRESOLUTION )      ;
         case IT_YResolution          : return( COL_EXIF_YRESOLUTION )      ;
         case IT_ResolutionUnit       : return( COL_EXIF_RESOLUTIONUNIT )   ;
         case IT_WhitePoint           : return( COL_EXIF_WHITEPOINT )       ;
         case IT_PrimaryChromaticities:
                                     return( COL_EXIF_PRIMARYCHROMATICITIES);
         case IT_YCbCrCoefficients    : return( COL_EXIF_YCBCRCOEFFICIENTS );
         case IT_YCbCrPositioning     : return( COL_EXIF_YCBCRPOSITIONING ) ;
         case IT_ReferenceBlackWhite:return( COL_EXIF_REFERENCEBLACKWHITE ) ;
         /*----------------------------------------------------------------*/
         case IT_ImageDescription     : return( COL_EXIF_IMAGEDESCRIPTION ) ;
         case IT_Make                 : return( COL_EXIF_MAKE )             ;
         case IT_Model                : return( COL_EXIF_MODEL )            ;
         case IT_Software             : return( COL_EXIF_SOFTWARE )         ;
         case IT_Artist               : return( COL_EXIF_ARTIST )           ;
         case IT_DateTime             : return( COL_EXIF_DATETIME )         ;
         case IT_Copyright            : return( COL_EXIF_COPYRIGHT )        ;
         case IT_ExifOffset           : return( COL_EXIF_EXIFOFFSET )       ;
         case IT_GPSOffset            : return( COL_EXIF_GPSOFFSET )        ;
         /*----------------------------------------------------------------*/
         case IT_ExposureTime         : return( COL_EXIF_EXPOSURETIME )     ;
         case IT_FNumber              : return( COL_EXIF_FNUMBER )          ;
         case IT_ExposureProgram      : return( COL_EXIF_EXPOSUREPROGRAM )  ;
         case IT_SpectralSensitivity :return( COL_EXIF_SPECTRALSENSITIVITY );
         case IT_ISOSpeedRatings      : return( COL_EXIF_ISOSPEEDRATINGS )  ;
         case IT_ExifVersion          : return( COL_EXIF_EXIFVERSION )      ;
         case IT_SubsecTime           : return( COL_EXIF_SUBSECTIME )       ;
         case IT_DateTimeOriginal     : return( COL_EXIF_DATETIMEORIGINAL ) ;
         case IT_SubsecTimeOriginal   :
                                      return( COL_EXIF_SUBSECTIMEORIGINAL ) ;
         case IT_DateTimeDigitized    : return( COL_EXIF_DATETIMEDIGITIZED );
         case IT_SubsecTimeDigitized   :
                                     return( COL_EXIF_SUBSECTIMEDIGITIZED ) ;
         case IT_ComponentConfiguration :
                                  return( COL_EXIF_COMPONENTCONFIGURATION ) ;
         case IT_CompressedBitsPerPixel :
                                  return( COL_EXIF_COMPRESSEDBITSPERPIXEL ) ;
         case IT_ShutterSpeedValue    : return( COL_EXIF_SHUTTERSPEEDVALUE );
         case IT_ApertureValue        : return( COL_EXIF_APERTUREVALUE )    ;
         case IT_BrightnessValue      : return( COL_EXIF_BRIGHTNESSVALUE )  ;
         case IT_ExposureBiasValue    : return( COL_EXIF_EXPOSUREBIASVALUE );
         case IT_MaxApertureValue     : return( COL_EXIF_MAXAPERTUREVALUE ) ;
         case IT_SubjectDistance      : return( COL_EXIF_SUBJECTDISTANCE )  ;
         case IT_MeteringMode         : return( COL_EXIF_METERINGMODE )     ;
         case IT_LightSource          : return( COL_EXIF_LIGHTSOURCE )      ;
         case IT_Flash                : return( COL_EXIF_FLASH )            ;
         case IT_FocalLength          : return( COL_EXIF_FOCALLENGTH )      ;
         case IT_FlashEnergy          : return( COL_EXIF_FLASHENERGY )      ;
         case IT_UserComment          : return( COL_EXIF_USERCOMMENT )      ;
         case IT_FlashPixVersion      : return( COL_EXIF_FLASHPIXVERSION )  ;
         case IT_ColorSpace           : return( COL_EXIF_COLORSPACE )       ;
         case IT_ExifImageWidth       : return( COL_EXIF_EXIFIMAGEWIDTH )   ;
         case IT_ExifImageHeight      : return( COL_EXIF_EXIFIMAGEHEIGHT )  ;
         case IT_RelatedSoundFile     : return( COL_EXIF_RELATEDSOUNDFILE ) ;
         case IT_FocalPlaneXResolution :
                                   return( COL_EXIF_FOCALPLANEXRESOLUTION ) ;
         case IT_FocalPlaneYResolution:
                                   return( COL_EXIF_FOCALPLANEYRESOLUTION ) ;
         case IT_FocalPlaneResolutionUnit :
                                return( COL_EXIF_FOCALPLANERESOLUTIONUNIT ) ;
         case IT_ExposureIndex        : return( COL_EXIF_EXPOSUREINDEX )    ;
         case IT_SensingMethod        : return( COL_EXIF_SENSINGMETHOD )    ;
         case IT_FileSource           : return( COL_EXIF_FILESOURCE )       ;
         case IT_SceneType            : return( COL_EXIF_SCENETYPE )        ;
         case IT_CustomRendered       : return( COL_EXIF_CUSTOMRENDERED )   ;
         case IT_ExposureMode         : return( COL_EXIF_EXPOSUREMODE )     ;
         case IT_WhiteBalance         : return( COL_EXIF_WHITEBALANCE )     ;
         case IT_DigitalZoomRatio     : return( COL_EXIF_DIGITALZOOMRATIO ) ;
         case IT_FocalLengthIn35mmFilm :
                                   return( COL_EXIF_FOCALLENGTHIN35MMFILM ) ;
         case IT_SceneCaptureType     : return( COL_EXIF_SCENECAPTURETYPE ) ;
         case IT_GainControl          : return( COL_EXIF_GAINCONTROL )      ;
         case IT_Contrast             : return( COL_EXIF_CONTRAST )         ;
         case IT_Saturation           : return( COL_EXIF_SATURATION )       ;
         case IT_Sharpness            : return( COL_EXIF_SHARPNESS )        ;
         case IT_SubjectDistanceRange :
                                    return( COL_EXIF_SUBJECTDISTANCERANGE ) ;
         case IT_ImageUniqueID        : return( COL_EXIF_IMAGEUNIQUEID )    ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( ( i_tcg & TIFF_TCG_GPS ) != 0 )
   {
      /*-------------------------------------------------------------------*/
      switch( i_tag )
      {
         /*----------------------------------------------------------------*/
         case IT_GPSVersionID         : return( COL_EXIF_GPSVERSIONID )     ;
         case IT_GPSLatitudeRef       : return( COL_EXIF_GPSLATITUDEREF )   ;
         case IT_GPSLatitude          : return( COL_EXIF_GPSLATITUDE )      ;
         case IT_GPSLongitudeRef      : return( COL_EXIF_GPSLONGITUDEREF )  ;
         case IT_GPSLongitude         : return( COL_EXIF_GPSLONGITUDE )     ;
         case IT_GPSAltitudeRef       : return( COL_EXIF_GPSALTITUDEREF )   ;
         case IT_GPSAltitude          : return( COL_EXIF_GPSALTITUDE )      ;
         case IT_GPSTimeStamp         : return( COL_EXIF_GPSTIMESTAMP )     ;
         case IT_GPSSatellites        : return( COL_EXIF_GPSSATELLITES )    ;
         case IT_GPSStatus            : return( COL_EXIF_GPSSTATUS )        ;
         case IT_GPSMeasureMode       : return( COL_EXIF_GPSMEASUREMODE )   ;
         case IT_GPSDOP               : return( COL_EXIF_GPSDOP )           ;
         case IT_GPSSpeedRef          : return( COL_EXIF_GPSSPEEDREF )      ;
         case IT_GPSSpeed             : return( COL_EXIF_GPSSPEED )         ;
         case IT_GPSTrackRef          : return( COL_EXIF_GPSTRACKREF )      ;
         case IT_GPSTrack             : return( COL_EXIF_GPSTRACK )         ;
         case IT_GPSImgDirectionRef   : return( COL_EXIF_GPSIMGDIRECTIONREF);
         case IT_GPSImgDirection      : return( COL_EXIF_GPSIMGDIRECTION )  ;
         case IT_GPSMapDatum          : return( COL_EXIF_GPSMAPDATUM )      ;
         case IT_GPSDestLatitudeRef   : return( COL_EXIF_GPSDESTLATITUDEREF);
         case IT_GPSDestLatitude      : return( COL_EXIF_GPSDESTLATITUDE )  ;
         case IT_GPSDestLongitudeRef  : return(COL_EXIF_GPSDESTLONGITUDEREF);
         case IT_GPSDestLongitude     : return( COL_EXIF_GPSDESTLONGITUDE ) ;
         case IT_GPSDestBearingRef    : return( COL_EXIF_GPSDESTBEARINGREF );
         case IT_GPSDestBearing       : return( COL_EXIF_GPSDESTBEARING )   ;
         case IT_GPSDestDistanceRef   : return( COL_EXIF_GPSDESTDISTANCEREF);
         case IT_GPSDestDistance      : return( COL_EXIF_GPSDESTDISTANCE )  ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( COL_NB ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Conversion of a byte array into strings separated with '.' )---------*/
static void st_tiff_exif_string_tab_byte( int i_nb, wxUint8 *p_b_byte,
                                          wxString &s_val
                                        )
{
   /*----------------------------------------------------------------------*/
   s_val.clear() ;
   /*----------------------------------------------------------------------*/
   if( i_nb <= 0 || p_b_byte == NULL ) { return ; }
   /*----------------------------------------------------------------------*/
   s_val.reserve( i_nb * 3 ) ;
   /*----------------------------------------------------------------------*/
   for( ; i_nb >= 1 ; --i_nb, ++p_b_byte )
   {
      /*-------------------------------------------------------------------*/
      s_val += wxString::Format( "%d", *p_b_byte ) ;
      /*-------------------------------------------------------------------*/
      if( i_nb > 1 ) { s_val += '.' ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Exif specifications say: YYYY:MM:DD HH:MM:SS                              !
! But unfortunately some devices don't follow it ...                        !
! For example: HTC Sensation Z710e (YYYY/MM/DD HH:MM:DD)                    !
+--------------------------------------------------------------------------*/
static int st_tiff_exif_conv_datetime_string( const wxString &s,
                                              wxDateTime     &dt
                                            )
{
   /*----------------------------------------------------------------------*/
   return( sr::init_date_ymdhms( s, "%4d%*[/:]%2d%*[/:]%2d %2d:%2d:%2d", dt )
         ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Conversion to string of an array of "unsigned rational" : int / int )*/
void CFileInit_tiff::string_tab_urat( int i_nb, wxUint32 *p_urat,
                                      wxString &s_val
                                    ) const
{
   /*----------------------------------------------------------------------*/
   s_val.clear() ;
   /*----------------------------------------------------------------------*/
   if( i_nb <= 0 || p_urat == NULL ) { return ; }
   /*----------------------------------------------------------------------*/
   for( ; i_nb >= 1 ; --i_nb, p_urat += 2 )
   {
      /*-------------------------------------------------------------------*/
      s_val += wxString::Format( "%u/%u",
                                 m_p_fnc_udw( *p_urat ),
                                 m_p_fnc_udw( *( p_urat + 1 ) )
                               ) ;
      /*-------------------------------------------------------------------*/
      if( i_nb > 1 ) { s_val += ',' ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*--( Conversion to string of GPS information )----------------------------*/
void CFileInit_tiff::string_tab_pos_gps( int i_nb, wxUint32 *p_urat,
                                         wxString &s_val
                                       ) const
{
   /*----------------------------------------------------------------------*/
   int i_num ;
   /*----------------------------------------------------------------------*/
   s_val.clear() ;
   /*----------------------------------------------------------------------*/
   if( i_nb <= 0 || p_urat == NULL ) { return ; }
   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < i_nb ; ++i_num, p_urat += 2 )
   {
      /*-------------------------------------------------------------------*/
      unsigned long ul_n = m_p_fnc_udw( *p_urat ) ;
      unsigned long ul_d = m_p_fnc_udw( *( p_urat + 1 ) ) ;
      /*--( Don't want "Division by zero" error ... )----------------------*/
      if( ul_d == 0 ) { ul_n = 0 ; ul_d = 1 ; }
      /*-------------------------------------------------------------------*/
      if( ul_n % ul_d == 0 )
      {  s_val += wxString::Format( "%lu", ul_n / ul_d ) ; }
      else
      {  s_val += wxString::Format( "%.2lf",
                                    sr::round_01( ( double )ul_n / ul_d )
                                  ) ;
      }
      /*-------------------------------------------------------------------*/
      switch( i_num )
      {  case 0  : s_val += 'd' ; break ;
         case 1  : s_val += 'm' ; break ;
         case 2  : s_val += 's' ; break ;
         default : s_val += '_' ; break ; // Who knows ...
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*--( Conversion to string of exif time information )----------------------*/
void CFileInit_tiff::string_tab_tms( int i_nb, wxUint32 *p_urat,
                                     wxString &s_val
                                   ) const
{
   /*----------------------------------------------------------------------*/
   int i_num ;
   /*----------------------------------------------------------------------*/
   s_val.clear() ;
   /*----------------------------------------------------------------------*/
   if( i_nb <= 0 || p_urat == NULL ) { return ; }
   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < i_nb && i_num < 3 ; ++i_num, p_urat += 2 )
   {
      /*-------------------------------------------------------------------*/
      unsigned long ul_n = m_p_fnc_udw( *p_urat ) ;
      unsigned long ul_d = m_p_fnc_udw( *( p_urat + 1 ) ) ;
      /*-------------------------------------------------------------------*/
      if( ul_d == 0 ) { ul_n = 0 ; ul_d = 1 ; }
      /*-------------------------------------------------------------------*/
      s_val += wxString::Format( "%02.0lf",( double )ul_n / ( double )ul_d );
      /*-------------------------------------------------------------------*/
      switch( i_num )
      {  case  0 : s_val += 'h' ; break ;
         case  1 : s_val += 'm' ; break ;
         case  2 : s_val += 's' ; break ;
         default : s_val += '_' ; break ;
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_tiff::read_tag( const int ifd_tag, const int i_col,
                              const str_tiff_ifd_entry &ifd_entry
                            )
{
   /*----------------------------------------------------------------------*/
   enu_tiff_tag_type tag_type                         ;
   int               i_nb_elmt                        ;
   int               i_elmt_size                      ;
   int               i_total_size                     ;
   char              *p_c_val                         ;
   char              tb_c_val[ g_co_i_string_sz_max ] ;
   const char        *p_c_tmp                         ;

   /*--( The values are low, so it is useless to use unsigned )------------*/
   long              l_val      = 0 ;
   long              l_nomin    = 0 ;
   long              l_denomin  = 1 ;
   double            do_val         ;
   wxString          s_val          ;

   /*--( Types and number of components )----------------------------------*/
   tag_type  = ( enu_tiff_tag_type )m_p_fnc_uw( ifd_entry.w_type ) ;
   i_nb_elmt = m_p_fnc_udw( ifd_entry.dw_nb_elmt ) ;

   /*--( Computation of the necessary size of the value )------------------*/
   switch( tag_type )
   {
      /*-------------------------------------------------------------------*/
      case TT_UBYTE : case TT_SBYTE : case TT_STRING : case TT_UNDEF :
         i_elmt_size = 1 ; break ;
      /*-------------------------------------------------------------------*/
      case TT_USHORT : case TT_SSHORT : i_elmt_size = 2 ; break ;
      case TT_ULONG  : case TT_SLONG  : i_elmt_size = 4 ; break ;
      case TT_URAT   : case TT_SRAT   : i_elmt_size = 8 ; break ;
      case TT_SFLOAT :                  i_elmt_size = 4 ; break ;
      case TT_DFLOAT :                  i_elmt_size = 8 ; break ;
      /*-------------------------------------------------------------------*/
      default : return( -1 ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   i_total_size = i_nb_elmt * i_elmt_size ;

   /*-----------------------------------------------------------------------+
   ! For the IPTC tags, of course, the already written code is used.        !
   +-----------------------------------------------------------------------*/
   if( ifd_tag == IT_Iptc1 || ifd_tag == IT_Iptc2 )
   {
      /*-------------------------------------------------------------------*/
      m_fi.read_iptc( m_fo_offset + m_p_fnc_udw( ifd_entry.dw_off_val ),
                      i_total_size, false
                    ) ;
      /*-------------------------------------------------------------------*/
      return( 0 ) ;
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! If the size is lower than the size of the offset then the data is      !
   ! stored in the offset.                                                  !
   +-----------------------------------------------------------------------*/
   if( i_total_size <= ( int )sizeof( ifd_entry.dw_off_val ) )
   {  p_c_val = ( char * )&ifd_entry.dw_off_val ; }
   else
   {  /*-------------------------------------------------------------------*/
      if( i_total_size > ( int )sizeof( tb_c_val ) )
      {  i_total_size = sizeof( tb_c_val ) ; }
      /*-------------------------------------------------------------------*/
      wxFileOffset fo_offset_val = m_p_fnc_udw( ifd_entry.dw_off_val ) ;
      if( !ok_to_read( fo_offset_val, i_total_size ) )
      {  return( -2 ) ; }
      /*-------------------------------------------------------------------*/
      if(    m_fa.set_offset( m_fo_offset + fo_offset_val ) != 0
          || m_fa.read_buffer( i_total_size, tb_c_val ) != 0
        )
      {  return( -3 ) ; }
      /*-------------------------------------------------------------------*/
      p_c_val = tb_c_val ;
      /*-------------------------------------------------------------------*/
   }

   /*--( Compute the value based on the type )-----------------------------*/
   switch( tag_type )
   {
      /*-------------------------------------------------------------------*/
      case TT_UNDEF : case TT_UBYTE : case TT_SBYTE :
         l_val = p_c_val[ 0 ] ; break ;
      /*-------------------------------------------------------------------*/
      case TT_SSHORT :
         l_val = m_p_fnc_sw( *( wxInt16 * )p_c_val ) ; break ;
      /*-------------------------------------------------------------------*/
      case TT_USHORT :
         l_val = m_p_fnc_uw( *( wxUint16 * )p_c_val ) ; break ;
      /*-------------------------------------------------------------------*/
      case TT_SLONG :
         l_val = m_p_fnc_sdw( *( wxInt32 * )p_c_val ) ; break ;
      /*-------------------------------------------------------------------*/
      case TT_ULONG :
         l_val = m_p_fnc_udw( *( wxUint32 * )p_c_val ) ; break ;
      /*-------------------------------------------------------------------*/
      case TT_SRAT :
      {  /*----------------------------------------------------------------*/
         l_nomin   = m_p_fnc_sdw( *( wxInt32 * )p_c_val ) ;
         l_denomin = m_p_fnc_sdw( *( wxInt32 * )( p_c_val + 4 ) ) ;
         /*----------------------------------------------------------------*/
         if( l_denomin == 0 ) { l_nomin = 0 ; l_denomin = 1 ; }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      case TT_URAT :
      {  /*----------------------------------------------------------------*/
         l_nomin   = m_p_fnc_udw( *( wxUint32 * )p_c_val ) ;
         l_denomin = m_p_fnc_udw( *( wxUint32 * )( p_c_val + 4 ) ) ;
         /*----------------------------------------------------------------*/
         if( l_denomin == 0 ) { l_nomin = 0 ; l_denomin = 1 ; }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }
      /*--( No default => compiler warning )-------------------------------*/
      default : break ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   switch( ifd_tag )
   {
      /*-------------------------------------------------------------------*/
      case IT_UserComment :
      {
         /*-----------------------------------------------------------------+
         ! The "UserComment" contains a header giving the encoding type of  !
         ! what follows.                                                    !
         +-----------------------------------------------------------------*/

         /*--( Minimal size not reached => by default ascii )--------------*/
         if( i_total_size <= 8 )
         {  s_val = wxString::From8BitData( p_c_val, i_total_size ) ; }
         else /*--( Unicode ? )--------------------------------------------*/
         if(    p_c_val[ 7 ] == '\0'
             && wxString::From8BitData( p_c_val ).MakeLower() == "unicode"
           )
         {  sr::utf16_to_string( p_c_val + 8, ( i_total_size - 8 ) / 2 * 2,
                                 s_val
                               ) ;
         }
         else /*------------------------------------------------------------+
              ! The other types values are:                                 !
              ! if( memicmp( p_b_val, "\0\0\0\0\0\0\0\0", 8 ) == 0          !
              ! if( memicmp( p_b_val, "JIS\0\0\0\0\0"   , 8 ) == 0          !
              ! if( memicmp( p_b_val, "ASCII\0\0\0", 8 ) == 0 )             !
              !                                                             !
              ! It seems that some soft tools don't store this header.      !
              +------------------------------------------------------------*/
         {
            /*--( Detection of a header by at least one '\0' )-------------*/
            int i_num = 7 ;
            while( i_num >= 0 && p_c_val[ i_num ] != '\0' )
            {  --i_num ; }
            /*--( Skip the header if any )---------------------------------*/
            if( i_num >= 0 ) { i_num = 8 ; } else { i_num = 0 ; }
            /*-------------------------------------------------------------*/
              s_val
            = wxString::From8BitData( p_c_val + i_num, i_total_size - i_num);
            /*-------------------------------------------------------------*/
         }

         /*----------------------------------------------------------------*/
         if( m_fi.prepare_string( s_val ) > 0 )
         {  m_f.val_s( i_col ) = s_val ; }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*--( The strings ... )----------------------------------------------*/
      case IT_ImageDescription :
      case IT_Make :
      case IT_Model :
      case IT_Software :
      case IT_Artist :
      case IT_Copyright :
      case IT_RelatedSoundFile :
      case IT_SpectralSensitivity :
      case IT_ImageUniqueID :
      case IT_GPSLatitudeRef :
      case IT_GPSLongitudeRef :
      case IT_GPSSatellites :
      case IT_GPSStatus :
      case IT_GPSMeasureMode :
      case IT_GPSSpeedRef :
      case IT_GPSTrackRef :
      case IT_GPSImgDirectionRef :
      case IT_GPSMapDatum :
      case IT_GPSDestLatitudeRef :
      case IT_GPSDestLongitudeRef :
      case IT_GPSDestBearingRef :
      case IT_GPSDestDistanceRef :
      /*--( Subsec : Yes, they are strings ! )-----------------------------*/
      case IT_SubsecTime :
      case IT_SubsecTimeOriginal :
      case IT_SubsecTimeDigitized :
         /*----------------------------------------------------------------*/
         if( m_fi.prepare_string( p_c_val, i_total_size, s_val ) > 0 )
         {  m_f.val_s( i_col ) = s_val ; }
         /*----------------------------------------------------------------*/
         break ;

      /*--( Dates )--------------------------------------------------------*/
      case IT_DateTime :
      case IT_DateTimeOriginal :
      case IT_DateTimeDigitized :
      {
         /*----------------------------------------------------------------*/
         wxDateTime dt ;
         /*----------------------------------------------------------------*/
         if( m_fi.prepare_string( p_c_val, i_total_size, s_val ) <= 0 )
         {  break ; }
         /*----------------------------------------------------------------*/
         if( st_tiff_exif_conv_datetime_string( s_val, dt ) == 0 )
         {  m_fi.init_date( i_col, dt ) ; }
         else /*--( Date "ko" )--------------------------------------------*/
         {  m_f.val_s( i_col ) = s_val ; }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*--( BPP on N SHORTs )----------------------------------------------*/
      case IT_BitsPerSample :
      {
         /*----------------------------------------------------------------*/
         int i_bpp ;
         int i_num ;
         /*----------------------------------------------------------------*/
         for( i_bpp = i_num = 0 ; i_num < i_nb_elmt ; ++i_num )
         {     i_bpp
            += m_p_fnc_uw( *( wxUint16 * )( &p_c_val[ i_elmt_size * i_num ] )
                         ) ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_ll( i_col ) = i_bpp ;
         m_f.val_s( i_col ) << i_bpp ;
         /*----------------------------------------------------------------*/
         break ;
      }

      /*--( Signed integers )----------------------------------------------*/
      case IT_ImageWidth :
      case IT_ImageLength :
         /*----------------------------------------------------------------*/
         m_f.val_ll( i_col ) = l_val ;
         m_f.val_s( i_col ) << l_val ;
         /*----------------------------------------------------------------*/
         break ;

      /*--( Unsigned integers )--------------------------------------------*/
      case IT_ISOSpeedRatings :
      case IT_ExifImageWidth :
      case IT_ExifImageHeight :
         /*----------------------------------------------------------------*/
         m_f.val_ll( i_col ) = ( unsigned long )l_val ;
         m_f.val_s( i_col ).Printf( "%lu", ( unsigned long )l_val ) ;
         /*----------------------------------------------------------------*/
         break ;

      /*--------------------------------------------------------------------+
      ! Offsets ...                                                         !
      +--------------------------------------------------------------------*/
      case IT_ExifOffset :
      case IT_GPSOffset :
         /*----------------------------------------------------------------*/
         m_f.val_ll( i_col ) = ( unsigned long )l_val ;
         m_f.val_s( i_col ).Printf( "%lu", ( unsigned long )l_val ) ;
         /*----------------------------------------------------------------*/
         read_ifd( l_val,
                   ifd_tag == IT_ExifOffset ? TIFF_TCG_STANDARD: TIFF_TCG_GPS
                 ) ;
         /*----------------------------------------------------------------*/
         break ;

      /*--------------------------------------------------------------------+
      ! IFD                                                                 !
      +--------------------------------------------------------------------*/
      case IT_SubIFDs :
      {
         /*----------------------------------------------------------------*/
         int          i_num     ;
         wxFileOffset fo_offset ;
         wxUint32     *p_dw     ;
         /*----------------------------------------------------------------*/
         for( i_num = 0, p_dw = ( wxUint32 * )p_c_val ;
              i_num < i_nb_elmt ;
              ++i_num, ++p_dw
            )
         {  /*-------------------------------------------------------------*/
            fo_offset = m_p_fnc_udw( *p_dw ) ;
            read_ifd( fo_offset, TIFF_TCG_ALL_EXCEPT_GPS ) ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*--------------------------------------------------------------------+
      ! Thumbnail data                                                      !
      +--------------------------------------------------------------------*/
      case IT_JpegOffsetSOI :
      case IT_JpegDataSize :
         /*----------------------------------------------------------------*/
         m_f.val_ll( i_col ) = l_val ;
         m_f.val_s( i_col ).Printf( "%ld", l_val ) ;
         /*----------------------------------------------------------------*/
         if( ifd_tag == IT_JpegOffsetSOI )
         {  m_f.set_thumbnail_offset( m_fo_offset + l_val ) ; }
         else
         if( ifd_tag == IT_JpegDataSize )
         {  m_f.set_thumbnail_size( l_val ) ; }
         /*----------------------------------------------------------------*/
         break ;

      /*--( Signed rationals )---------------------------------------------*/
      case IT_BrightnessValue :
      case IT_ExposureBiasValue :
         /*----------------------------------------------------------------*/
         do_val = ( double )l_nomin / l_denomin ;
         m_f.val_do( i_col ) = do_val           ;
         /*----------------------------------------------------------------*/
         if( l_denomin == 1 )
         {  m_f.val_s( i_col ).Printf( "EV%ld", l_nomin ) ; }
         else
         {  m_f.val_s( i_col ).Printf( "EV%0.1lf", sr::round_1( do_val ) ) ;}
         /*----------------------------------------------------------------*/
         break ;

      /*--( Unsigned rationals )-------------------------------------------*/
      case IT_XResolution :
      case IT_YResolution :
      case IT_FlashEnergy :
      case IT_FocalPlaneXResolution :
      case IT_FocalPlaneYResolution :
      case IT_ExposureIndex :
      case IT_CompressedBitsPerPixel :
      case IT_DigitalZoomRatio :
      case IT_GPSDOP :
      case IT_GPSSpeed :
      case IT_GPSTrack :
      case IT_GPSImgDirection :
      case IT_GPSDestBearing :
      case IT_GPSDestDistance :
         /*----------------------------------------------------------------*/
         do_val = ( double )( unsigned long )l_nomin
                          / ( unsigned long )l_denomin ;
         m_f.val_do( i_col ) = do_val ;
         /*----------------------------------------------------------------*/
         if( l_denomin == 1 )
         {  m_f.val_s( i_col ).Printf( "%lu", ( unsigned long )l_nomin ) ; }
         else
         {  m_f.val_s( i_col ).Printf( "%0.2lf", sr::round_01( do_val ) ) ; }
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_ExposureTime :
         /*----------------------------------------------------------------*/
         m_fi.init_exif_exposuretime( ( unsigned long )l_nomin,
                                      ( unsigned long )l_denomin
                                    ) ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_SubjectDistance :
      case IT_GPSAltitude :
         /*----------------------------------------------------------------*/
         if( l_nomin == 0 || l_denomin == 0 )
         {  /*-------------------------------------------------------------*/
            m_f.val_do( i_col ) = 0 ;
            m_f.val_s( i_col ) = "unknown" ;
            /*-------------------------------------------------------------*/
         }
         else
         {  /*-------------------------------------------------------------*/
            m_f.val_do( i_col ) = ( double )( unsigned long )l_nomin
                                          / ( unsigned long )l_denomin ;
            /*-------------------------------------------------------------*/
            if( l_denomin == 1 )
            {  m_f.val_s( i_col ).Printf( "%lum", ( unsigned long )l_nomin );
            }
            else
            {  m_f.val_s( i_col ).Printf( "%0.2lfm",
                                          sr::round_01(
                                           ( double )( unsigned long )l_nomin
                                         / ( unsigned long )l_denomin
                                                      )
                                        ) ;
            }
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_FocalLength :
         /*----------------------------------------------------------------*/
         m_fi.init_exif_focallength(   ( double )
                                       ( unsigned long )l_nomin
                                     / ( unsigned long )l_denomin
                                   ) ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_FocalLengthIn35mmFilm :
         /*----------------------------------------------------------------*/
         m_fi.init_exif_focallength35mmfilm( l_val ) ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_SceneType :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case 1  : p_c_tmp = "directly photographed image" ; break ;
            default : p_c_tmp = "reserved" ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_CustomRendered :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case 0  : p_c_tmp = "normal"   ; break ;
            case 1  : p_c_tmp = "custom"   ; break ;
            default : p_c_tmp = "reserved" ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_ExposureMode :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case 0  : p_c_tmp = "auto"         ; break ;
            case 1  : p_c_tmp = "manual"       ; break ;
            case 2  : p_c_tmp = "auto bracket" ; break ;
            default : p_c_tmp = "reserved"     ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_WhiteBalance :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case 0  : p_c_tmp = "auto"     ; break ;
            case 1  : p_c_tmp = "manual"   ; break ;
            default : p_c_tmp = "reserved" ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_SceneCaptureType :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case 0  : p_c_tmp = "standard"    ; break ;
            case 1  : p_c_tmp = "landscape"   ; break ;
            case 2  : p_c_tmp = "portrait"    ; break ;
            case 3  : p_c_tmp = "night scene" ; break ;
            default : p_c_tmp = "reserved"    ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_GainControl :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case 0  : p_c_tmp = "none"           ; break ;
            case 1  : p_c_tmp = "low gain up"    ; break ;
            case 2  : p_c_tmp = "high gain up"   ; break ;
            case 3  : p_c_tmp = "low gain down"  ; break ;
            case 4  : p_c_tmp = "high gain down" ; break ;
            default : p_c_tmp = "reserved"       ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_Contrast :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case 0  : p_c_tmp = "normal"   ; break ;
            case 1  : p_c_tmp = "soft"     ; break ;
            case 2  : p_c_tmp = "hard"     ; break ;
            default : p_c_tmp = "reserved" ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_Saturation :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case 0  : p_c_tmp = "normal"   ; break ;
            case 1  : p_c_tmp = "low"      ; break ;
            case 2  : p_c_tmp = "high"     ; break ;
            default : p_c_tmp = "reserved" ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_Sharpness :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case 0  : p_c_tmp = "normal"   ; break ;
            case 1  : p_c_tmp = "soft"     ; break ;
            case 2  : p_c_tmp = "hard"     ; break ;
            default : p_c_tmp = "reserved" ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_SubjectDistanceRange :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case 0  : p_c_tmp = "unknown"      ; break ;
            case 1  : p_c_tmp = "macro"        ; break ;
            case 2  : p_c_tmp = "close view"   ; break ;
            case 3  : p_c_tmp = "distant view" ; break ;
            default : p_c_tmp = "reserved"     ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_ComponentConfiguration :
      {
         /*----------------------------------------------------------------*/
         int i_num ;
         /*----------------------------------------------------------------*/
         s_val.clear() ;
         /*----------------------------------------------------------------*/
         for( i_num = 0 ; i_num < i_total_size ; ++i_num )
         {  /*-------------------------------------------------------------*/
            switch( p_c_val[ i_num ] )
            {
               /*----------------------------------------------------------*/
               case 1 : s_val += 'Y'  ; break ;
               case 2 : s_val += "Cb" ; break ;
               case 3 : s_val += "Cr" ; break ;
               case 4 : s_val += 'R'  ; break ;
               case 5 : s_val += 'G'  ; break ;
               case 6 : s_val += 'B'  ; break ;
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = s_val ;
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      case IT_ColorSpace :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case 1      : p_c_tmp = "sRGB"         ; break ;
            case 0xFFFF : p_c_tmp = "uncalibrated" ; break ;
            default     : p_c_tmp = "reserved"     ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_FileSource :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case 3  : p_c_tmp = "DSC"      ; break ;
            default : p_c_tmp = "reserved" ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_SensingMethod :
         /*--( The 6 is not there ... )------------------------------------*/
         switch( l_val )
         {  case 1  : p_c_tmp = "undefined"                      ; break ;
            case 2  : p_c_tmp = "one-chip color area sensor"     ; break ;
            case 3  : p_c_tmp = "two-chip color area sensor"     ; break ;
            case 4  : p_c_tmp = "three-chip color area sensor"   ; break ;
            case 5  : p_c_tmp = "color sequential area sensor"   ; break ;
            case 7  : p_c_tmp = "trilinear sensor"               ; break ;
            case 8  : p_c_tmp = "color sequential linear sensor" ; break ;
            default : p_c_tmp = "reserved"                       ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_Orientation :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {
            /*--------------------------------------------------------------+
            ! The "_HM" suffix stands for "Horizontal Mirror"               !
            +--------------------------------------------------------------*/
            case 1  : m_fi.init_exif_rotation( IV_ROTATION_0 )      ; break ;
            case 2  : m_fi.init_exif_rotation( IV_ROTATION_0_HM )   ; break ;
            case 3  : m_fi.init_exif_rotation( IV_ROTATION_180 )    ; break ;
            case 4  : m_fi.init_exif_rotation( IV_ROTATION_180_HM ) ; break ;
            case 5  : m_fi.init_exif_rotation( IV_ROTATION_90_HM )  ; break ;
            case 6  : m_fi.init_exif_rotation( IV_ROTATION_90 )     ; break ;
            case 7  : m_fi.init_exif_rotation( IV_ROTATION_270_HM ) ; break ;
            case 8  : m_fi.init_exif_rotation( IV_ROTATION_270 )    ; break ;
            /*-------------------------------------------------------------*/
            default : m_fi.init_exif_rotation( IV_ROTATION_NB )     ; break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_ResolutionUnit :
      case IT_FocalPlaneResolutionUnit :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case 2  : p_c_tmp = "inch"      ; break ;
            case 3  : p_c_tmp = "cm"        ; break ;
            default : p_c_tmp = "undefined" ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_WhitePoint :
         /*----------------------------------------------------------------*/
         string_tab_urat( wxMin( i_nb_elmt, 2 ), ( wxUint32 * )p_c_val,
                          m_f.val_s( i_col )
                        ) ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_PrimaryChromaticities :
         /*----------------------------------------------------------------*/
         string_tab_urat( wxMin( i_nb_elmt, 6 ), ( wxUint32 * )p_c_val,
                          m_f.val_s( i_col )
                        ) ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_YCbCrCoefficients   :
         /*----------------------------------------------------------------*/
         string_tab_urat( wxMin( i_nb_elmt, 3 ), ( wxUint32 * )p_c_val,
                          m_f.val_s( i_col )
                        ) ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_ReferenceBlackWhite :
         /*----------------------------------------------------------------*/
         string_tab_urat( wxMin( i_nb_elmt, 6 ), ( wxUint32 * )p_c_val,
                          m_f.val_s( i_col )
                        ) ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_YCbCrPositioning :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case 1  : p_c_tmp = "centered"  ; break ;
            case 2  : p_c_tmp = "co-sited"  ; break ;
            default : p_c_tmp = "undefined" ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_ExposureProgram :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case 1  : p_c_tmp = "manual control"    ; break ;
            case 2  : p_c_tmp = "program normal"    ; break ;
            case 3  : p_c_tmp = "aperture priority" ; break ;
            case 4  : p_c_tmp = "shutter priority"  ; break ;
            case 5  : p_c_tmp = "creative program"  ; break ;
            case 6  : p_c_tmp = "action program"    ; break ;
            case 7  : p_c_tmp = "portrait mode"     ; break ;
            case 8  : p_c_tmp = "landscape mode"    ; break ;
            default : p_c_tmp = "undefined"         ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_ExifVersion :
         /*----------------------------------------------------------------*/
           m_f.val_s( i_col )
         = wxString::From8BitData( p_c_val , wxMin( i_nb_elmt, 4 ) ) ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_ShutterSpeedValue :
      {  /*----------------------------------------------------------------*/
         do_val = 1 / pow( 2.0, ( double )l_nomin / l_denomin ) ;
         m_fi.init_exif_shutterspeedvalue( do_val ) ;
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      case IT_FNumber :
         /*----------------------------------------------------------------*/
         do_val = ( double )( unsigned long )l_nomin
                          / ( unsigned long )l_denomin ;
         m_f.val_do( i_col ) = do_val ;
         m_f.val_s( i_col ).Printf( "F%0.1lf", sr::round_1( do_val ) ) ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_ApertureValue :
      case IT_MaxApertureValue :
         /*----------------------------------------------------------------*/
         do_val = pow( 1.4142, ( double )( unsigned long )l_nomin
                                       / ( unsigned long )l_denomin
                     ) ;
         m_f.val_do( i_col ) = do_val ;
         m_f.val_s( i_col ).Printf( "F%0.1lf", sr::round_1( do_val ) ) ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_MeteringMode :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case   0 : p_c_tmp = "unknown"                 ; break ;
            case   1 : p_c_tmp = "average"                 ; break ;
            case   2 : p_c_tmp = "center weighted average" ; break ;
            case   3 : p_c_tmp = "spot"                    ; break ;
            case   4 : p_c_tmp = "multi-spot"              ; break ;
            case   5 : p_c_tmp = "pattern"                 ; break ;
            case   6 : p_c_tmp = "partial"                 ; break ;
            case 255 : p_c_tmp = "other"                   ; break ;
            default  : p_c_tmp = "undefined"               ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_LightSource :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case   0 : p_c_tmp = "unknown"                ; break ;
            case   1 : p_c_tmp = "daylight"               ; break ;
            case   2 : p_c_tmp = "fluorescent"            ; break ;
            case   3 : p_c_tmp = "tungsten"               ; break ;
            case   4 : p_c_tmp = "flash"                  ; break ;
            case   9 : p_c_tmp = "fine weather"           ; break ;
            case  10 : p_c_tmp = "cloudy weather"         ; break ;
            case  11 : p_c_tmp = "shade"                  ; break ;
            case  12 : p_c_tmp = "daylight fluorescent"   ; break ;
            case  13 : p_c_tmp = "day white fluorescent"  ; break ;
            case  14 : p_c_tmp = "cool white fluorescent" ; break ;
            case  15 : p_c_tmp = "white fluorescent"      ; break ;
            case  17 : p_c_tmp = "standard light a"       ; break ;
            case  18 : p_c_tmp = "standard light b"       ; break ;
            case  19 : p_c_tmp = "standard light c"       ; break ;
            case  20 : p_c_tmp = "d55"                    ; break ;
            case  21 : p_c_tmp = "d65"                    ; break ;
            case  22 : p_c_tmp = "d75"                    ; break ;
            case  23 : p_c_tmp = "d50"                    ; break ;
            case  24 : p_c_tmp = "iso studio tungsten"    ; break ;
            case 255 : p_c_tmp = "other"                  ; break ;
            default  : p_c_tmp = "reserved"               ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_Flash :
      {
         /*----------------------------------------------------------------*/
         if( l_val == 0x20 )
         {  m_f.val_s( i_col ) = "no flash function" ;
            break ;
         }
         /*----------------------------------------------------------------*/
         wxString s_flash ;
         /*----------------------------------------------------------------*/
           s_flash
         = ( ( l_val & 0x01 ) == 0 ? "no flash" : "flash" ) ;
         /*----------------------------------------------------------------*/
         switch( l_val & 0x06 )
         {  /*-------------------------------------------------------------*/
            case 0x04 :
               s_flash.append( ", return light not detected" ) ;
               break ;
            case 0x06 :
               s_flash.append( ", return light detected" ) ;
               break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         switch( l_val & 0x18 )
         {  /*-------------------------------------------------------------*/
            case 0x08 : s_flash.append( ", forced" )     ; break ;
            case 0x10 : s_flash.append( ", suppressed" ) ; break ;
            case 0x18 : s_flash.append( ", auto" )       ; break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         if( ( l_val & 0x40 ) != 0 )
         {  s_flash.append( ", red-eye reduction mode" ) ; }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = s_flash ;
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      case IT_FlashPixVersion :
         /*--( Normally 4 characters )-------------------------------------*/
           m_f.val_s( i_col )
         = wxString::From8BitData( p_c_val, wxMin( i_nb_elmt, 4 ) ) ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_GPSVersionID :
         /*----------------------------------------------------------------*/
         st_tiff_exif_string_tab_byte( wxMin( i_nb_elmt, 4 ),
                                       ( wxUint8 * )p_c_val,
                                       m_f.val_s( i_col )
                                     ) ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_GPSLatitude :
      case IT_GPSLongitude :
      case IT_GPSDestLatitude :
      case IT_GPSDestLongitude :
         /*----------------------------------------------------------------*/
         string_tab_pos_gps( wxMin( i_nb_elmt, 3 ), ( wxUint32 * )p_c_val,
                             m_f.val_s( i_col )
                           ) ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_GPSTimeStamp :
         /*----------------------------------------------------------------*/
         string_tab_tms( wxMin( i_nb_elmt, 3 ), ( wxUint32 * )p_c_val,
                         m_f.val_s( i_col )
                       ) ;
         /*----------------------------------------------------------------*/
         break ;

      /*-------------------------------------------------------------------*/
      case IT_GPSAltitudeRef :
         /*----------------------------------------------------------------*/
         switch( l_val )
         {  case 0  : p_c_tmp = "sea level" ; break ;
            default : p_c_tmp = "reserved"  ; break ;
         }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = p_c_tmp ;
         /*----------------------------------------------------------------*/
         break ;

      /*--( For unmanaged tags nothing is done )---------------------------*/
      default : return( 0 ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_tiff::read_ifd( wxFileOffset fo_offset, int i_tcg )
{
   /*----------------------------------------------------------------------*/
   str_tiff_ifd_header tiff_ifd_header       ;
   wxMemoryBuffer      mb_buffer_ifd_entry   ;
   str_tiff_ifd_entry  *p_tiff_ifd_entry     ;
   int                 i_nb_ifd_entry        ;
   int                 i_num                 ;
   wxFileOffset        fo_offset_repos       ;
   size_t              sz_size_ifd           ;
   int                 ifd_tag               ;
   int                 i_col                 ;
   bool                boo_thumbnail = false ;

   /*--( Check that the offset is in the limits )--------------------------*/
   if( !ok_to_read( fo_offset, sizeof( tiff_ifd_header ) ) )
   {  return( -1 ) ; }

   /*--( Go to the beginning and read the number of IFD entries )----------*/
   if(    m_fa.set_offset( m_fo_offset + fo_offset ) != 0
       || m_fa.read_buffer( sizeof( tiff_ifd_header ), &tiff_ifd_header
                          ) != 0
     )
   {  return( -2 ) ; }

   /*----------------------------------------------------------------------*/
   i_nb_ifd_entry = m_p_fnc_uw( tiff_ifd_header.w_nb ) ;
   sz_size_ifd    = i_nb_ifd_entry * sizeof( str_tiff_ifd_entry ) ;

   /*----------------------------------------------------------------------*/
   if( !ok_to_read( fo_offset + sizeof( tiff_ifd_header ), sz_size_ifd ) )
   {  return( -3 ) ; }

   /*--( Read the complete table )-----------------------------------------*/
   if( m_fa.read_buffer( sz_size_ifd, mb_buffer_ifd_entry ) != 0 )
   {  return( -4 ) ; }
   /*----------------------------------------------------------------------*/
   p_tiff_ifd_entry = ( str_tiff_ifd_entry * )mb_buffer_ifd_entry.GetData() ;

   /*--( Save offset for restoration on exit )-----------------------------*/
   fo_offset_repos = m_fa.get_offset() ;

   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < i_nb_ifd_entry ; ++i_num )
   {
      /*--( Tag value )----------------------------------------------------*/
      ifd_tag = m_p_fnc_uw( p_tiff_ifd_entry[ i_num ].w_tag ) ;

      /*--------------------------------------------------------------------+
      ! The problem is the management of the SUBIFDs and the possibility to !
      ! treat, with the same code, the TIFF/NEF/EXIF ... cases.             !
      ! The most complete for the SUBIFDs "situation" are the ".nef"        !
      ! (nikon raw)                                                         !
      ! A ".nef" can contain many thumbnails and the "raw" data is only     !
      ! stored at the end of the ifd serie.                                 !
      ! A lot of exif information is kept on the in thumbnails IFDs.        !
      ! To know if an IFD is associated to a thumbnail a test is done on    !
      ! tag "IT_NewSubFileType"                                             !
      !                                                                     !
      ! To make things simpler here is what has been decided:               !
      ! If an info is contained in an IDF related to:                       !
      ! => A thumbnail: no reservation but storage                          !
      ! => "Full" image": reservation and storage                           !
      !                                                                     !
      ! So at the end, the exif information related to the "full" image     !
      ! won't be lost.                                                      !
      +--------------------------------------------------------------------*/

      /*--( Info thumbnail or full image ? )-------------------------------*/
      if( ifd_tag == IT_NewSubFileType )
      {
         /*----------------------------------------------------------------*/
         if( (   m_p_fnc_udw( p_tiff_ifd_entry[ i_num ].dw_off_val ) & 0x01
             ) != 0
           )
         {  /*-------------------------------------------------------------*/
            boo_thumbnail = true ;
            /*--------------------------------------------------------------+
            ! Don't extract the "image" information.                        !
            +---------------------------------------------------------------+
            ! In the NEF D200 files, some tags including "orientation"      !
            ! appear only in the first IFD which contains a tag             !
            ! "NewSubFileType" set to 1 ...                                 !
            ! This simple modification seems to give correct results.       !
            +--------------------------------------------------------------*/
            i_tcg &= ~TIFF_TCG_BASE ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         continue ;
         /*----------------------------------------------------------------*/
      }

      /*--( Is it a "column" ? )-------------------------------------------*/
      if(    ifd_tag == IT_SubIFDs
          || ifd_tag == IT_Iptc1
          || ifd_tag == IT_Iptc2
        )
      {  i_col = COL_NB ; }
      else
      {  /*--( It is a known column )--------------------------------------*/
         i_col = st_tiff_conv_tiff_tag_col( ifd_tag, i_tcg ) ;
         if( i_col < 0 || i_col >= COL_NB ) { continue ; }
         /*-----------------------------------------------------------------+
         ! cf paragraph here upper                                          !
         ! "thumbnail" => take care to not destroy a reserved value         !
         +-----------------------------------------------------------------*/
         if( boo_thumbnail )
         {  if( m_fi.is_col_reserved( i_col ) ) { continue ; } }
         else
         {  if( !m_fi.reserve_col( i_col ) ) { continue ; } }
         /*----------------------------------------------------------------*/
      }

      /*--( Get the value )------------------------------------------------*/
      if( read_tag( ifd_tag, i_col, p_tiff_ifd_entry[ i_num ] ) != 0 )
      {  return( -5 ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*-----------------------------------------------------------------------+
   ! Set the pointer at the right place. Reading of the tag has moved it.   !
   +-----------------------------------------------------------------------*/
   if( m_fa.set_offset( fo_offset_repos ) != 0 )
   {  return( -6 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_tiff::tiff_init( wxFileOffset fo_offset, wxULongLong ull_size )
{
   /*----------------------------------------------------------------------*/
   str_tiff_header header ;

   /*----------------------------------------------------------------------*/
   m_fo_offset = fo_offset ;
   m_ull_size  = ull_size  ;

   /*--( Read the header )-------------------------------------------------*/
   if(    !ok_to_read( 0, sizeof( header ) )
       || m_fa.set_offset( m_fo_offset ) != 0
       || m_fa.read_buffer( sizeof( header ), &header ) != 0
     )
   {  return( -1 ) ; }

   /*--( Integer "endian"ness )--------------------------------------------*/
   switch( header.w_align )
   {
      /*-------------------------------------------------------------------*/
      case SR_TWOCC( 'I', 'I' ) :
         m_p_fnc_sw  = sr::conv_le_sw  ;
         m_p_fnc_uw  = sr::conv_le_uw  ;
         m_p_fnc_sdw = sr::conv_le_sdw ;
         m_p_fnc_udw = sr::conv_le_udw ;
         break ;
      /*-------------------------------------------------------------------*/
      case SR_TWOCC( 'M', 'M' ) :
         m_p_fnc_sw  = sr::conv_be_sw  ;
         m_p_fnc_uw  = sr::conv_be_uw  ;
         m_p_fnc_sdw = sr::conv_be_sdw ;
         m_p_fnc_udw = sr::conv_be_udw ;
         break ;
      /*-------------------------------------------------------------------*/
      default : return( -2 ) ;
      /*-------------------------------------------------------------------*/
   }

   /*--( Then there must be a constant )-----------------------------------*/
   switch( m_p_fnc_uw( header.w_mark ) )
   {
      /*-------------------------------------------------------------------*/
      case SR_TWOCC( '*', '\0' ) : // Tiff standard
         m_p_c_type_det = "tif" ;
         break ;
      /*-------------------------------------------------------------------*/
      case SR_TWOCC( 'R', 'O'  ) : // Raw Olympus (ORF)
      case SR_TWOCC( 'R', 'S'  ) : // Raw Olympus (ORF)
         m_p_c_type_det = "orf" ;
         break ;
      /*-------------------------------------------------------------------*/
      case SR_TWOCC( 'U', '\0' ) : // Raw Leica (RAW), Raw Panasonic (RW2)
         m_p_c_type_det = "raw" ;
         break ;
      /*-------------------------------------------------------------------*/
      default: return( -3 ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Read and check the offset of the next IFD )--------------------------*/
int CFileInit_tiff::read_offset_ifd( wxFileOffset &fo_offset_ifd )
{
   /*----------------------------------------------------------------------*/
   str_tiff_offset_ifd offset_ifd ;
   /*----------------------------------------------------------------------*/
   if(    !ok_to_read( m_fa.get_offset() - m_fo_offset, sizeof( offset_ifd ))
       || m_fa.read_buffer( sizeof( offset_ifd ), &offset_ifd ) != 0
     )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   fo_offset_ifd = m_p_fnc_udw( offset_ifd.dw_offset ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( On exit the file pointer is positioned at the end of the data )------*/
int CFileInit_tiff::read_exif( wxFileOffset fo_offset, wxULongLong ull_size )
{
   /*----------------------------------------------------------------------*/
   wxFileOffset fo_offset_ifd ;

   /*----------------------------------------------------------------------*/
   if( tiff_init( fo_offset, ull_size ) != 0 )
   {  return( -1 ) ; }

   /*--( Begin with the "full" image )-------------------------------------*/
   if( read_offset_ifd( fo_offset_ifd ) != 0 || fo_offset_ifd == 0 )
   {  return( -2 ) ; }
   /*-----------------------------------------------------------------------+
   ! The base data: dimensions and bpp have already been read in the        !
   ! "globals" of the jfif or the "general" part of the tiff                !
   +-----------------------------------------------------------------------*/
   if( read_ifd( fo_offset_ifd, TIFF_TCG_STANDARD ) != 0 )
   {  return( -3 ) ; }

   /*--( Then the thumbnail )----------------------------------------------*/
   if( read_offset_ifd( fo_offset_ifd ) != 0 || fo_offset_ifd == 0 )
   {  return( -4 ) ; }
   if( read_ifd( fo_offset_ifd, TIFF_TCG_THUMBNAIL ) != 0 )
   {  return( -5 ) ; }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( On exit the file pointer is positioned at the end of the data )------*/
int CFileInit_tiff::read_tiff( wxFileOffset fo_offset, wxULongLong ull_size )
{
   /*----------------------------------------------------------------------*/
   wxFileOffset fo_offset_ifd ;
   bool         boo_embedded_tif = ( fo_offset > 0 ) ;

   /*--( By default it is a "standard" tif type )--------------------------*/
   if( !boo_embedded_tif ) { m_fi.m_s_type_det = "tif" ; }

   /*----------------------------------------------------------------------*/
   if( tiff_init( fo_offset, ull_size ) != 0 )
   {  return( -1 ) ; }

   /*--( More accurate detection might have been set )---------------------*/
   if( !boo_embedded_tif ) { m_fi.m_s_type_det = m_p_c_type_det ; }

   /*----------------------------------------------------------------------*/
   m_fi.m_s_type_det = m_p_c_type_det ;

   /*--( Read the IFDs potentially chained )-------------------------------*/
   do
   {  /*--( Begin with the offset of the IFD )-----------------------------*/
      if( read_offset_ifd( fo_offset_ifd ) != 0 )
      {  return( -2 ) ; }

      /*--( Zero at the end means "end" )----------------------------------*/
      if(    fo_offset_ifd != 0
          && read_ifd( fo_offset_ifd, TIFF_TCG_ALL_EXCEPT_GPS ) != 0
        )
      {  return( -3 ) ; }

      /*-------------------------------------------------------------------*/
   } while( fo_offset_ifd != 0 ) ;

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_tiff()
{
   /*----------------------------------------------------------------------*/
   int i_ret ;
   /*----------------------------------------------------------------------*/
   if( ( i_ret = tiff_read_tiff( 0, m_f.get_size() ) ) == 0 )
   {  m_f.set_image_offset( 0 ) ; }
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
!                                                                           !
! Those functions are directly used by other init modules ...               !
!                                                                           !
+--------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
int CFileInit::tiff_read_exif( wxFileOffset fo_offset, wxULongLong ull_size )
{  return( CFileInit_tiff( *this ).read_exif( fo_offset, ull_size ) ) ; }

/*-------------------------------------------------------------------------*/
int CFileInit::tiff_read_tiff( wxFileOffset fo_offset, wxULongLong ull_size )
{  return( CFileInit_tiff( *this ).read_tiff( fo_offset, ull_size ) ) ; }

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                        End of file CFileInit_tiff.cpp                     !
+==========================================================================*/
