/*====================================+=====================================+
! File CFileInit_qtf.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                       QuickTime File Format                               !
!                                                                           !
! Audio file: ".m4a", ".m4b", ".m4r" ...                                    !
! Video file: ".mov", ".mp4" ...                                            !
!                                                                           !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! An issue with the "qtf" files is the compressed header.                   !
! The solution chosen is to use stream buffers to read the data: one for    !
! the uncompressed file content and one for zipped part.                    !
!                                                                           !
! Strings are assumed to be UTF8 encoded.                                   !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <wx/wfstream.h>
#include <wx/zstream.h>
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*--( Copyright ASCII character )------------------------------------------*/
const char co_c_copyright = 0xA9 ;
/*-------------------------------------------------------------------------*/
struct str_qtf_atom ;
struct str_qtf_mvhd ;
struct str_qtf_mdhd ;
struct str_qtf_hdlr ;
struct str_qtf_stsd_header ;
struct str_qtf_stsd_audio ;
struct str_qtf_stsd_video ;
struct str_qtf_stts ;

/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_qtf : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_qtf( CFileInit &parent )
                   : CFileInit_type_base( parent ),
                     m_boo_in_cmov( false ),
                     m_p_input( NULL ), // for cppcheck
                     m_e_mdia_type( CFileInit_qtf::E_MDIA_TYPE_NB ),
                     m_l_time_scale( 0 ), m_l_sample_duration( 0 ),
                     m_boo_has_video( false ), m_i_duration( -1 ),
                     m_i_nb_channel( -1 ), m_i_samprate( -1 )
      {  ; }

   /*----------------------------------------------------------------------*/
   public :
      /*--------------------------------------------------------------------+
      ! The "moov" atom can compressed. So It will be loaded in a buffer    !
      ! and read from it.                                                   !
      +--------------------------------------------------------------------*/
      bool          m_boo_in_cmov ;
      wxInputStream *m_p_input    ;

      /*--------------------------------------------------------------------+
      ! The validity duration of "m_e_type_media" is the same as the        !
      ! current "mdia" atom.                                                !
      +--------------------------------------------------------------------*/
      enum e_mdia_type
      {  E_MDIA_TYPE_AUDIO, E_MDIA_TYPE_VIDEO, E_MDIA_TYPE_NB } ;
      e_mdia_type m_e_mdia_type       ;
      /*-------------------------------------------------------------------*/
      long        m_l_time_scale      ;
      long        m_l_sample_duration ;
      /*--( Bitrate doesn't seem available ... )---------------------------*/
      bool        m_boo_has_video     ; // Video/Audio qtf file ?
      int         m_i_duration        ;
      int         m_i_nb_channel      ;
      int         m_i_samprate        ;
      wxString    m_s_audio_info      ;
      wxString    m_s_video_info      ;

   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      int skip_nb_byte( wxFileOffset val ) ;
      int read_buffer( void *p_buffer, size_t sz ) ;
      int read_tb_c( size_t sz_size, wxString &s ) ;
      int read_atom( str_qtf_atom &atom ) ;
      int read_data_no_swap( wxUint32 &dw ) ;
      int read_data( wxUint16 &w ) ;
      int read_data( wxUint32 &dw ) ;
      int read_data( wxUint64 &ddw ) ;
      int read_data( str_atom_short &atom_short ) ;
      int read_data( str_qtf_mvhd &mvhd ) ;
      int read_data( str_qtf_mdhd &mdhd ) ;
      int read_data( str_qtf_hdlr &hdlr ) ;
      int read_data( str_qtf_stsd_audio &stsd ) ;
      int read_data( str_qtf_stsd_video &stsd ) ;
      int read_data( str_qtf_stts &stts ) ;
      /*-------------------------------------------------------------------*/
      int read_atom_ilst( wxLongLong &ll_size_ilst ) ;
      int read_atom_meta( wxLongLong &ll_size_meta ) ;
      int read_atom_udta( wxLongLong &ll_size_udta ) ;
      int read_atom_stbl_audio( wxLongLong &ll_size_stbl ) ;
      int read_atom_stbl_video( wxLongLong &ll_size_stbl ) ;
      int read_atom_minf( wxLongLong &ll_size_minf ) ;
      int read_atom_mdia( wxLongLong &ll_size_mdia ) ;
      int read_atom_trak( wxLongLong &ll_size_trak ) ;
      int read_atom_moov( wxLongLong &ll_size_moov ) ;
      int qtf_read() ;

   /*----------------------------------------------------------------------*/
} ;

/*--------------------------------------------------------------------------+
! This structure is used to read qtf atoms:                                 !
! DO NOT USE IT FOR "DIRECT" READ !!!                                       !
+--------------------------------------------------------------------------*/
struct str_qtf_atom
{
   /*----------------------------------------------------------------------*/
   wxUint32   m_dw_id         ;
   wxLongLong m_ll_data_size  ;
   int        m_i_header_size ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*-------------------------------------------------------------------------*/
struct str_qtf_mvhd
{
   /*----------------------------------------------------------------------*/
   wxUint8  ___b_version            ;
   wxUint8  ___tb_b_flags[ 3 ]      ;
   wxUint32 ___dw_creation_time     ; // Sec since 01/01/1904
   wxUint32 ___dw_modification_time ; // Sec since 01/01/1904
   /*----------------------------------------------------------------------*/
   wxUint32 dw_time_scale           ; // time units / sec
   wxUint32 dw_duration             ; // duration in time scale units
/*
   BYTE tb_b_preferred_rate    [  4 ] ;
   BYTE tb_b_preferred_volume  [  2 ] ;
   BYTE tb_b_reserved          [ 10 ] ; // 0
   BYTE tb_b_matrix            [ 36 ] ;
   BYTE tb_b_preview_time      [  4 ] ;
   BYTE tb_b_preview_duration  [  4 ] ;
   BYTE tb_b_poster_time       [  4 ] ;
   BYTE tb_b_selection_time    [  4 ] ;
   BYTE tb_b_selection_duration[  4 ] ;
   BYTE tb_b_current_time      [  4 ] ;
   BYTE tb_b_next_track_id     [  4 ] ;
*/
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_qtf_mdhd
{
   /*----------------------------------------------------------------------*/
   wxUint8  ___b_version            ;
   wxUint8  ___tb_b_flags[ 3 ]      ;
   wxUint32 ___dw_creation_time     ; // Sec since 01/01/1904
   wxUint32 ___dw_modification_time ; // Sec since 01/01/1904
   /*----------------------------------------------------------------------*/
   wxUint32 dw_time_scale           ; // time units / sec
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_qtf_hdlr
{
   /*----------------------------------------------------------------------*/
   wxUint8  ___b_version                 ;
   wxUint8  ___tb_b_flags[ 3 ]           ;
   wxUint32 ___dw_component_type         ;
   wxUint32 dw_component_subtype         ; // video/sound
   wxUint32 ___dw_component_manufacturer ;
   wxUint32 ___dw_component_flags        ;
   wxUint32 ___dw_component_flags_mask   ;
   /* Variable string component name */  ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_qtf_stsd_header
{
   /*----------------------------------------------------------------------*/
   wxUint8  ___b_version_0                ;
   wxUint8  ___tb_b_flags[ 3 ]            ;
   wxUint32 ___dw_number_of_entries       ;
   /*----------------------------------------------------------------------*/
   wxUint32 ___dw_sample_description_size ;
   wxUint32 dw_data_format                ;
   wxUint8  ___tb_reserved[ 6 ]           ;
   wxUint16 ___w_data_reference_index     ;
   /*----------------------------------------------------------------------*/
   wxUint16 ___w_version_1                ;
   wxUint16 ___w_revision_level           ;
   wxUint32 ___dw_vendor                  ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_qtf_stsd_audio
{
   /*----------------------------------------------------------------------*/
   str_qtf_stsd_header header             ;
   /*----------------------------------------------------------------------*/
   wxUint16 w_number_of_channels          ;
   wxUint16 ___w_sample_size              ;
   wxUint16 ___w_compression_id           ;
   wxUint16 ___w_packet_size              ;
   wxUint32 ___w_sample_rate              ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_qtf_stsd_video
{
   /*----------------------------------------------------------------------*/
   str_qtf_stsd_header header             ;
   /*----------------------------------------------------------------------*/
   wxUint32 ___dw_temporal_quality        ;
   wxUint32 ___dw_spacial_quality         ;
   wxUint16 w_width                       ;
   wxUint16 w_height                      ;
   wxUint32 ___dw_horizontal_resolution   ;
   wxUint32 ___dw_vertival_resolution     ;
   wxUint32 ___dw_data_size               ;
   wxUint16 ___w_frame_count              ;
   wxUint32 ___dw_compressor_name         ;
   wxUint16 ___w_depth                    ;
   wxUint16 ___w_color_table_id           ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_qtf_stts
{
   /*----------------------------------------------------------------------*/
   wxUint8  ___b_version_0                ;
   wxUint8  ___tb_b_flags[ 3 ]            ;
   wxUint32 ___dw_number_of_entries       ;
   /*----------------------------------------------------------------------*/
   wxUint32 ___dw_sample_count            ;
   wxUint32 dw_sample_duration            ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::skip_nb_byte( wxFileOffset val )
{
   /*--( wxZlibInputStream is not seekable )-------------------------------*/
   if( m_boo_in_cmov )
   {  /*--( Read to simulate the skip )------------------------------------*/
      char   tb_c_dummy[ 1024 ] ;
      size_t sz_read            ;
      /*-------------------------------------------------------------------*/
      for( ; val > 0 ; val -= sz_read )
      {
         /*----------------------------------------------------------------*/
         sz_read = wxMin( val, ( int )sizeof( tb_c_dummy ) ) ;
         /*----------------------------------------------------------------*/
         if( m_p_input->Read( tb_c_dummy, sz_read ).LastRead() != sz_read )
         {  return( -1 ) ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   else
   if( m_p_input->SeekI( val, wxFromCurrent ) == wxInvalidOffset )
   {  return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_buffer( void *p_buffer, size_t sz )
{
   /*----------------------------------------------------------------------*/
   if( m_p_input->Read( p_buffer, sz ).LastRead() != sz )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_tb_c( size_t sz_size, wxString &s )
{
   /*----------------------------------------------------------------------*/
   char   tb_c_val[ g_co_i_string_sz_max * 2 ] ;
   size_t sz_size_to_read ;

   /*----------------------------------------------------------------------*/
   s.clear() ;
   /*----------------------------------------------------------------------*/
   if( sz_size == 0 ) { return( 0 ) ; }
   /*----------------------------------------------------------------------*/
   sz_size_to_read = wxMin( sz_size, sizeof( tb_c_val ) ) ;
   /*----------------------------------------------------------------------*/
   if(    read_buffer( tb_c_val, sz_size_to_read ) != 0
       || (    sz_size_to_read < sz_size
            && skip_nb_byte( sz_size - sz_size_to_read ) != 0
          )
     )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   sr::prepare_string( tb_c_val, sz_size_to_read, s, sr::CONV_FROM_UTF8 ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! This function takes into account the special meaning of atom's size.      !
+--------------------------------------------------------------------------*/
int CFileInit_qtf::read_atom( str_qtf_atom &atom )
{
   /*----------------------------------------------------------------------*/
   str_atom atom_base ;
   /*----------------------------------------------------------------------*/
   if( read_buffer( &atom_base, sizeof( atom_base ) ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   atom_base.dw_size = wxUINT32_SWAP_ALWAYS( atom_base.dw_size ) ;
// atom.dw_id   = wxUINT32_SWAP_ALWAYS( atom.dw_id   ) ;
#endif
   /*----------------------------------------------------------------------*/
   atom.m_dw_id = atom_base.dw_id ;
   atom.m_i_header_size = sizeof( atom_base ) ;
   /*-----------------------------------------------------------------------+
   ! Two size values have special meanings :                                !
   ! 0 => The real size is the file size minus the offset of the atom       !
   ! 1 => The real size is in a following 64 bits integer                   !
   +-----------------------------------------------------------------------*/
   if( atom_base.dw_size == 0 )
   {
      /*--( Doesn't sound right in the compressed part )-------------------*/
      if( m_boo_in_cmov ) { return( -2 ) ; }
      /*-------------------------------------------------------------------*/
        atom.m_ll_data_size
      = m_f.get_size() - m_p_input->TellI() - sizeof( atom_base ) ;
      /*-------------------------------------------------------------------*/
   }
   else
   if( atom_base.dw_size == 1 )
   {
      /*-------------------------------------------------------------------*/
      wxUint64 ddw_size ;
      /*-------------------------------------------------------------------*/
      if( read_data( ddw_size ) != 0 ) { return( -3 ) ; }
      /*-------------------------------------------------------------------*/
      atom.m_i_header_size += sizeof( ddw_size ) ;
      atom.m_ll_data_size   = ddw_size ;
      /*-------------------------------------------------------------------*/
   }
   else
   {  atom.m_ll_data_size = atom_base.dw_size ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_data_no_swap( wxUint32 &dw )
{
   /*----------------------------------------------------------------------*/
   if( read_buffer( &dw, sizeof( dw ) ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_data( wxUint16 &w )
{
   /*----------------------------------------------------------------------*/
   if( read_buffer( &w, sizeof( w ) ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   w = wxUINT16_SWAP_ALWAYS( w ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_data( wxUint32 &dw )
{
   /*----------------------------------------------------------------------*/
   if( read_buffer( &dw, sizeof( dw ) ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   dw = wxUINT32_SWAP_ALWAYS( dw ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_data( wxUint64 &ddw )
{
   /*----------------------------------------------------------------------*/
   if( read_buffer( &ddw, sizeof( ddw ) ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   ddw = wxUINT64_SWAP_ALWAYS( ddw ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_data( str_atom_short &atom_short )
{
   /*----------------------------------------------------------------------*/
   if( read_buffer( &atom_short, sizeof( atom_short ) ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   atom_short.w_size = wxUINT16_SWAP_ALWAYS( atom_short.w_size ) ;
// atom_short.w_id   = wxUINT16_SWAP_ALWAYS( atom_short.w_id   ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_data( str_qtf_mvhd &mvhd )
{
   /*----------------------------------------------------------------------*/
   if( read_buffer( &mvhd, sizeof( mvhd ) ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   mvhd.dw_time_scale = wxUINT32_SWAP_ALWAYS( mvhd.dw_time_scale ) ;
   mvhd.dw_duration   = wxUINT32_SWAP_ALWAYS( mvhd.dw_duration   ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_data( str_qtf_mdhd &mdhd )
{
   /*----------------------------------------------------------------------*/
   if( read_buffer( &mdhd, sizeof( mdhd ) ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   mdhd.dw_time_scale = wxUINT32_SWAP_ALWAYS( mdhd.dw_time_scale ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_data( str_qtf_hdlr &hdlr )
{
   /*----------------------------------------------------------------------*/
   if( read_buffer( &hdlr, sizeof( hdlr ) ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_data( str_qtf_stsd_audio &stsd )
{
   /*----------------------------------------------------------------------*/
   if( read_buffer( &stsd, sizeof( stsd ) ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
     stsd.w_number_of_channels
   = wxUINT16_SWAP_ALWAYS( stsd.w_number_of_channels ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_data( str_qtf_stsd_video &stsd )
{
   /*----------------------------------------------------------------------*/
   if( read_buffer( &stsd, sizeof( stsd ) ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   stsd.w_width  = wxUINT16_SWAP_ALWAYS( stsd.w_width  ) ;
   stsd.w_height = wxUINT16_SWAP_ALWAYS( stsd.w_height ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_data( str_qtf_stts &stts )
{
   /*----------------------------------------------------------------------*/
   if( read_buffer( &stts, sizeof( stts ) ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   stts.dw_sample_duration = wxUINT32_SWAP_ALWAYS( stts.dw_sample_duration );
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static int st_qtf_conv_tag_col( bool boo_video, wxUint32 dw_tag )
{
   /*----------------------------------------------------------------------*/
   switch( dw_tag )
   {
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( co_c_copyright,'d','a','y' ) :
         return( boo_video ? COL_VIDTAG_CRE_DATE : COL_AUDTAG_YEAR ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( co_c_copyright,'c','m','t' ) :
         return( boo_video ? COL_VIDTAG_COMMENT : COL_AUDTAG_COMMENT ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( co_c_copyright,'n','a','m' ) :
         return( boo_video ? COL_VIDTAG_TITLE : COL_AUDTAG_TITLE ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( co_c_copyright,'s','r','c' ) :
      case SR_FOURCC( co_c_copyright,'a','u','t' ) :
         return( boo_video ? COL_VIDTAG_SOURCE : COL_NB ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( co_c_copyright,'A','R','T' ) :
         return( boo_video ? COL_VIDTAG_ARTIST : COL_AUDTAG_ARTIST ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( 'a','A','R','T' ) :
         return( boo_video ? COL_VIDTAG_ARTIST : COL_AUDTAG_ORG_ART ) ;
      /*--( iTunes seems to use this one for videos too )------------------*/
      case SR_FOURCC( co_c_copyright,'a','l','b' ) :
         return( boo_video ? COL_VIDTAG_SOURCE : COL_AUDTAG_ALBUM ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( co_c_copyright,'w','r','t' ) :
         return( boo_video ? COL_NB : COL_AUDTAG_COMPOS ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( co_c_copyright,'p','r','d' ) :
         return( boo_video ? COL_VIDTAG_ENGINEER : COL_NB ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( co_c_copyright,'d','i','r' ) :
         return( boo_video ? COL_VIDTAG_TECHNICIAN : COL_NB ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( co_c_copyright,'p','r','f' ) :
         return( boo_video ? COL_VIDTAG_SUBJECT : COL_NB ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( 'k','e','y','w' ) :
         return( boo_video ? COL_VIDTAG_KEYWORDS : COL_NB ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( co_c_copyright,'d','e','s' ) :
      case SR_FOURCC( 'd','e','s','c' ) :
         return( boo_video ? COL_VIDTAG_SUBJECT : COL_NB ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( co_c_copyright,'c','p','y' ) :
      case SR_FOURCC( 'c','p','r','t' ) :
         return( boo_video ? COL_VIDTAG_COPYRIGHT : COL_AUDTAG_COPYRIGHT ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( co_c_copyright,'g','e','n' ) :
      case SR_FOURCC( 'g','n','r','e' ) :
         return( boo_video ? COL_VIDTAG_GENRE : COL_AUDTAG_GENRE ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( co_c_copyright,'s','w','r' ) :
      case SR_FOURCC( co_c_copyright,'t','o','o' ) :
         return( boo_video ? COL_VIDTAG_SOFTWARE : COL_AUDTAG_ENCOD_BY ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( COL_NB ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_atom_ilst( wxLongLong &ll_size_ilst )
{
   /*----------------------------------------------------------------------*/
   str_qtf_atom atom            ;
   str_qtf_atom atom_data       ;
   wxUint32     dw_type         ;
   int          i_col           ;
   wxLongLong   ll_size         ;
   wxLongLong   ll_size_to_skip ;
   bool         boo_supported   ;
   wxString     s_val           ;

   /*----------------------------------------------------------------------*/
   while(    ll_size_ilst >= sizeof( str_atom )
          && read_atom( atom ) == 0
        )
   {
      /*-------------------------------------------------------------------*/
      ll_size         = atom.m_ll_data_size ;
      ll_size_to_skip = ll_size - atom.m_i_header_size ;
      /*--( Then comes the "data" atom )-----------------------------------*/
      if(    ll_size_to_skip < sizeof( str_atom )
          || read_atom( atom_data ) != 0
        )
      {  return( -1 ) ; }
      /*-------------------------------------------------------------------*/
      ll_size_to_skip -= atom_data.m_i_header_size ;

      /*--( It can contain "mean" associated to "---" tags ... )-----------*/
      boo_supported = ( atom_data.m_dw_id == SR_FOURCC( 'd','a','t','a' ) ) ;

      /*--( Then the data type and 4 other bytes to skip )-----------------*/
      if( boo_supported )
      {  /*----------------------------------------------------------------*/
         if(    ll_size_to_skip < 2 * sizeof( dw_type )
             || read_data( dw_type ) != 0
             || skip_nb_byte( sizeof( wxUint32 ) ) != 0
           )
         {  return( -2 ) ; }
         /*----------------------------------------------------------------*/
         ll_size_to_skip -= 2 * sizeof( dw_type ) ;
         /*----------------------------------------------------------------*/
      }

      /*--( Skip the data ? )----------------------------------------------*/
      if( !boo_supported )
      {  ; }
      else /*--( Image included ? )----------------------------------------*/
      if( atom.m_dw_id == SR_FOURCC( 'c','o','v','r' ) )
      {
         /*-----------------------------------------------------------------+
         ! The image info extraction doesn't handle the wxInputStream       !
         ! input.                                                           !
         +-----------------------------------------------------------------*/
         if( !m_boo_in_cmov && m_f.get_image_offset() == -1 )
         {
            /*-------------------------------------------------------------*/
            wxString s_mime ;
            /*--( Convert type to a "mime" understandable by the function )*/
            switch( dw_type )
            {  case 0x0D : s_mime = "jpg" ; break ;
               case 0x0E : s_mime = "png" ; break ;
            }
            /*--( Supported type ? )---------------------------------------*/
            if( !s_mime.empty() )
            {
               /*----------------------------------------------------------*/
               m_f.val_s( m_boo_has_video
                          ? COL_VIDTAG_IMG_FORMAT : COL_AUDTAG_IMG_FORMAT
                        ) = s_mime ;
               /*----------------------------------------------------------*/
               m_fi.read_embedded_image_info( m_fa.get_offset(),
                                              ll_size_to_skip.GetValue(),
                                              s_mime
                                            ) ;
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      else /*--( Some work has to be done for some numerical data )--------*/
      if( dw_type == 0 )
      {
         /*----------------------------------------------------------------*/
         switch( atom.m_dw_id )
         {  /*-------------------------------------------------------------*/
            case SR_FOURCC( co_c_copyright,'g','e','n' ) :
            case SR_FOURCC( 'g','n','r','e' ) :
            {
               /*----------------------------------------------------------*/
               wxUint16 w_genre ;
               /*----------------------------------------------------------*/
               if(    m_boo_has_video
                   || ll_size_to_skip < sizeof( w_genre )
                   || read_data( w_genre ) != 0
                 )
               {  break ; }
               /*----------------------------------------------------------*/
               ll_size_to_skip -= sizeof( w_genre ) ;
               /*----------------------------------------------------------*/
               --w_genre ;
               /*----------------------------------------------------------*/
               if(    m_fi.reserve_col( COL_AUDTAG_GENRE )
                   && m_fi.mp3_genre_id_ok( w_genre )
                 )
               {    m_f.val_s( COL_AUDTAG_GENRE )
                  = m_fi.mp3_get_genre( w_genre ) ;
               }
               /*----------------------------------------------------------*/
               break ;
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
            case SR_FOURCC( 't','r','k','n' ) :
            {
               /*----------------------------------------------------------*/
               wxUint32 dw_num ;
               wxUint16 w_nb   ;
               /*--( Track number and number of tacks )--------------------*/
               if(    m_boo_has_video
                   || ll_size_to_skip < sizeof( dw_num ) + sizeof( w_nb )
                   || read_data( dw_num ) != 0
                   || read_data( w_nb ) != 0
                 )
               {  break ; }
               /*----------------------------------------------------------*/
               ll_size_to_skip -= sizeof( dw_num ) + sizeof( w_nb ) ;
               /*----------------------------------------------------------*/
               if( m_fi.reserve_col( COL_AUDTAG_TRACK_NUM ) )
               {  m_f.val_ll( COL_AUDTAG_TRACK_NUM ) = dw_num ;
                  m_f.val_s( COL_AUDTAG_TRACK_NUM ) << dw_num ;
               }
               /*----------------------------------------------------------*/
               if( w_nb > 0 && m_fi.reserve_col( COL_AUDTAG_TRACK_NB ) )
               {  m_f.val_ll( COL_AUDTAG_TRACK_NB ) = w_nb ;
                  m_f.val_s( COL_AUDTAG_TRACK_NB ) << w_nb ;
               }
               /*----------------------------------------------------------*/
               break ;
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
            case SR_FOURCC( 'd','i','s','k' ) :
            {
               /*----------------------------------------------------------*/
               wxUint32 dw_nb ;
               /*----------------------------------------------------------*/
               if(    m_boo_has_video
                   || ll_size_to_skip < sizeof( dw_nb )
                   || read_data( dw_nb ) != 0
                 )
               {  break ; }
               /*----------------------------------------------------------*/
               ll_size_to_skip -= sizeof( dw_nb ) ;
               /*----------------------------------------------------------*/
               if( m_fi.reserve_col( COL_AUDTAG_DISK ) )
               {  m_f.val_ll( COL_AUDTAG_DISK ) = dw_nb ;
                  m_f.val_s( COL_AUDTAG_DISK ) << dw_nb ;
               }
               /*----------------------------------------------------------*/
               break ;
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      else /*--( Else only the text information is kept )------------------*/
      if( dw_type == 1 )
      {
         /*--( Convert id to "internal" col )------------------------------*/
         i_col = st_qtf_conv_tag_col( m_boo_has_video, atom.m_dw_id ) ;
         /*----------------------------------------------------------------*/
         if( m_fi.reserve_col( i_col ) )
         {
            /*-------------------------------------------------------------*/
            if(    read_tb_c( ll_size_to_skip.GetValue(), s_val ) == 0
                && !s_val.empty()
              )
            {  m_f.val_s( i_col ) = s_val ; }
            /*-------------------------------------------------------------*/
            ll_size_to_skip = 0 ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      if(    ll_size_to_skip > 0
          && skip_nb_byte( ll_size_to_skip.GetValue() ) != 0
        )
      {  return( -4 ) ; }
      /*-------------------------------------------------------------------*/
      ll_size_ilst -= ll_size ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_atom_meta( wxLongLong &ll_size_meta )
{
   /*----------------------------------------------------------------------*/
   str_qtf_atom atom            ;
   wxLongLong   ll_size         ;
   wxLongLong   ll_size_to_skip ;

   /*--( The first four bytes are useless )--------------------------------*/
   if( skip_nb_byte( sizeof( wxUint32 ) ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   ll_size_meta -= sizeof( wxUint32 ) ;

   /*----------------------------------------------------------------------*/
   while(    ll_size_meta >= sizeof( str_atom )
          && read_atom( atom ) == 0
        )
   {
      /*-------------------------------------------------------------------*/
      ll_size         = atom.m_ll_data_size ;
      ll_size_to_skip = ll_size - atom.m_i_header_size ;
      /*-------------------------------------------------------------------*/
      if(    atom.m_dw_id == SR_FOURCC( 'i','l','s','t' )
          && read_atom_ilst( ll_size_to_skip ) != 0
        )
      {   return( -2 ) ; }

      /*-------------------------------------------------------------------*/
      if(    ll_size_to_skip > 0
          && skip_nb_byte( ll_size_to_skip.GetValue() ) != 0
        )
      {  return( -3 ) ; }
      /*-------------------------------------------------------------------*/
      ll_size_meta -= ll_size ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_atom_udta( wxLongLong &ll_size_udta )
{
   /*----------------------------------------------------------------------*/
   str_qtf_atom atom            ;
   int          i_col           ;
   wxLongLong   ll_size         ;
   wxLongLong   ll_size_to_skip ;
   wxString     s_val           ;
   bool         boo_end         ;

   /*----------------------------------------------------------------------*/
   if(    ll_size_udta < sizeof( str_atom )
       || read_atom( atom ) != 0
     )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   ll_size         = atom.m_ll_data_size ;
   ll_size_to_skip = ll_size - atom.m_i_header_size ;
   /*----------------------------------------------------------------------+
   ! iTunes metadata ?                                                     !
   +----------------------------------------------------------------------*/
   if( atom.m_dw_id == SR_FOURCC( 'm','e','t','a' ) )
   {
      /*--( All the data left are "used/skipped" by the meta atom func )--*/
      ll_size_udta = 0 ;
      /*------------------------------------------------------------------*/
      return( read_atom_meta( ll_size_to_skip ) ) ;
      /*------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------+
   ! Or standard info                                                      !
   +----------------------------------------------------------------------*/
   do
   {
      /*--( Convert id to "internal" col )---------------------------------*/
      i_col = st_qtf_conv_tag_col( m_boo_has_video, atom.m_dw_id ) ;
      /*-------------------------------------------------------------------*/
      if( m_fi.reserve_col( i_col ) )
      {
         /*-----------------------------------------------------------------+
         ! If it begins with a "(c)" then the atom is short with size       !
         ! and type                                                         !
         +-----------------------------------------------------------------*/
         if( *( ( char * )&atom.m_dw_id ) == co_c_copyright )
         {  /*-------------------------------------------------------------*/
            if( skip_nb_byte( sizeof( str_atom_short ) ) != 0 )
            {  return( -2 ) ; }
            /*-------------------------------------------------------------*/
            ll_size_to_skip -= sizeof( str_atom_short ) ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         if(    read_tb_c( ll_size_to_skip.GetValue(), s_val ) == 0
             && !s_val.empty()
           )
         {  m_f.val_s( i_col ) = s_val ; }
         /*----------------------------------------------------------------*/
         ll_size_to_skip = 0 ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      if(    ll_size_to_skip > 0
          && skip_nb_byte( ll_size_to_skip.GetValue() ) != 0
        )
      {  return( -3 ) ; }
      /*-------------------------------------------------------------------*/
      ll_size_udta -= ll_size ;

      /*--( Get next atom )------------------------------------------------*/
      boo_end = (    ll_size_udta < sizeof( str_atom )
                  || read_atom( atom ) != 0
                ) ;
      /*-------------------------------------------------------------------*/
      if( !boo_end )
      {  ll_size         = atom.m_ll_data_size ;
         ll_size_to_skip = ll_size - atom.m_i_header_size ;
      }
      /*-------------------------------------------------------------------*/
   } while( !boo_end ) ;

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static wxString st_qtf_info_fcc_audio( wxUint32 dw_fcc )
{
   /*----------------------------------------------------------------------*/
   switch( dw_fcc )
   {
      /*-------------------------------------------------------------------*/
      case SR_FOURCC('N','O','N','E') : return( "PCM" ) ;
      case SR_FOURCC('r','a','w',' ') : return( "PCM 8 bit" ) ;
      case SR_FOURCC('t','w','o','s') : return( "PCM 16 bit BE" ) ;
      case SR_FOURCC('s','o','w','t') : return( "PCM 16 bit LE" ) ;
      case SR_FOURCC('f','l','3','2') : return( "PCM 32 bit float" ) ;
      case SR_FOURCC('f','l','6','4') : return( "PCM 64 bit float" ) ;
      case SR_FOURCC('i','n','2','4') : return( "PCM 24 bit BE" ) ;
      case SR_FOURCC('i','n','3','2') : return( "PCM 32 bit BE" ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC('a','l','a','w') : return( "ALAW" ) ;
      case SR_FOURCC('u','l','a','w') : return( "MULAW" ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC('d','v','c','a') : return( "DV Audio" ) ;
      case SR_FOURCC('Q','D','M','C') : return( "QDesign" ) ;
      case SR_FOURCC('Q','D','M','2') : return( "QDesign 2" ) ;
      case SR_FOURCC('Q','c','l','p') : return( "Qualcomm" ) ;
      case SR_FOURCC('m','p','4','a') : return( "AAC" ) ;
      case SR_FOURCC('s','a','m','r') : return( "AMR Audio" ) ;
      /*-------------------------------------------------------------------*/
      case SR_FOURCC( 0x6D,0x73,0x00,0x55 ) :
      case SR_FOURCC( '.','m','p','3' ) :
                                        return( "Mpeg L3" ) ;
      /*-------------------------------------------------------------------*/
      default :
      {
         /*----------------------------------------------------------------*/
         const char *p_c_fcc = ( char * )&dw_fcc ;
         /*----------------------------------------------------------------*/
         return(   "Audio ("
                 + sr::byte_to_string( p_c_fcc[ 0 ] )
                 + sr::byte_to_string( p_c_fcc[ 1 ] )
                 + sr::byte_to_string( p_c_fcc[ 2 ] )
                 + sr::byte_to_string( p_c_fcc[ 3 ] )
                 + ')'
               ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_atom_stbl_audio( wxLongLong &ll_size_stbl )
{
   /*----------------------------------------------------------------------*/
   str_qtf_atom atom            ;
   wxLongLong   ll_size         ;
   wxLongLong   ll_size_to_skip ;

   /*----------------------------------------------------------------------*/
   while(    ll_size_stbl >= sizeof( str_atom )
          && read_atom( atom ) == 0
        )
   {
      /*-------------------------------------------------------------------*/
      ll_size         = atom.m_ll_data_size ;
      ll_size_to_skip = ll_size - atom.m_i_header_size ;
      /*-------------------------------------------------------------------*/
      switch( atom.m_dw_id )
      {
         /*----------------------------------------------------------------*/
         case SR_FOURCC( 's','t','s','d' ) :
         {
            /*-------------------------------------------------------------*/
            str_qtf_stsd_audio stsd   ;
            wxString           s_info ;
            /*-------------------------------------------------------------*/
            if( ll_size < sizeof( stsd ) ) { break ; }
            /*-------------------------------------------------------------*/
            if( read_data( stsd ) != 0 ) { return( -1 ) ; }
            /*-------------------------------------------------------------*/
            ll_size_to_skip -= sizeof( stsd ) ;
            /*--------------------------------------------------------------+
            ! The sampling rate here can be zero that's why the value of    !
            ! the "mdhd" atom is used.                                      !
            +--------------------------------------------------------------*/
            m_i_nb_channel = stsd.w_number_of_channels ;
            /*--( fcc are not swapped )------------------------------------*/
            m_fi.add_codec_info(
                         st_qtf_info_fcc_audio( stsd.header.dw_data_format ),
                         m_s_audio_info
                               ) ;
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      if(    ll_size_to_skip > 0
          && skip_nb_byte( ll_size_to_skip.GetValue() ) != 0
        )
      {  return( -2 ) ; }
      /*-------------------------------------------------------------------*/
      ll_size_stbl -= ll_size ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_atom_stbl_video( wxLongLong &ll_size_stbl )
{
   /*----------------------------------------------------------------------*/
   str_qtf_atom atom            ;
   wxLongLong   ll_size         ;
   wxLongLong   ll_size_to_skip ;

   /*----------------------------------------------------------------------*/
   while(    ll_size_stbl >= sizeof( str_atom )
          && read_atom( atom ) == 0
        )
   {
      /*-------------------------------------------------------------------*/
      ll_size         = atom.m_ll_data_size ;
      ll_size_to_skip = ll_size - atom.m_i_header_size ;
      /*-------------------------------------------------------------------*/
      switch( atom.m_dw_id )
      {
         /*----------------------------------------------------------------*/
         case SR_FOURCC( 's','t','s','d' ) :
         {
            /*-------------------------------------------------------------*/
            str_qtf_stsd_video stsd   ;
            wxString           s_info ;
            /*-------------------------------------------------------------*/
            if( ll_size < sizeof( stsd ) ) { break ; }
            /*-------------------------------------------------------------*/
            if( read_data( stsd ) != 0 ) { return( -1 ) ; }
            /*-------------------------------------------------------------*/
            ll_size_to_skip -= sizeof( stsd ) ;
            /*-------------------------------------------------------------*/
            m_fi.info_fcc_video( ( char * )&stsd.header.dw_data_format,
                                 s_info
                               ) ;
            m_fi.add_codec_info( s_info, m_s_video_info ) ;
            /*-------------------------------------------------------------*/
            if( stsd.w_width != 0 && stsd.w_height != 0 )
            {  m_fi.init_video_x_y( stsd.w_width, stsd.w_height ) ; }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }

         /*----------------------------------------------------------------*/
         case SR_FOURCC( 's','t','t','s' ) :
         {
            /*-------------------------------------------------------------*/
            str_qtf_stts stts ;
            /*-------------------------------------------------------------*/
            if( ll_size < sizeof( stts ) ) { break ; }
            /*-------------------------------------------------------------*/
            if( read_data( stts ) != 0 ) { return( -2 ) ; }
            /*-------------------------------------------------------------*/
            ll_size_to_skip -= sizeof( stts ) ;
            m_l_sample_duration = stts.dw_sample_duration ;
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      if(    ll_size_to_skip > 0
          && skip_nb_byte( ll_size_to_skip.GetValue() ) != 0
        )
      {  return( -3 ) ; }
      /*-------------------------------------------------------------------*/
      ll_size_stbl -= ll_size ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_atom_minf( wxLongLong &ll_size_minf )
{
   /*----------------------------------------------------------------------*/
   str_qtf_atom atom            ;
   wxLongLong   ll_size         ;
   wxLongLong   ll_size_to_skip ;

   /*----------------------------------------------------------------------*/
   while(    ll_size_minf >= sizeof( str_atom )
          && read_atom( atom ) == 0
        )
   {
      /*-------------------------------------------------------------------*/
      ll_size         = atom.m_ll_data_size ;
      ll_size_to_skip = ll_size - atom.m_i_header_size ;
      /*-------------------------------------------------------------------*/
      switch( atom.m_dw_id )
      {
         /*--( Sound media information atom )------------------------------*/
         case SR_FOURCC( 's','m','h','d' ) :
            m_e_mdia_type = CFileInit_qtf::E_MDIA_TYPE_AUDIO ;
            break ;
         /*--( Video media information atom )------------------------------*/
         case SR_FOURCC( 'v','m','h','d' ) :
            m_e_mdia_type = CFileInit_qtf::E_MDIA_TYPE_VIDEO ;
            break ;
         /*----------------------------------------------------------------*/
         case SR_FOURCC( 's','t','b','l' ) :
            /*-------------------------------------------------------------*/
            switch( m_e_mdia_type )
            {  /*----------------------------------------------------------*/
               case CFileInit_qtf::E_MDIA_TYPE_AUDIO :
                  read_atom_stbl_audio( ll_size_to_skip ) ;
                  break ;
               /*----------------------------------------------------------*/
               case CFileInit_qtf::E_MDIA_TYPE_VIDEO :
                  read_atom_stbl_video( ll_size_to_skip ) ;
                  break ;
               /*----------------------------------------------------------*/
               default : break ;
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
            break ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      if(    ll_size_to_skip > 0
          && skip_nb_byte( ll_size_to_skip.GetValue() ) != 0
        )
      {  return( -1 ) ; }
      /*-------------------------------------------------------------------*/
      ll_size_minf -= ll_size ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_atom_mdia( wxLongLong &ll_size_mdia )
{
   /*----------------------------------------------------------------------*/
   str_qtf_atom atom             ;
   long         l_time_scale = 0 ;
   wxLongLong   ll_size          ;
   wxLongLong   ll_size_to_skip  ;

   /*----------------------------------------------------------------------*/
   m_e_mdia_type = CFileInit_qtf::E_MDIA_TYPE_NB ;
   /*----------------------------------------------------------------------*/
   while(    ll_size_mdia >= sizeof( str_atom )
          && read_atom( atom ) == 0
        )
   {
      /*-------------------------------------------------------------------*/
      ll_size         = atom.m_ll_data_size ;
      ll_size_to_skip = ll_size - atom.m_i_header_size ;
      /*-------------------------------------------------------------------*/
      switch( atom.m_dw_id )
      {
         /*--( Media Header Atom )-----------------------------------------*/
         case SR_FOURCC( 'm','d','h','d' ) :
         {
            /*-------------------------------------------------------------*/
            str_qtf_mdhd mdhd ;
            /*-------------------------------------------------------------*/
            if( ll_size < sizeof( mdhd ) ) { break ; }
            /*-------------------------------------------------------------*/
            if( read_data( mdhd ) != 0 ) { return( -1 ) ; }
            /*-------------------------------------------------------------*/
            ll_size_to_skip -= sizeof( mdhd ) ;
            /*-------------------------------------------------------------*/
            l_time_scale = mdhd.dw_time_scale ;
            if( l_time_scale <= 0 ) { break ; }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*--( Handler Reference Atom )------------------------------------*/
         case SR_FOURCC( 'h','d','l','r' ) :
         {
            /*-------------------------------------------------------------*/
            str_qtf_hdlr hdlr ;
            /*-------------------------------------------------------------*/
            if( ll_size < sizeof( hdlr ) ) { break ; }
            /*-------------------------------------------------------------*/
            if( read_data( hdlr ) != 0 ) { return( -1 ) ; }
            /*-------------------------------------------------------------*/
            ll_size_to_skip -= sizeof( hdlr ) ;
            /*-------------------------------------------------------------*/
            switch( hdlr.dw_component_subtype )
            {
               /*--( Sound media information atom )------------------------*/
               case SR_FOURCC( 's','o','u','n' ) :
                  m_e_mdia_type = CFileInit_qtf::E_MDIA_TYPE_AUDIO ;
                  break ;
               /*--( Video media information atom )------------------------*/
               case SR_FOURCC( 'v','i','d','e' ) :
                  m_e_mdia_type = CFileInit_qtf::E_MDIA_TYPE_VIDEO ;
                  break ;
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*--( Media Information atom )------------------------------------*/
         case SR_FOURCC( 'm','i','n','f' ) :
            read_atom_minf( ll_size_to_skip ) ;
            break ;
         /*--( User defined data atom )------------------------------------*/
         case SR_FOURCC( 'u','d','t','a' ) :
            read_atom_udta( ll_size_to_skip ) ;
            break ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      if(    ll_size_to_skip > 0
          && skip_nb_byte( ll_size_to_skip.GetValue() ) != 0
        )
      {  return( -2 ) ; }
      /*-------------------------------------------------------------------*/
      ll_size_mdia -= ll_size ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( m_e_mdia_type == CFileInit_qtf::E_MDIA_TYPE_VIDEO )
   {  m_boo_has_video = true ; }

   /*----------------------------------------------------------------------*/
   if( l_time_scale > 0 )
   {
      /*-------------------------------------------------------------------*/
      switch( m_e_mdia_type )
      {
         /*----------------------------------------------------------------*/
         case CFileInit_qtf::E_MDIA_TYPE_VIDEO :
            m_l_time_scale = l_time_scale ;
            break ;
         /*----------------------------------------------------------------*/
         case CFileInit_qtf::E_MDIA_TYPE_AUDIO :
            m_i_samprate = l_time_scale ;
            break ;
         /*----------------------------------------------------------------*/
         default: break ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::read_atom_trak( wxLongLong &ll_size_trak )
{
   /*----------------------------------------------------------------------*/
   str_qtf_atom atom            ;
   wxLongLong   ll_size         ;
   wxLongLong   ll_size_to_skip ;

   /*----------------------------------------------------------------------*/
   while(    ll_size_trak >= sizeof( str_atom )
          && read_atom( atom ) == 0
        )
   {
      /*-------------------------------------------------------------------*/
      ll_size         = atom.m_ll_data_size ;
      ll_size_to_skip = ll_size - atom.m_i_header_size ;
      /*-------------------------------------------------------------------*/
      switch( atom.m_dw_id )
      {
         /*--( Media atom )------------------------------------------------*/
         case SR_FOURCC( 'm','d','i','a' ) :
            read_atom_mdia( ll_size_to_skip ) ;
            break ;
         /*--( User defined data atom )------------------------------------*/
         case SR_FOURCC( 'u','d','t','a' ) :
            read_atom_udta( ll_size_to_skip ) ;
            break ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      if(    ll_size_to_skip > 0
          && skip_nb_byte( ll_size_to_skip.GetValue() ) != 0
        )
      {  return( -2 ) ; }
      /*-------------------------------------------------------------------*/
      ll_size_trak -= ll_size ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( For a file with compressed header this function will recurse )-------*/
int CFileInit_qtf::read_atom_moov( wxLongLong &ll_size_moov )
{
   /*----------------------------------------------------------------------*/
   str_qtf_atom atom            ;
   wxLongLong   ll_size         ;
   wxLongLong   ll_size_to_skip ;

   /*----------------------------------------------------------------------*/
   while(    ll_size_moov >= sizeof( str_atom )
          && read_atom( atom ) == 0
        )
   {
      /*-------------------------------------------------------------------*/
      ll_size         = atom.m_ll_data_size ;
      ll_size_to_skip = ll_size - atom.m_i_header_size ;
      /*-------------------------------------------------------------------*/
      switch( atom.m_dw_id )
      {
         /*--( Movie header: there is the "global" information )-----------*/
         case SR_FOURCC( 'm','v','h','d' ) :
         {
            /*--------------------------------------------------------------+
            ! The "time_scale" defined in the "mvhd" is not used to compute !
            ! the FPS because this time_scale is "global" not specific to   !
            ! the video.                                                    !
            +--------------------------------------------------------------*/
            str_qtf_mvhd mvhd ;
            /*-------------------------------------------------------------*/
            if( ll_size < sizeof( mvhd ) ) { break ; }
            /*-------------------------------------------------------------*/
            if( read_data( mvhd ) != 0 ) { return( -1 ) ; }
            /*-------------------------------------------------------------*/
            ll_size_to_skip -= sizeof( mvhd ) ;
            /*-------------------------------------------------------------*/
            if( mvhd.dw_time_scale != 0 )
            {  m_i_duration = mvhd.dw_duration / mvhd.dw_time_scale ; }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*--( Track list )------------------------------------------------*/
         case SR_FOURCC( 't','r','a','k' ) :
            read_atom_trak( ll_size_to_skip ) ;
            break ;
         /*--( User defined data atom )------------------------------------*/
         case SR_FOURCC( 'u','d','t','a' ) :
            read_atom_udta( ll_size_to_skip ) ;
            break ;

         /*-----------------------------------------------------------------+
         ! Movie resource compressed ?                                      !
         ! For now only zlib is supported                                   !
         +-----------------------------------------------------------------*/
         case SR_FOURCC( 'c','m','o','v' ) :
         {
            /*-------------------------------------------------------------*/
            wxUint32   dw_temp ;
            wxLongLong ll_decomp_size ;

            /*--( No compressed compressed data ... )----------------------*/
            if( m_boo_in_cmov )
            {  return( -2 ) ; }
            m_boo_in_cmov = true ;

            /*-------------------------------------------------------------*/
            if( read_atom( atom ) != 0 )
            {  return( -3 ) ; }
            ll_size_to_skip -= atom.m_i_header_size ;

            /*-------------------------------------------------------------*/
            if(    atom.m_dw_id != SR_FOURCC( 'd','c','o','m' )
                || atom.m_ll_data_size != 12
              )
            {  break ; }

            /*--( Compression method )-------------------------------------*/
            if( read_data_no_swap( dw_temp ) != 0 )
            {  return( -4 ) ; }
            ll_size_to_skip -= sizeof( dw_temp ) ;
            if( dw_temp != SR_FOURCC( 'z','l','i','b' ) )
            {  /*----------------------------------------------------------*/
               char *p_c_temp = ( char * )&dw_temp ;
               m_f.val_s( COL_VIDEO_INFO ).Printf(
                          "Unsupported header compression format (%c%c%c%c)",
                          p_c_temp[0], p_c_temp[1], p_c_temp[2], p_c_temp[3]
                                                 ) ;
               break ;
               /*----------------------------------------------------------*/
            }

            /*--( The atom of the compressed data )------------------------*/
            if( read_atom( atom ) != 0 )
            {  return( -5 ) ; }
            ll_size_to_skip -= atom.m_i_header_size ;
            if( atom.m_dw_id != SR_FOURCC( 'c','m','v','d' ) )
            {  break ; }

            /*--( The uncompressed size )----------------------------------*/
            if( read_data( dw_temp ) != 0 )
            {  return( -6 ) ; }
            ll_decomp_size = dw_temp ;
            ll_size_to_skip -= sizeof( dw_temp ) ;

            /*--( Create decompression elements )--------------------------*/
            wxZlibInputStream zl_input( *m_p_input ) ;
            /*--( Switch the streams )-------------------------------------*/
            wxInputStream *p_input_sav = m_p_input ;
            m_p_input = &zl_input ;

            /*--( The first atom should be a "moov" )----------------------*/
            if(    read_atom( atom ) == 0
                && atom.m_dw_id == SR_FOURCC( 'm','o','o','v' )
              )
            {
               /*----------------------------------------------------------*/
               ll_decomp_size -= atom.m_i_header_size ;
               /*--( Read the "uncompressed" data )------------------------*/
               read_atom_moov( ll_decomp_size ) ;
               /*----------------------------------------------------------*/
            }

            /*--( Back to "normal" )---------------------------------------*/
            m_p_input = p_input_sav ;
            m_boo_in_cmov = false   ;
            /*--( The complete atom has been read )------------------------*/
            ll_size_to_skip = 0 ;
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      if(    ll_size_to_skip > 0
          && skip_nb_byte( ll_size_to_skip.GetValue() ) != 0
        )
      {  return( -12 ) ; }
      /*-------------------------------------------------------------------*/
      ll_size_moov -= ll_size ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( m_l_sample_duration > 0 && m_l_time_scale > 0 )
   {  m_fi.init_video_fps( ( double )m_l_time_scale / m_l_sample_duration );}
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_qtf::qtf_read()
{
   /*----------------------------------------------------------------------*/
   str_qtf_atom atom            ;
   wxLongLong   ll_size_to_skip ;
   bool         boo_end = false ;

   /*-----------------------------------------------------------------------+
   ! Conversion to an input stream for base class compatibility with        !
   ! wxZlibInputStream (case of cmov atom)                                  !
   +-----------------------------------------------------------------------*/
   wxFileInputStream file_input( m_fa ) ;

   /*--( For now the file is read )----------------------------------------*/
   m_p_input = &file_input ;
   /*----------------------------------------------------------------------*/
   do
   {
      /*-------------------------------------------------------------------*/
      if( read_atom( atom ) != 0 )
      {  return( -1 ) ; }
      /*-------------------------------------------------------------------*/
      if( atom.m_ll_data_size == 0 || atom.m_dw_id == 0 )
      {  return( -2 ) ; }
      /*-------------------------------------------------------------------*/
      ll_size_to_skip = atom.m_ll_data_size - atom.m_i_header_size ;
      /*--( Movie resource )-----------------------------------------------*/
      if( atom.m_dw_id == SR_FOURCC( 'm','o','o','v' ) )
      {
         /*----------------------------------------------------------------*/
         read_atom_moov( ll_size_to_skip ) ;
         /*--( Don't need anything else )----------------------------------*/
         boo_end = true ;
         /*----------------------------------------------------------------*/
      }
      else /*--( Jump to the next atom )-----------------------------------*/
      if(    ll_size_to_skip > 0
          && m_fa.skip_nb_byte( ll_size_to_skip.GetValue() ) != 0
        )
      {  return( -3 ) ; }

      /*-------------------------------------------------------------------*/
   } while( !boo_end ) ;

   /*-----------------------------------------------------------------------+
   ! Store built info                                                       !
   +-----------------------------------------------------------------------*/
   if( m_i_duration >= 0 )
   {
      /*-------------------------------------------------------------------*/
      if( m_boo_has_video )
      {  m_fi.init_video_duration( m_i_duration ) ; }
      else
      {  m_fi.init_audio_duration( m_i_duration ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( m_i_nb_channel > 0 )
   {
      /*-------------------------------------------------------------------*/
      if( m_boo_has_video )
      {  m_fi.init_video_channel( m_i_nb_channel ) ; }
      else
      {  m_fi.init_audio_channel( m_i_nb_channel ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( m_i_samprate > 0 )
   {
      /*-------------------------------------------------------------------*/
      if( m_boo_has_video )
      {  m_fi.init_video_samprate( m_i_samprate ) ; }
      else
      {  m_fi.init_audio_samprate( m_i_samprate ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( !m_s_audio_info.empty() || !m_s_video_info.empty() )
   {
      /*-------------------------------------------------------------------*/
        m_f.val_s( m_boo_has_video ? COL_VIDEO_INFO : COL_AUDIO_INFO )
      = m_fi.concat_av_codecs( m_s_audio_info, m_s_video_info ) ;
      /*-------------------------------------------------------------------*/
   }

   /*--( Convert numbers to strings )--------------------------------------*/
   if( !m_boo_has_video ) { m_f.init_info_s_track() ; }

   /*----------------------------------------------------------------------*/
   m_fi.m_s_type_det = ( m_boo_has_video ? "mov" : "m4a" ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_qtf()
{
   /*----------------------------------------------------------------------*/
   m_s_type_det = "m4a/mov" ;
   /*----------------------------------------------------------------------*/
   return( CFileInit_qtf( *this ).qtf_read() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                       End of file CFileInit_qtf.cpp                       !
+==========================================================================*/
