/*====================================+=====================================+
! File CFileInit_iptc.cpp             ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                      IPTC Annotation of image files                       !
!                                                                           !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! "iptc" is not extension.                                                  !
! It's a metadata format used by other types: mainly "jpg" et "tif"         !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "CApp.h"
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
struct str_iptc_header_8BIM ;
struct str_iptc_header_tag  ;

/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_iptc : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_iptc( CFileInit &parent ) : CFileInit_type_base( parent )
      {  m_fi.set_conv_from_8bit() ; }
      /*-------------------------------------------------------------------*/
      int  read_be_data( str_iptc_header_8BIM &ih8 ) ;
      int  read_be_data( str_iptc_header_tag &iht ) ;
      void init_datetime() ;
      int  read_text_segment( size_t &sz_size_segment ) ;
      int  read_photoshop( size_t sz_size ) ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*-------------------------------------------------------------------------*/
struct str_iptc_header_8BIM
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_ident               ;
   wxUint8  tb_b_segment_type[ 2 ] ;
   wxUint8  b_hinfo              ; // 1st byte of size to detect segment type
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_iptc_header_tag
{
   /*----------------------------------------------------------------------*/
   wxUint8  tb_b_ident[ 2 ] ;
   wxUint8  b_type          ;
   wxUint16 w_size          ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*-------------------------------------------------------------------------*/
int CFileInit_iptc::read_be_data( str_iptc_header_8BIM &ih8 )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( ih8 ), &ih8 ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   ih8.dw_ident = wxUINT32_SWAP_ALWAYS( ih8.dw_ident ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_iptc::read_be_data( str_iptc_header_tag &iht )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( iht ), &iht ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   iht.w_size = wxUINT16_SWAP_ALWAYS( iht.w_size ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( "Simple" trick to convert two tags (date&time) to one col (datetime))*/
#define IPTC_DATE_TIME_SEP    " "

/*-------------------------------------------------------------------------*/
typedef char t_iptc_header_photoshop[ 14 ] ;

/*-------------------------------------------------------------------------*/
enum enu_iptc_tag
{
   /*----------------------------------------------------------------------*/
   TI_BYLINE           = 0x50 ,
   TI_BYLINE_TITLE     = 0x55 ,
   TI_COPYRIGHT        = 0x74 ,
   TI_CREDIT           = 0x6E ,
   TI_SOURCE           = 0x73 ,
   TI_CAPTION_WRITER   = 0x7A ,
   TI_CAPTION          = 0x78 ,
   TI_HEADLINE         = 0x69 ,
   TI_SPECIAL_INSTR    = 0x28 ,
   TI_OBJECT_NAME      = 0x05 ,
   TI_DATE_CREATED     = 0x37 ,
   TI_TIME_CREATED     = 0x3C ,
   TI_DATE_RELEASED    = 0x1E ,
   TI_TIME_RELEASED    = 0x23 ,
   TI_CITY             = 0x5A ,
   TI_SUBLOCATION      = 0x5C ,
   TI_STATE            = 0x5F ,
   TI_COUNTRY_CODE     = 0x64 ,
   TI_COUNTRY          = 0x65 ,
   TI_ORIGINAL_TRSF    = 0x67 ,
   TI_CATEGORY         = 0x0F ,
   TI_SUPPL_CATEGORIES = 0x14 ,
   TI_EDIT_STATUS      = 0x07 ,
   TI_JOBID            = 0x16 ,
   TI_PROGRAM          = 0x41 ,
   TI_PROGRAM_VERSION  = 0x46 ,
   TI_PRIORITY         = 0x0A ,
   TI_OBJECT_CYCLE     = 0x4B ,
   TI_KEYWORDS         = 0x19
   /*----------------------------------------------------------------------*/
} ;

/*--------------------------------------------------------------------------+
! It seems that the date/time format is: YYYYMMDD_HHMMSSxHHMM               !
! But as it is not clear for now the time diff is not taken into account    !
+---------------------------------------------------------------------------+
! It seems that the syntax should be the same as for pdf: UTC +- time diff  !
! But other the softs handling IPTC don't seem to take this into account.   !
+--------------------------------------------------------------------------*/
static int st_iptc_conv_datetime_string( const wxString &s ,
                                         wxDateTime     &dt
                                       )
{
   /*--( Only the year is mandatory )--------------------------------------*/
   if( s.size() < 4 ) { return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   return( sr::init_date_ymdhms(
                            s, "%4d%2d%2d" IPTC_DATE_TIME_SEP "%2d%2d%2d", dt
                               )
         ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static int st_jpg_conv_iptc_tag_col( int i_tag )
{
   /*----------------------------------------------------------------------*/
   switch( i_tag )
   {
      /*-------------------------------------------------------------------*/
      case TI_BYLINE           : return( COL_IPTC_BYLINE            ) ;
      case TI_BYLINE_TITLE     : return( COL_IPTC_BYLINE_TITLE      ) ;
      case TI_COPYRIGHT        : return( COL_IPTC_COPYRIGHT         ) ;
      case TI_CREDIT           : return( COL_IPTC_CREDIT            ) ;
      case TI_SOURCE           : return( COL_IPTC_SOURCE            ) ;
      case TI_CAPTION_WRITER   : return( COL_IPTC_CAPTION_WRITER    ) ;
      case TI_CAPTION          : return( COL_IPTC_CAPTION           ) ;
      case TI_HEADLINE         : return( COL_IPTC_HEADLINE          ) ;
      case TI_SPECIAL_INSTR    : return( COL_IPTC_SPECIAL_INSTR     ) ;
      case TI_OBJECT_NAME      : return( COL_IPTC_OBJECT_NAME       ) ;
      case TI_DATE_CREATED     : return( COL_IPTC_DATETIME_CREATED  ) ;
      case TI_TIME_CREATED     : return( COL_IPTC_DATETIME_CREATED  ) ;
      case TI_DATE_RELEASED    : return( COL_IPTC_DATETIME_RELEASED ) ;
      case TI_TIME_RELEASED    : return( COL_IPTC_DATETIME_RELEASED ) ;
      case TI_CITY             : return( COL_IPTC_CITY              ) ;
      case TI_SUBLOCATION      : return( COL_IPTC_SUBLOCATION       ) ;
      case TI_STATE            : return( COL_IPTC_STATE             ) ;
      case TI_COUNTRY_CODE     : return( COL_IPTC_COUNTRY_CODE      ) ;
      case TI_COUNTRY          : return( COL_IPTC_COUNTRY           ) ;
      case TI_ORIGINAL_TRSF    : return( COL_IPTC_ORIGINAL_TRSF     ) ;
      case TI_CATEGORY         : return( COL_IPTC_CATEGORY          ) ;
      case TI_SUPPL_CATEGORIES : return( COL_IPTC_SUPPL_CATEGORIES  ) ;
      case TI_EDIT_STATUS      : return( COL_IPTC_EDIT_STATUS       ) ;
      case TI_JOBID            : return( COL_IPTC_JOBID             ) ;
      case TI_PROGRAM          : return( COL_IPTC_PROGRAM           ) ;
      case TI_PROGRAM_VERSION  : return( COL_IPTC_PROGRAM_VERSION   ) ;
      case TI_PRIORITY         : return( COL_IPTC_PRIORITY          ) ;
      case TI_OBJECT_CYCLE     : return( COL_IPTC_OBJECT_CYCLE      ) ;
      case TI_KEYWORDS         : return( COL_IPTC_KEYWORDS          ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( COL_NB ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! As the date and time are stored in two different tags, the date/time      !
! value is extracted once the initialization is "complete".                 !
+--------------------------------------------------------------------------*/
void CFileInit_iptc::init_datetime()
{
   /*----------------------------------------------------------------------*/
   wxDateTime        dt ;
   CFile::t_s_val_it it ;

   /*----------------------------------------------------------------------*/
   if(    ( it = m_f.find_map_s_val( COL_IPTC_DATETIME_CREATED )
          ) != m_f.get_map_s_val().end()
       && st_iptc_conv_datetime_string( it->second, dt ) == 0
     )
   {  m_fi.init_date( COL_IPTC_DATETIME_CREATED, dt ) ; }

   /*----------------------------------------------------------------------*/
   if(    ( it = m_f.find_map_s_val( COL_IPTC_DATETIME_RELEASED )
          ) != m_f.get_map_s_val().end()
       && st_iptc_conv_datetime_string( it->second, dt ) == 0
     )
   {  m_fi.init_date( COL_IPTC_DATETIME_RELEASED, dt ) ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_iptc::read_text_segment( size_t &sz_size_segment )
{
   /*----------------------------------------------------------------------*/
   str_iptc_header_tag ht    ;
   wxString            s_val ;
   int                 i_col ;

   /*----------------------------------------------------------------------*/
   while( sz_size_segment > sizeof( str_iptc_header_tag ) )
   {
      /*--( Header )-------------------------------------------------------*/
      if( read_be_data( ht ) != 0 ) { return( -1 ) ; }

      /*-------------------------------------------------------------------*/
      sz_size_segment -= sizeof( ht ) ;

      /*--( Column KO if the code is unknown )-----------------------------*/
      if( ht.tb_b_ident[ 0 ] != 0x1C || ht.tb_b_ident[ 1 ] != 0x02 )
      {  i_col = COL_NB ; }
      else
      {  i_col = st_jpg_conv_iptc_tag_col( ht.b_type ) ; }

      /*--------------------------------------------------------------------+
      ! Reserve the column and allow multiple values for some of them       !
      +--------------------------------------------------------------------*/
      if(    i_col == COL_IPTC_SUPPL_CATEGORIES
          || i_col == COL_IPTC_KEYWORDS
        )
      {
         /*----------------------------------------------------------------*/
         if( m_fi.file_read_tb_c( ht.w_size, s_val ) != 0 )
         {  return( -2 ) ; }
         /*----------------------------------------------------------------*/
         if( !s_val.empty() )
         {  sr::append_with_sep( s_val, ", ", m_f.val_s( i_col ) ) ; }
         /*----------------------------------------------------------------*/
      }
      else /*--( From two separated only one is built )--------------------*/
      if(    i_col == COL_IPTC_DATETIME_CREATED
          || i_col == COL_IPTC_DATETIME_RELEASED
        )
      {
         /*----------------------------------------------------------------*/
         if( m_fi.file_read_tb_c( ht.w_size, s_val ) != 0 )
         {  return( -3 ) ; }

         /*-----------------------------------------------------------------+
         ! Directly point to the right string to avoid multiple map access  !
         +-----------------------------------------------------------------*/
         wxString *p_sic = &m_f.val_s( i_col ) ;

         /*--( And add the separator if a value was already present )------*/
         if( p_sic->empty() )
         {  /*-------------------------------------------------------------*/
            if(    ht.b_type == TI_DATE_CREATED
                || ht.b_type == TI_DATE_RELEASED
              )
            {  *p_sic = s_val + IPTC_DATE_TIME_SEP ; }
            else
            {  *p_sic = IPTC_DATE_TIME_SEP + s_val ; }
            /*-------------------------------------------------------------*/
         }
         else
         {  /*-------------------------------------------------------------*/
            int i_pos_sep = p_sic->find( IPTC_DATE_TIME_SEP ) ;
            /*--( Not twice the same tag ... )-----------------------------*/
            if(    i_pos_sep == 0
                && (    ht.b_type == TI_DATE_CREATED
                     || ht.b_type == TI_DATE_RELEASED
                   )
              )
            {  p_sic->insert( 0, s_val ) ; }
            else
            if(    i_pos_sep != 0
                && (    ht.b_type == TI_TIME_CREATED
                     || ht.b_type == TI_TIME_RELEASED
                   )
              )
            {  p_sic->append( s_val ) ; }
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      else
      if( m_fi.reserve_col( i_col ) )
      {
         /*----------------------------------------------------------------*/
         if( m_fi.file_read_tb_c( ht.w_size, s_val ) != 0 )
         {  return( -4 ) ; }
         /*----------------------------------------------------------------*/
         if( !s_val.empty() )
         {  m_f.val_s( i_col ) = s_val ; }
         /*----------------------------------------------------------------*/
      }
      else
      {  /*----------------------------------------------------------------*/
         if( m_fa.skip_nb_byte( ht.w_size ) != 0 ) { return( -5 ) ; }
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      sz_size_segment -= ht.w_size ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_iptc::read_photoshop( size_t sz_size )
{
   /*----------------------------------------------------------------------*/
   t_iptc_header_photoshop header_phot                    ;
   str_iptc_header_8BIM    header_8BIM                    ;
   wxUint16                w_marker_size                  ;
   size_t                  sz_size_segment                ;
   size_t                  sz_size_to_skip                ;
   bool                    boo_text_segment_found = false ;

   /*--( Photoshop header )------------------------------------------------*/
   if(    sz_size <= ( int )sizeof( header_phot )
       || m_fa.read_buffer( sizeof( header_phot ), &header_phot ) != 0
          /*--( The final zero ('\0') is taken into account )--------------*/
       || memcmp( header_phot, "Photoshop 3.0", sizeof( header_phot ) ) != 0
     )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   sz_size -= sizeof( header_phot ) ;
   /*----------------------------------------------------------------------*/
   while(    sz_size > sizeof( str_iptc_header_8BIM )
          && !boo_text_segment_found
        )
   {
      /*-------------------------------------------------------------------*/
      do
      {
         /*--( 8BIM header )-----------------------------------------------*/
         if( read_be_data( header_8BIM ) != 0 )
         {  return( -2 ) ; }
         /*--( Problem ... of pad eventually )-----------------------------*/
         if( *( char * )&header_8BIM.dw_ident == '\0' )
         {  /*-------------------------------------------------------------*/
            if( --sz_size <= sizeof( str_iptc_header_8BIM ) )
            {  return( -3 ) ; }
            /*--( Go back ... )--------------------------------------------*/
            if( m_fa.skip_nb_byte( - ( int )sizeof( header_8BIM ) + 1 ) != 0)
            {  return( -4 ) ; }
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      } while( *( char * )&header_8BIM.dw_ident == '\0' ) ;

      /*-------------------------------------------------------------------*/
      if(    header_8BIM.dw_ident
          != wxUINT32_SWAP_ON_LE( SR_FOURCC( '8','B','I','M' ) )
        )
      {  return( -5 ) ; }

      /*--------------------------------------------------------------------+
      ! Two possible formats:                                               !
      ! - PS6  => size + string (of size len) + 2 char of '0' padding       !
      ! - else => 4 0x00                                                    !
      +--------------------------------------------------------------------*/
        sz_size_to_skip
      = ( header_8BIM.b_hinfo == 0 ? 3 : header_8BIM.b_hinfo + 2 ) ;

      /*-------------------------------------------------------------------*/
      if( m_fa.skip_nb_byte( sz_size_to_skip ) != 0 )
      {  return( -6 ) ; }

      /*--( The size of the segment )--------------------------------------*/
      if( m_fa.read_be_data( w_marker_size ) != 0 )
      {  return( -7 ) ; }
      /*-------------------------------------------------------------------*/
      sz_size_segment = w_marker_size ;
      if( sz_size_segment == 0 ) { return( -8 ) ; }

      /*-------------------------------------------------------------------*/
         sz_size
      -=   sizeof( header_8BIM )
         + sizeof( w_marker_size ) + sz_size_to_skip + sz_size_segment ;

      /*--( "Text" segment ? )---------------------------------------------*/
      if(    header_8BIM.tb_b_segment_type[ 0 ] == 0x04
          && header_8BIM.tb_b_segment_type[ 1 ] == 0x04
        )
      {  /*--( What has been read is subtracted from the size )------------*/
         if( read_text_segment( sz_size_segment ) != 0 )
         {  return( -9 ) ; }
         /*----------------------------------------------------------------*/
         boo_text_segment_found = true ;
         /*----------------------------------------------------------------*/
      }

      /*--( Skip the rest )------------------------------------------------*/
      if( sz_size_segment > 0 && m_fa.skip_nb_byte( sz_size_segment ) != 0 )
      {  return( -10 ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::read_iptc( wxFileOffset fo_offset, size_t sz_size,
                          bool boo_photoshop
                        )
{
   /*----------------------------------------------------------------------*/
   CFileInit_iptc iptc( *this ) ;

   /*----------------------------------------------------------------------*/
   if( m_fa.set_offset( fo_offset ) != 0 ) { return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   int i_ret ;
   /*----------------------------------------------------------------------*/
   if( boo_photoshop )
   {  i_ret = iptc.read_photoshop( sz_size ) ; }
   else
   {  i_ret = iptc.read_text_segment( sz_size ) ; }

   /*--( Initialisation of the date/time fetched separately )--------------*/
   iptc.init_datetime() ;
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                       End of file CFileInit_iptc.cpp                      !
+==========================================================================*/
