/*====================================+=====================================+
! File CFileInit_flac.cpp             ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                       Audio file: ".flac"                                 !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
struct str_flac_stream_info ;

/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_flac : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_flac( CFileInit &parent ) : CFileInit_type_base( parent )
      {  ; }
      /*-------------------------------------------------------------------*/
      int read_be_data( str_flac_stream_info &fsi ) ;
      int read_picture_info( int &i_nb_byte_read ) ;
      int flac_read() ;

   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*-------------------------------------------------------------------------*/
struct str_flac_stream_info
{
   /*----------------------------------------------------------------------*/
   wxUint16 ___w_min_block_size         ;
   wxUint16 ___w_max_block_size         ;
   // cppcheck-suppress unusedStructMember
   char     ___tb_c_min_frame_size[ 3 ] ;
   // cppcheck-suppress unusedStructMember
   char     ___tb_c_max_frame_size[ 3 ] ;
   /*-----------------------------------------------------------------------+
   ! ddw_info is structured like this:                                      !
   ! - 20 bits : sampling rate                                              !
   ! -  3 bits : number of channels                                         !
   ! -  5 bits : bits per sample                                            !
   ! - 36 bits : number of samples                                          !
   +-----------------------------------------------------------------------*/
   wxUint64 ddw_info ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*-------------------------------------------------------------------------*/
int CFileInit_flac::read_be_data( str_flac_stream_info &fsi )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( fsi ), &fsi ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   fsi.ddw_info = wxUINT64_SWAP_ALWAYS( fsi.ddw_info ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_flac::read_picture_info( int &i_nb_byte_read )
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_length      ;
   wxUint32 dw_mime_length ;
   wxUint32 dw_desc_length ;
   wxString s_mime         ;

   /*----------------------------------------------------------------------*/
   m_fi.set_conv_from_8bit() ;

   /*--( First, skip the picture type )------------------------------------*/
   if( m_fa.skip_nb_byte( sizeof( dw_length ) ) != 0 )
   {  return( -1 ) ; }

   /*--( The length of the mime type )-------------------------------------*/
   if( m_fa.read_be_data( dw_mime_length ) != 0 )
   {  return( -2 ) ; }
   /*--( The mime type )---------------------------------------------------*/
   if( m_fi.file_read_tb_c( dw_mime_length, s_mime ) != 0 || s_mime.empty() )
   {  return( -3 ) ; }
   /*--( Store it )--------------------------------------------------------*/
   if( m_fi.prepare_string( s_mime ) > 0 )
   {  m_f.val_s( COL_AUDTAG_IMG_FORMAT ) = s_mime ; }

   /*--( The length of the description of the picture )--------------------*/
   if( m_fa.read_be_data( dw_desc_length ) != 0 )
   {  return( -4 ) ; }
   /*--( The description and some other info are skipped )-----------------*/
   if( m_fa.skip_nb_byte( dw_desc_length + 4 * sizeof( dw_length ) ) != 0 )
   {  return( -5 ) ; }

   /*--( Then the length of the picture data )-----------------------------*/
   if( m_fa.read_be_data( dw_length ) != 0 )
   {  return( -6 ) ; }
   /*--( Then the picture itself: the offset will be "kept" )--------------*/
   if( m_fi.read_embedded_image_info( m_fa.get_offset(), dw_length, s_mime
                                    ) != 0
     )
   {  return( -7 ) ; }

   /*--( The number of bytes read or skipped )-----------------------------*/
   i_nb_byte_read =   sizeof( dw_length )
                    + sizeof( dw_length ) + dw_mime_length
                    + sizeof( dw_length ) + dw_desc_length
                    + sizeof( dw_length ) * 4
                    + sizeof( dw_length ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_flac::flac_read()
{
   /*----------------------------------------------------------------------*/
   wxUint32             dw_header                    ;
   wxUint32             dw_block_header              ;
   str_flac_stream_info stream_info                  ;
   bool                 boo_stream_info_done = false ;
   bool                 boo_comment_done     = false ;
   bool                 boo_picture_done     = false ;
   bool                 boo_last_header_block        ;
   int                  i_block_type                 ;
   wxFileOffset         fo_block_size                ;
   int                  i_samprate                   ;
   wxUint64             ddw_nb_sample                ;

   /*--( Header )----------------------------------------------------------*/
   if(    m_fa.read_le_data( dw_header ) != 0
       || dw_header != SR_FOURCC( 'f','L','a','C' )
     )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   do
   {
      /*--( Metadata block header )----------------------------------------*/
      if( m_fa.read_be_data( dw_block_header ) != 0 )
      {  return( -2 ) ; }
      /*-------------------------------------------------------------------*/
      boo_last_header_block = ( ( dw_block_header & 0x80000000 ) != 0 ) ;
      i_block_type = ( ( dw_block_header >> 24 ) & 0x07 ) ;
      fo_block_size = ( dw_block_header & 0x00FFFFFF ) ;

      /*-------------------------------------------------------------------*/
      switch( i_block_type )
      {
         /*-----------------------------------------------------------------+
         ! STREAMINFO                                                       !
         +-----------------------------------------------------------------*/
         case 0 :
            /*-------------------------------------------------------------*/
            if( boo_stream_info_done ) { break ; }
            boo_stream_info_done = true ;
            /*-------------------------------------------------------------*/
            if( read_be_data( stream_info ) != 0 )
            {  return( -3 ) ; }
            fo_block_size -= sizeof( stream_info ) ;
            /*-------------------------------------------------------------*/
            i_samprate = ( ( stream_info.ddw_info >> 44 ) & 0xFFFFF ) ;
            if( i_samprate <= 0 ) { return( -4 ) ; }
            /*-------------------------------------------------------------*/
              ddw_nb_sample
            = (   stream_info.ddw_info
                & ( ( ( wxUint64 )0x0F << 32 ) | 0xFFFFFFFF )
              ) ;
            /*-------------------------------------------------------------*/
            m_fi.init_audio_channel( ( ( stream_info.ddw_info >> 41 ) & 0x7 )
                                     + 1
                                   ) ;
            m_fi.init_audio_samprate( i_samprate ) ;
            m_fi.init_audio_duration( ddw_nb_sample / i_samprate ) ;
            /*-------------------------------------------------------------*/
            break ;

         /*-----------------------------------------------------------------+
         ! VORBIS_COMMENT                                                   !
         +-----------------------------------------------------------------*/
         case 4 :
         {
            /*-------------------------------------------------------------*/
            if( boo_comment_done ) { break ; }
            boo_comment_done = true ;
            /*-------------------------------------------------------------*/
            wxFileOffset fo_sav = m_fa.get_offset() ;
            /*--( It's an audio comment )----------------------------------*/
            if( m_fi.ogg_read_vorbis_comment( false ) != 0 )
            {  return( -4 ) ; }
            /*-------------------------------------------------------------*/
            fo_block_size -= m_fa.get_offset() - fo_sav ;
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }

         /*-----------------------------------------------------------------+
         ! PICTURE                                                          !
         +-----------------------------------------------------------------*/
         case 6 :
         {
            /*-------------------------------------------------------------*/
            int i_read_length ;
            /*-------------------------------------------------------------*/
            if( boo_picture_done ) { break ; }
            boo_picture_done = true ;
            /*-------------------------------------------------------------*/
            if( read_picture_info( i_read_length ) != 0 )
            {  return( -5 ) ; }
            /*-------------------------------------------------------------*/
            fo_block_size -= i_read_length ;
            /*-------------------------------------------------------------*/
            break ;
         }
         /*----------------------------------------------------------------*/
      }

      /*--( Jump over the rest of the block )------------------------------*/
      if( m_fa.skip_nb_byte( fo_block_size ) != 0 )
      {  return( -6 ) ; }
      /*-------------------------------------------------------------------*/

   } while(    !boo_last_header_block
            && (    !boo_stream_info_done
                 || !boo_comment_done
                 || !boo_picture_done
               )
          ) ;

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_flac()
{
   /*----------------------------------------------------------------------*/
   m_s_type_det = "flac" ;
   /*----------------------------------------------------------------------*/
   return( CFileInit_flac( *this ).flac_read() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                      End of file CFileInit_flac.cpp                       !
+==========================================================================*/
