/*====================================+=====================================+
! File CFileInit_crx.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!               Chrome extension installation file: ".crx"                  !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <memory>
#include <wx/wfstream.h>
#include <wx/zipstrm.h>
#include <wx/sstream.h>
#include <wx/regex.h>
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
struct str_crx_header ;

/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_crx : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_crx( CFileInit &parent ) : CFileInit_type_base( parent )
      {  ; }
      /*-------------------------------------------------------------------*/
      int read_le_data( str_crx_header &crx ) ;
      int tag_regex( const wxString &s_buffer,
                     const wxString &s_var, int i_col
                   ) ;
      int crx_read() ;

   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*-------------------------------------------------------------------------*/
struct str_crx_header
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_magic            ; // Cr24
   wxUint32 dw_version          ; // Version crx file format (currently 2)
   wxUint32 dw_key_length       ; // Length of the RSA public key in bytes
   wxUint32 dw_signature_length ; // The length of the signature in bytes.
   /*----------------------------------------------------------------------*/
/*
   Then comes:
   dw_key_length bytes of the public key
   dw_signature_length bytes of the signature
*/
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*-------------------------------------------------------------------------*/
int CFileInit_crx::read_le_data( str_crx_header &crx )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( crx ), &crx ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxLITTLE_ENDIAN
   crx.dw_version = wxUINT32_SWAP_ALWAYS( crx.dw_version ) ;
   crx.dw_key_length = wxUINT32_SWAP_ALWAYS( crx.dw_key_length ) ;
   crx.dw_signature_length = wxUINT32_SWAP_ALWAYS( crx.dw_signature_length );
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_crx::tag_regex( const wxString &s_buffer,
                              const wxString &s_var, int i_col
                            )
{
   /*--( Regular expression adapted to the name )--------------------------*/
   const wxString s_regex( "\"" + s_var + "\"\\s*?:\\s*?\"(.+?)\"" ) ;
   wxRegEx regex( s_regex, wxRE_ADVANCED ) ;

   /*----------------------------------------------------------------------*/
   if( !regex.IsValid() )
   {  wxLogError( "Pb exp crx tag_regex #%s#", s_regex ) ;
      return( -1 ) ;
   }

   /*----------------------------------------------------------------------*/
   if( regex.Matches( s_buffer ) && m_fi.reserve_col( i_col ) )
   {
      /*-------------------------------------------------------------------*/
      wxString s_val = regex.GetMatch( s_buffer, 1 ) ;
      /*-------------------------------------------------------------------*/
      if( m_fi.prepare_string( s_val ) > 0 )
      {
         /*--( Add some text for the "minimum chrome version" )------------*/
         if( i_col == COL_DOC_APPLICATION )
         {  s_val.insert( 0, "Chrome: " ) ; }
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = s_val ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_crx::crx_read()
{
   /*----------------------------------------------------------------------*/
   str_crx_header header ;

   /*--( Skip the header placed before the Zip files data )----------------*/
   if( read_le_data( header ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   if( header.dw_magic != SR_FOURCC( 'C','r','2','4' ) )
   {  return( -2 ) ; }
   /*--( Skip key and signature )------------------------------------------*/
   if( m_fa.skip_nb_byte( header.dw_key_length + header.dw_signature_length
                        ) != 0
     )
   {  return( -3 ) ; }

   /*--( The zip data should begin here )----------------------------------*/
   wxFileInputStream           finput( m_fa )      ;
   wxZipInputStream            zip_input( finput ) ;
   std::auto_ptr< wxZipEntry > ptr_zip_entry       ;

   /*----------------------------------------------------------------------*/
   if( !zip_input.IsOk() ) { return( -4 ) ; }
   /*----------------------------------------------------------------------*/
   m_fi.init_doc_zip_info( zip_input ) ;

   /*--( Looking for the file containing the metadata )--------------------*/
   do
   {  ptr_zip_entry.reset( zip_input.GetNextEntry() ) ;
   } while(    ptr_zip_entry.get() != NULL
            && ptr_zip_entry->GetInternalName().CmpNoCase( "manifest.json"
                                                         ) != 0
          ) ;
   /*----------------------------------------------------------------------*/
   if( ptr_zip_entry.get() == NULL ) { return( -5 ) ; }
   /*--( The data is supposed to be not very large )-----------------------*/
   if( ptr_zip_entry->GetSize() > 5 * 1024 ) { return( -6 ) ; }

   /*----------------------------------------------------------------------*/
   wxStringOutputStream s_os_buffer ;

   /*--( Full data read into the associated string )-----------------------*/
   zip_input.Read( s_os_buffer ) ;

   /*----------------------------------------------------------------------*/
   tag_regex( s_os_buffer.GetString(), "name", COL_DOC_TITLE ) ;
   tag_regex( s_os_buffer.GetString(), "description", COL_DOC_INFO ) ;
   tag_regex( s_os_buffer.GetString(), "version", COL_DOC_VERSION ) ;
   tag_regex( s_os_buffer.GetString(), "minimum_chrome_version",
              COL_DOC_APPLICATION
            ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_crx()
{
   /*----------------------------------------------------------------------*/
   m_s_type_det = "crx" ;
   /*----------------------------------------------------------------------*/
   return( CFileInit_crx( *this ).crx_read() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                      End of file CFileInit_crx.cpp                        !
+==========================================================================*/
