/*====================================+=====================================+
! File CFileInit_crw.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!       CRW: Canon RAW image format. It uses a specific format: CIFF        !
!                                                                           !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! Even if they are not really EXIF, the metadata are stored in this group   !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
struct str_ciff_entry ;

/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_crw : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_crw( CFileInit &parent ) : CFileInit_type_base( parent ),
                                           m_p_fnc_sw( NULL ), //for cppcheck
                                           m_p_fnc_uw( NULL ), //for cppcheck
                                           m_p_fnc_sdw( NULL ),//for cppcheck
                                           m_p_fnc_udw( NULL ) //for cppcheck
      {  ; }
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      wxFileOffset m_fo_offset_first_heap ;
      /*-------------------------------------------------------------------*/
      wxInt16  ( *m_p_fnc_sw  )( wxInt16  sw  ) ;
      wxUint16 ( *m_p_fnc_uw  )( wxUint16 uw  ) ;
      wxInt32  ( *m_p_fnc_sdw )( wxInt32  sdw ) ;
      wxUint32 ( *m_p_fnc_udw )( wxUint32 udw ) ;
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      int read_tag( wxFileOffset fo_offset,
                    const str_ciff_entry &ciff_entry
                  ) ;
      int read_tab_entry( wxFileOffset fo_offset, long l_size ) ;
      int ciff_init() ;
      int ciff_read() ;
   /*----------------------------------------------------------------------*/
} ;



/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*--------------------------------------------------------------------------+
! As the file can be BE or LE, the "read_??_data" method are not used.      !
! The conversion is directly done here.                                     !
+--------------------------------------------------------------------------*/
struct str_ciff_header
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_align      ;
   wxUint32 dw_offset    ;
   char     tb_c_id[ 8 ] ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_ciff_offset
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_offset ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_ciff_heap_nb_entry
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_nb ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_ciff_entry
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_type        ;
   wxUint32 dw_size       ;
   wxUint32 dw_offset_rel ;
   /*----------------------------------------------------------------------*/
} ;

/*--( For now this one is not used )---------------------------------------*/
#if SR_NOT_USED_FOR_NOW
struct str_ciff_canon_sensor
{
   /*----------------------------------------------------------------------*/
   // cppcheck-suppress unusedStructMember
   char     ___tb_c_dummy_1[ 2 ] ;
   wxUint16 w_width              ;
   wxUint16 w_height             ;
   // cppcheck-suppress unusedStructMember
   char     ___tb_c_dummy_2[ 4 ] ;
   wxUint16 w_x_offset           ;
   wxUint16 w_y_offset           ;
   wxUint16 w_x_out_base         ;
   wxUint16 w_y_out_base         ;
   /*----------------------------------------------------------------------*/
} ;
#endif

/*-------------------------------------------------------------------------*/
struct str_ciff_image_spec
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_imagewidth          ;
   wxUint32 dw_imageheight         ;
   wxUint32 dw_pixel_aspect_ratio  ;
   wxUint32 dw_rotation_angle      ;
   wxUint32 dw_component_bit_depth ;
   wxUint32 dw_color_bit_depth     ;
   wxUint32 dw_color_bw            ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_ciff_shot_info
{
   /*----------------------------------------------------------------------*/
   // cppcheck-suppress unusedStructMember
   char     ___tb_c_dummy_1[ 4 ] ;
   wxUint16 w_iso                ;
   wxUint16 w_exposure_accuracy  ;
   wxUint16 w_av                 ;
   wxUint16 w_tv                 ;
   wxUint16 w_exposure_comp      ;
   wxUint16 w_white_balance      ;

/* Difficult to have reliable info and comparison "points" for some info.
   In particular the "FNumber" doesn't seem to be what can be found in a
   few documentations. It seems more to be the "av actual" ... ???? ....
   For now (waiting for better ) av is copied into fnumber.

   BYTE tb_b_dummy_2          [  2 ] ;
   BYTE tb_b_seq_number       [  2 ] ;
   BYTE tb_b_dummy_3          [  8 ] ;
   BYTE tb_b_ixus_afp_point   [  2 ] ;
   BYTE tb_b_flash_exp_comp   [  2 ] ;
   BYTE tb_b_auto_exp_bracket [  2 ] ;
   BYTE tb_b_aeb_bracket      [  2 ] ;
   BYTE tb_b_dummy_4          [  2 ] ;
   BYTE tb_b_focus_dist_upper [  2 ] ;
   BYTE tb_b_focus_dist_lower [  2 ] ;
   BYTE tb_b_fnumber          [  2 ] ;

   There is more ...
*/
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_ciff_camera_settings
{
   /*----------------------------------------------------------------------*/
   // cppcheck-suppress unusedStructMember
   char     ___tb_c_dummy_1[  2 ] ;
   wxUint16 w_macro_mode          ;
   wxUint16 w_self_timer_period   ;
   wxUint16 w_quality             ;
   wxUint16 w_flash_mode          ;
   wxUint16 w_continuous_drive    ;
   // cppcheck-suppress unusedStructMember
   char     ___tb_c_dummy_2[  2 ] ;
   wxUint16 w_focus_mode          ;
   // cppcheck-suppress unusedStructMember
   char     ___tb_c_dummy_3[  4 ] ;
   wxUint16 w_image_size_type     ;
   wxUint16 w_easy_mode           ;
   wxUint16 w_digital_zoom        ;
   wxUint16 w_contrast            ; // Correspondence not found
   wxUint16 w_saturation          ; // Correspondence not found
   wxUint16 w_sharpness           ; // Correspondence not found
   wxUint16 w_camera_iso          ;
   wxUint16 w_metering_mode       ;
   wxUint16 w_focus_type          ;
   wxUint16 w_afp_point           ;
   wxUint16 w_exposure_mode       ;

/* There is more ...
*/
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_ciff_focal_length
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_type_scale ;
   wxUint16 w_focal_mm   ;
   wxUint16 w_sensor_x   ; // in 1/1000 inches
   wxUint16 w_sensor_y   ; // in 1/1000 inches
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_ciff_colorspace
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_colorspace ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*-------------------------------------------------------------------------*/
enum enu_ciff_tag
{
   /*----------------------------------------------------------------------*/
   TC_ID_DESCRIPTION         = 0x0805,
   TC_ID_MAKE_AND_MODEL      = 0x080a,
   TC_ID_FIRMWARE            = 0x080b,
   TC_ID_COMPONENT           = 0x080c,
   TC_ID_OWNER_NAME          = 0x0810,
   TC_ID_IMAGE_TYPE          = 0x0815,
   TC_ID_ORIGINAL_FILE_NAME  = 0x0816,
   TC_ID_THUMBNAIL_FILE_NAME = 0x0817,
   /*----------------------------------------------------------------------*/
   TC_ID_IMAGE_SPEC          = 0x1810,
   /*---------------------------------------------------------------------*/
   TC_ID_TIMESTAMP           = 0x180e,
   /*----------------------------------------------------------------------*/
   TC_ID_SHOT_INFO           = 0x102a,
   TC_ID_CAMERA_SETTINGS     = 0x102d,
   /*----------------------------------------------------------------------*/
   TC_ID_FOCAL_LENGTH        = 0x5029,
   TC_ID_COLORSPACE          = 0x10b4,
   /*----------------------------------------------------------------------*/
   TC_ID_JPEG_FROM_RAW       = 0x2007,
   TC_ID_THUMBNAIL           = 0x2008
   /*----------------------------------------------------------------------*/
} ;

/*--------------------------------------------------------------------------+
! Case not complete because there is no 1-1 link between ciff elmt and cols !
+--------------------------------------------------------------------------*/
static int st_ciff_conv_ciff_tag_col( int i_tag )
{
   /*----------------------------------------------------------------------*/
   switch( i_tag )
   {
      /*-------------------------------------------------------------------*/
      case TC_ID_DESCRIPTION : return( COL_EXIF_IMAGEDESCRIPTION       ) ;
      case TC_ID_FIRMWARE    : return( COL_EXIF_SOFTWARE               ) ;
      case TC_ID_COMPONENT   : return( COL_EXIF_COMPONENTCONFIGURATION ) ;
      case TC_ID_OWNER_NAME  : return( COL_EXIF_ARTIST                 ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( COL_NB ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static double st_ciff_canon_ev( short sh_val )
{
   /*----------------------------------------------------------------------*/
   int   i_signe ;
   short sh_frac ;
   /*----------------------------------------------------------------------*/
   if( sh_val >= 0 )
   {  i_signe = 1 ; }
   else
   {  i_signe = -1 ; sh_val = - sh_val ; }
   /*----------------------------------------------------------------------*/
   sh_frac  = sh_val & 0x1f ;
   sh_val  -= sh_frac       ;
   /*--( Convert 1/3 and 2/3 codes )---------------------------------------*/
   switch( sh_frac )
   {
      /*-------------------------------------------------------------------*/
      case 0x0c : sh_frac = 0x20 / 3 ; break ;
      case 0x14 : sh_frac = 0x40 / 3 ; break ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( ( double )i_signe * ( sh_val + sh_frac ) / 0x20 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_crw::read_tag( wxFileOffset fo_offset,
                             const str_ciff_entry &ciff_entry
                           )
{
   /*----------------------------------------------------------------------*/
   wxString     s_val                                ;
   int          i_tag                                ;
   long         l_size                               ;
   wxFileOffset fo_offset_abs                        ;
   wxFileOffset fo_offset_rel                        ;
   char         tb_c_val[ g_co_i_string_sz_max * 2 ] ;
   const char   *p_c_tmp                             ;

   /*----------------------------------------------------------------------*/
   i_tag         = m_p_fnc_uw( ciff_entry.w_type ) ;
   /*----------------------------------------------------------------------*/
   l_size        = m_p_fnc_udw( ciff_entry.dw_size ) ;
   fo_offset_rel = m_p_fnc_udw( ciff_entry.dw_offset_rel ) ;
   /*----------------------------------------------------------------------*/
   fo_offset_abs = fo_offset + fo_offset_rel ;

   /*--( Sub-table ? )-----------------------------------------------------*/
   switch( i_tag >> 8 )
   {
      /*-------------------------------------------------------------------*/
      case 0x28 : case 0x30 :
         return( read_tab_entry( fo_offset_abs, l_size ) ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   switch( i_tag )
   {
      /*--( Make & Model are in the same tag: 2 strings separated by '\0' )*/
      case TC_ID_MAKE_AND_MODEL :
      {
         /*----------------------------------------------------------------*/
         int i_lg_make ;
         /*----------------------------------------------------------------*/
         if( l_size >= ( int )sizeof( tb_c_val ) )
         {  l_size = sizeof( tb_c_val ) - 1 ; }
         /*----------------------------------------------------------------*/
         if(    m_fa.set_offset( fo_offset_abs ) != 0
             || m_fa.read_buffer( l_size, tb_c_val ) != 0
           )
         {  return( -1 ) ; }
         /*----------------------------------------------------------------*/
         tb_c_val[ l_size ] = '\0' ;
         i_lg_make = strlen( tb_c_val ) ;
         /*----------------------------------------------------------------*/
         if(    m_fi.reserve_col( COL_EXIF_MAKE )
             && m_fi.prepare_string( tb_c_val, i_lg_make, s_val ) > 0
           )
         {  m_f.val_s( COL_EXIF_MAKE ) = s_val ; }
         /*----------------------------------------------------------------*/
         if(    m_fi.reserve_col( COL_EXIF_MODEL )
             && m_fi.prepare_string( tb_c_val + i_lg_make + 1,
                                     l_size   - i_lg_make - 1,
                                     s_val
                                   ) > 0
           )
         {  m_f.val_s( COL_EXIF_MODEL ) = s_val ; }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*--------------------------------------------------------------------+
      ! The "reservation" is done only for non-empty strings                !
      ! They seem to be not always stored in the "right order"              !
      +--------------------------------------------------------------------*/
      case TC_ID_DESCRIPTION :
      case TC_ID_FIRMWARE :
      case TC_ID_COMPONENT :
      case TC_ID_OWNER_NAME :
      {
         /*----------------------------------------------------------------*/
         int i_col = st_ciff_conv_ciff_tag_col( i_tag ) ;
         /*----------------------------------------------------------------*/
         if(    m_fa.set_offset( fo_offset_abs ) != 0
             || m_fi.file_read_tb_c( l_size, s_val ) != 0
           )
         {  break ; }
         /*----------------------------------------------------------------*/
         if( !s_val.empty() && m_fi.reserve_col( i_col ) )
         {  m_f.val_s( i_col ) = s_val ; }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      case TC_ID_IMAGE_SPEC :
      {
         /*----------------------------------------------------------------*/
         str_ciff_image_spec ciff_img ;
         /*----------------------------------------------------------------*/
         if( l_size < ( int )sizeof( ciff_img ) )
         {  wxFAIL ; break ; }
         /*----------------------------------------------------------------*/
         if(    m_fa.set_offset( fo_offset_abs ) != 0
             || m_fa.read_buffer( sizeof( ciff_img ), &ciff_img ) != 0
           )
         {  return( -3 ) ; }

         /*----------------------------------------------------------------*/
         m_fi.init_img_x_y( m_p_fnc_udw( ciff_img.dw_imagewidth  ),
                            m_p_fnc_udw( ciff_img.dw_imageheight )
                          ) ;
         m_fi.init_img_bpp( m_p_fnc_udw( ciff_img.dw_color_bit_depth ) ) ;
         /*----------------------------------------------------------------*/
         switch( m_p_fnc_udw( ciff_img.dw_rotation_angle ) )
         {
            /*-------------------------------------------------------------*/
            case   0 : m_fi.init_exif_rotation( IV_ROTATION_0 )   ; break ;
            case  90 : m_fi.init_exif_rotation( IV_ROTATION_90 )  ; break ;
            case 180 : m_fi.init_exif_rotation( IV_ROTATION_180 ) ; break ;
            case 270 : m_fi.init_exif_rotation( IV_ROTATION_270 ) ; break ;
            /*-------------------------------------------------------------*/
            default  : m_fi.init_exif_rotation( IV_ROTATION_NB )  ; break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      case TC_ID_SHOT_INFO :
      {
         /*----------------------------------------------------------------*/
         str_ciff_shot_info ciff_shot ;

         /*----------------------------------------------------------------*/
         if( l_size < ( int )sizeof( ciff_shot ) )
         {  wxFAIL ; break ; }
         /*----------------------------------------------------------------*/
         if(    m_fa.set_offset( fo_offset_abs ) != 0
             || m_fa.read_buffer( sizeof( ciff_shot ), &ciff_shot ) != 0
           )
         {  break ; }

         /*--( ISO )-------------------------------------------------------*/
         if( m_fi.reserve_col( COL_EXIF_ISOSPEEDRATINGS ) )
         {
            /*-------------------------------------------------------------*/
            int i_iso
              =   50.0 * pow( 2.0, m_p_fnc_uw( ciff_shot.w_iso ) / 32.0 )
                / 16.0 ;
            /*-------------------------------------------------------------*/
            m_f.val_ll( COL_EXIF_ISOSPEEDRATINGS ) = i_iso ;
            m_f.val_s( COL_EXIF_ISOSPEEDRATINGS ) << i_iso ;
            /*-------------------------------------------------------------*/
         }

         /*--( Aperture Value )--------------------------------------------*/
         if( m_fi.reserve_col( COL_EXIF_APERTUREVALUE ) )
         {
            /*-------------------------------------------------------------*/
            double do_av = pow( 2.0, m_p_fnc_sw( ciff_shot.w_av ) / 64.0 ) ;
            /*--( reserve the col unconditionally )------------------------*/
            m_fi.reserve_col( COL_EXIF_FNUMBER ) ;
            /*-------------------------------------------------------------*/
            if( sr::round_0( do_av * 10 ) != 0 )
            {  /*----------------------------------------------------------*/
               wxString s_av ;
               /*----------------------------------------------------------*/
               s_av.Printf( "F%0.1lf", sr::round_1( do_av ) ) ;
               /*----------------------------------------------------------*/
               m_f.val_do( COL_EXIF_APERTUREVALUE ) = do_av ;
               m_f.val_s( COL_EXIF_APERTUREVALUE )  = s_av  ;
               /*----------------------------------------------------------*/
               m_f.val_do( COL_EXIF_FNUMBER )       = do_av ;
               m_f.val_s( COL_EXIF_FNUMBER )        = s_av  ;
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
         }

         /*--( Shutter speed )---------------------------------------------*/
         if( m_fi.reserve_col( COL_EXIF_SHUTTERSPEEDVALUE ) )
         {
            /*-------------------------------------------------------------*/
            double do_tv = pow( 0.5, m_p_fnc_sw( ciff_shot.w_tv ) / 32.0 ) ;
            /*-------------------------------------------------------------*/
            if( sr::round_0( do_tv * 100000 ) != 0 )
            {  m_fi.init_exif_shutterspeedvalue( do_tv ) ; }
            /*-------------------------------------------------------------*/
         }

         /*--( Exposure comp )---------------------------------------------*/
         if( m_fi.reserve_col( COL_EXIF_EXPOSUREBIASVALUE ) )
         {
            /*-------------------------------------------------------------*/
            short shi_ec = m_p_fnc_sw( ciff_shot.w_exposure_comp ) ;
            /*-------------------------------------------------------------*/
            m_f.val_s( COL_EXIF_EXPOSUREBIASVALUE ).Printf(
                                    "EV%0.1lf",
                                    sr::round_1( st_ciff_canon_ev( shi_ec ) )
                                                          ) ;
            /*-------------------------------------------------------------*/
         }

         /*--( White Balance )---------------------------------------------*/
         if( m_fi.reserve_col( COL_EXIF_WHITEBALANCE ) )
         {
            /*-------------------------------------------------------------*/
            switch( m_p_fnc_uw( ciff_shot.w_white_balance ) )
            {  /*----------------------------------------------------------*/
               case  0 : p_c_tmp = "auto"                 ; break ;
               case  1 : p_c_tmp = "daylight"             ; break ;
               case  2 : p_c_tmp = "cloudy"               ; break ;
               case  3 : p_c_tmp = "tungsten"             ; break ;
               case  4 : p_c_tmp = "fluorescent"          ; break ;
               case  5 : p_c_tmp = "flash"                ; break ;
               case  6 : p_c_tmp = "custom"               ; break ;
               case  7 : p_c_tmp = "black and white"      ; break ;
               case  8 : p_c_tmp = "shade"                ; break ;
               case  9 : p_c_tmp = "manual temperature"   ; break ;
               case 14 : p_c_tmp = "daylight fluorescent" ; break ;
               case 17 : p_c_tmp = "underwater"           ; break ;
               default : p_c_tmp = "unknown"              ; break ;
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
            m_f.val_s( COL_EXIF_WHITEBALANCE ) = p_c_tmp ;
            /*-------------------------------------------------------------*/
         }

         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      case TC_ID_CAMERA_SETTINGS :
      {
         /*----------------------------------------------------------------*/
         str_ciff_camera_settings ciff_camera ;

         /*----------------------------------------------------------------*/
         if( l_size < ( int )sizeof( ciff_camera ) )
         {  wxFAIL ; break ; }
         /*----------------------------------------------------------------*/
         if(    m_fa.set_offset( fo_offset_abs ) != 0
             || m_fa.read_buffer( sizeof( ciff_camera ), &ciff_camera ) != 0
           )
         {  break ; }

         /*--( Flash )-----------------------------------------------------*/
         if( m_fi.reserve_col( COL_EXIF_FLASH ) )
         {
            /*-------------------------------------------------------------*/
            switch( m_p_fnc_uw( ciff_camera.w_flash_mode ) )
            {  /*----------------------------------------------------------*/
               case  0 : p_c_tmp = "no flash"                      ; break ;
               case  1 : p_c_tmp = "auto"                          ; break ;
               case  2 : p_c_tmp = "flash"                         ; break ;
               case  3 : p_c_tmp = "red-eye reduction mode"        ; break ;
               case  4 : p_c_tmp = "slow-sync"                     ; break ;
               case  5 : p_c_tmp = "red-eye reduction mode (auto)" ; break ;
               case  6 : p_c_tmp = "red-eye reduction mode (on)"   ; break ;
               case 16 : p_c_tmp = "external flash"                ; break ;
               default : p_c_tmp = "undefined"                     ; break ;
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
            m_f.val_s( COL_EXIF_FLASH ) = p_c_tmp ;
            /*-------------------------------------------------------------*/
         }

         /*--( Assisted mode: program )------------------------------------*/
         if( m_fi.reserve_col( COL_EXIF_EXPOSUREPROGRAM ) )
         {
            /*-------------------------------------------------------------*/
            switch( m_p_fnc_uw( ciff_camera.w_easy_mode ) )
            {  /*----------------------------------------------------------*/
               case  0 : p_c_tmp = "full auto"     ; break ;
               case  1 : p_c_tmp = "manual"        ; break ;
               case  2 : p_c_tmp = "landscape"     ; break ;
               case  3 : p_c_tmp = "fast shutter"  ; break ;
               case  4 : p_c_tmp = "slow shutter"  ; break ;
               case  5 : p_c_tmp = "night"         ; break ;
               case  6 : p_c_tmp = "black & white" ; break ;
               case  7 : p_c_tmp = "sepia"         ; break ;
               case  8 : p_c_tmp = "portrait"      ; break ;
               case  9 : p_c_tmp = "sports"        ; break ;
               case 10 : p_c_tmp = "macro"         ; break ;
               case 11 : p_c_tmp = "pan focus"     ; break ;
               default : p_c_tmp = "undefined"     ; break ;
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
            m_f.val_s( COL_EXIF_EXPOSUREPROGRAM ) = p_c_tmp ;
            /*-------------------------------------------------------------*/
         }

         /*--( Metering Mode )---------------------------------------------*/
         if( m_fi.reserve_col( COL_EXIF_METERINGMODE ) )
         {
            /*-------------------------------------------------------------*/
            switch( m_p_fnc_uw( ciff_camera.w_metering_mode ) )
            {  /*----------------------------------------------------------*/
               case  0 : p_c_tmp = "default"                 ; break ;
               case  1 : p_c_tmp = "spot"                    ; break ;
               case  3 : p_c_tmp = "evaluative"              ; break ;
               case  4 : p_c_tmp = "partial"                 ; break ;
               case  5 : p_c_tmp = "center weighted average" ; break ;
               default : p_c_tmp = "undefined"               ; break ;
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
            m_f.val_s( COL_EXIF_METERINGMODE ) = p_c_tmp ;
            /*-------------------------------------------------------------*/
         }

         /*--( Exposure Mode )---------------------------------------------*/
         if( m_fi.reserve_col( COL_EXIF_EXPOSUREMODE ) )
         {
            /*-------------------------------------------------------------*/
            switch( m_p_fnc_uw( ciff_camera.w_exposure_mode ) )
            {  /*----------------------------------------------------------*/
               case  0 : p_c_tmp = "auto"              ; break ;
               case  1 : p_c_tmp = "program"           ; break ;
               case  2 : p_c_tmp = "shutter priority"  ; break ; // tv
               case  3 : p_c_tmp = "aperture priority" ; break ; // av
               case  4 : p_c_tmp = "manual"            ; break ;
               case  5 : p_c_tmp = "depth of field"    ; break ;
               default : p_c_tmp = "unknown"           ; break ;
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
            m_f.val_s( COL_EXIF_EXPOSUREMODE ) = p_c_tmp ;
            /*-------------------------------------------------------------*/
         }

         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*--( The data is in dir structure because it's <= 8 chars )---------*/
      case TC_ID_FOCAL_LENGTH :
      {
         /*----------------------------------------------------------------*/
         if( !m_fi.reserve_col( COL_EXIF_FOCALLENGTH ) ) { break ; }
         /*----------------------------------------------------------------*/
         str_ciff_focal_length ciff_focal ;
         memcpy( &ciff_focal, &ciff_entry.dw_size, sizeof( ciff_focal ) ) ;
         /*----------------------------------------------------------------*/
         double do_focale ;
         /*----------------------------------------------------------------*/
         do_focale = m_p_fnc_uw( ciff_focal.w_focal_mm ) ;
         /*-----------------------------------------------------------------+
         ! Here it's a bit of reverse engineering ... by comparing          !
         ! results with different other softwares                           !
         +-----------------------------------------------------------------*/
         if( m_p_fnc_uw( ciff_focal.w_type_scale ) == 2 )
         {  do_focale /= 32 ; }
         /*----------------------------------------------------------------*/
         if( do_focale > 0 ) { m_fi.init_exif_focallength( do_focale ) ; }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      case TC_ID_COLORSPACE :
      {
         /*----------------------------------------------------------------*/
         str_ciff_colorspace ciff_col ;

         /*----------------------------------------------------------------*/
         if( l_size < ( int )sizeof( ciff_col ) )
         {  wxFAIL ; break ; }
         /*--( Subject Distance )------------------------------------------*/
         if(    m_fa.set_offset( fo_offset_abs ) != 0
             || m_fa.read_buffer( sizeof( ciff_col ), &ciff_col ) != 0
           )
         {  break ; }

         /*----------------------------------------------------------------*/
         if( m_fi.reserve_col( COL_EXIF_COLORSPACE ) )
         {
            /*-------------------------------------------------------------*/
            switch( m_p_fnc_uw( ciff_col.w_colorspace ) )
            {  case 1      : p_c_tmp = "sRGB"         ; break ;
               case 2      : p_c_tmp = "Adobe RGB"    ; break ;
               case 0xFFFF : p_c_tmp = "uncalibrated" ; break ;
               default     : p_c_tmp = "unknown"      ; break ;
            }
            /*-------------------------------------------------------------*/
            m_f.val_s( COL_EXIF_COLORSPACE ) = p_c_tmp ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      case TC_ID_TIMESTAMP :
      {
         /*----------------------------------------------------------------*/
         const int co_i_col = COL_EXIF_DATETIMEORIGINAL ;
         wxUint32  dw_t ;
         /*----------------------------------------------------------------*/
         if( l_size < 4 ) { wxFAIL ; break ; }
         /*----------------------------------------------------------------*/
         if( !m_fi.reserve_col( co_i_col ) ) { break ; }
         /*----------------------------------------------------------------*/
         if(    m_fa.set_offset( fo_offset_abs ) != 0
             || m_fa.read_data( dw_t ) != 0
           )
         {  return( -8 ) ; }

         /*----------------------------------------------------------------*/
         wxDateTime dt ;
         /*----------------------------------------------------------------*/
         if( sr::init_datetime( ( time_t )m_p_fnc_udw( dw_t ), dt ) == 0 )
         {  m_fi.init_date( co_i_col,  dt ) ; }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*--------------------------------------------------------------------+
      ! The smallest thumbnail is taken.                                    !
      ! Note that after initialization the size is -1                       !
      +--------------------------------------------------------------------*/
      case TC_ID_JPEG_FROM_RAW :
      case TC_ID_THUMBNAIL :
      {
         /*----------------------------------------------------------------*/
         bool boo_smaller = ( l_size < m_f.get_thumbnail_size() ) ;
         /*----------------------------------------------------------------*/
         if( m_fi.reserve_col( COL_EXIF_THUMBNAIL_OFFSET ) || boo_smaller )
         {  /*--( The absolute is saved and the relative displayed )-------*/
            m_f.set_thumbnail_offset( fo_offset_abs ) ;
            m_f.val_s( COL_EXIF_THUMBNAIL_OFFSET ).Printf(
                                       _T( "%" ) wxLongLongFmtSpec _T( "d" ),
                                       fo_offset_rel
                                                         ) ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         if( m_fi.reserve_col( COL_EXIF_THUMBNAIL_SIZE ) || boo_smaller )
         {  /*-------------------------------------------------------------*/
            m_f.set_thumbnail_size( l_size ) ;
            m_f.val_s( COL_EXIF_THUMBNAIL_SIZE ).Printf( "%ld", l_size ) ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_crw::read_tab_entry( wxFileOffset fo_offset, long l_size )
{
   /*----------------------------------------------------------------------*/
   str_ciff_offset        ciff_offset          ;
   str_ciff_heap_nb_entry ciff_nb_entry        ;
   wxMemoryBuffer         mb_buffer_ciff_entry ;
   str_ciff_entry         *p_ciff_entry        ;
   int                    i_nb_entry           ;
   int                    i_num                ;

   /*--( The "beginning" is "at the end" ... )-----------------------------*/
   if(    m_fa.set_offset( fo_offset + l_size - sizeof( ciff_offset ) ) != 0
       || m_fa.read_buffer( sizeof( ciff_offset ), &ciff_offset ) != 0
     )
   {  return( -1 ) ; }

   /*--( Get the number of entries in the tab )----------------------------*/
   if(    m_fa.set_offset( fo_offset + m_p_fnc_udw( ciff_offset.dw_offset )
                         ) != 0
       || m_fa.read_buffer( sizeof( ciff_nb_entry ), &ciff_nb_entry ) != 0
     )
   {  return( -2 ) ; }

   /*----------------------------------------------------------------------*/
   i_nb_entry = m_p_fnc_uw( ciff_nb_entry.w_nb ) ;

   /*--( Read the complete table )-----------------------------------------*/
   if( m_fa.read_buffer( i_nb_entry * sizeof( str_ciff_entry ),
                         mb_buffer_ciff_entry
                       ) != 0
     )
   {  return( -3 ) ; }
   /*----------------------------------------------------------------------*/
   p_ciff_entry = ( str_ciff_entry * )mb_buffer_ciff_entry.GetData() ;

   /*--( And scan ... )----------------------------------------------------*/
   for( i_num = 0 ; i_num < i_nb_entry ; ++i_num )
   {
      /*-------------------------------------------------------------------*/
      if( read_tag( fo_offset, p_ciff_entry[ i_num ] ) != 0 )
      {  return( -4 ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}


/*-------------------------------------------------------------------------*/
int CFileInit_crw::ciff_init()
{
   /*----------------------------------------------------------------------*/
   str_ciff_header header ;

   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( header ), &header ) != 0 )
   {  return( -1 ) ; }

   /*--( Integer "endian"nity )--------------------------------------------*/
   switch( header.w_align )
   {
      /*-------------------------------------------------------------------*/
      case SR_TWOCC( 'I', 'I' ) :
         m_p_fnc_sw  = sr::conv_le_sw  ;
         m_p_fnc_uw  = sr::conv_le_uw  ;
         m_p_fnc_sdw = sr::conv_le_sdw ;
         m_p_fnc_udw = sr::conv_le_udw ;
         break ;
      case SR_TWOCC( 'M', 'M' ) :
         m_p_fnc_sw  = sr::conv_be_sw  ;
         m_p_fnc_uw  = sr::conv_be_uw  ;
         m_p_fnc_sdw = sr::conv_be_sdw ;
         m_p_fnc_udw = sr::conv_be_udw ;
         break ;
      /*-------------------------------------------------------------------*/
      default : return( -2 ) ;
      /*-------------------------------------------------------------------*/
   }

   /*--( Minimal control )-------------------------------------------------*/
   if( memcmp( header.tb_c_id, "HEAPCCDR", 8 ) != 0 )
   {  return( -3 ) ; }

   /*----------------------------------------------------------------------*/
   m_fo_offset_first_heap = m_p_fnc_udw( header.dw_offset ) ;

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( On exit the file pointer will be at the data end )-------------------*/
int CFileInit_crw::ciff_read()
{
   /*----------------------------------------------------------------------*/
   if( ciff_init() != 0  )
   {  return( -1 ) ; }

   /*--( Then the "loading/analyse" is done from the end of the file )-----*/
   if( read_tab_entry( m_fo_offset_first_heap,
                       m_f.get_size().ToULong() - m_fo_offset_first_heap
                     ) != 0
     )
   {  return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_crw()
{
   /*----------------------------------------------------------------------*/
   m_s_type_det = "crw" ;
   /*----------------------------------------------------------------------*/
   return( CFileInit_crw( *this ).ciff_read() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                       End of file CFileInit_crw.cpp                       !
+==========================================================================*/
