/*====================================+=====================================+
! File CFileInit_asf.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                      Audio file: ".wma"                                   !
!                      Video file: ".asf", ".wmv"                           !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
struct str_GUID ;
struct str_asf_object_header ;
struct str_asf_file_header ;
struct str_asf_file_properties ;
struct str_asf_stream_properties ;
struct str_asf_audio_specific_properties ;
struct str_asf_video_specific_properties ;
struct str_asf_codec_list ;
struct str_asf_codec_entries ;
struct str_asf_header_extension_object ;
struct str_asf_metadata_library_object ;
struct str_asf_metadata_library_record_object ;
struct str_asf_content_description ;
struct str_asf_extended_content_description ;
struct str_asf_descriptor_val ;

/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_asf : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_asf( CFileInit &parent ) : CFileInit_type_base( parent )
      {  ; }
      /*-------------------------------------------------------------------*/
      void swap_le_data( str_GUID &guid ) ;
      void swap_le_data( str_asf_object_header &aoh ) ;
      int  read_le_data( str_GUID &guid ) ;
      int  read_le_data( str_asf_object_header &aoh ) ;
      int  read_le_data( str_asf_file_header &afh ) ;
      int  read_le_data( str_asf_file_properties &afp ) ;
      int  read_le_data( str_asf_stream_properties &asp ) ;
      int  read_le_data( str_asf_audio_specific_properties &aap ) ;
      int  read_le_data( str_asf_video_specific_properties &avp ) ;
      int  read_le_data( str_asf_codec_list &acl ) ;
      int  read_le_data( str_asf_codec_entries &ace ) ;
      int  read_le_data( str_asf_header_extension_object &ahe ) ;
      int  read_le_data( str_asf_metadata_library_object &aml ) ;
      int  read_le_data( str_asf_metadata_library_record_object &amr ) ;
      int  read_le_data( str_asf_content_description &acd ) ;
      int  read_le_data( str_asf_extended_content_description &aec ) ;
      int  read_le_data( str_asf_descriptor_val &adv ) ;
      /*-------------------------------------------------------------------*/
      int  read_varchar_c( wxString &s ) ;
      int  read_varchar_wc( wxString &s ) ;
      int  read_var_name( wxString &s_var ) ;
      int  read_var_val( wxString &s_val ) ;
      int  image_info( bool boo_video, size_t sz_size ) ;
      int  asf_read() ;

   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*-------------------------------------------------------------------------*/
struct str_GUID
{
   /*----------------------------------------------------------------------*/
   wxUint32 Data1      ;
   wxUint16 Data2      ;
   wxUint16 Data3      ;
   wxUint8  Data4[ 8 ] ;
   /*----------------------------------------------------------------------*/
   bool operator == ( const str_GUID &g2 )
   {  return( memcmp( this, &g2, sizeof( str_GUID ) ) == 0 ) ; }
   bool operator != ( const str_GUID &g2 )
   {  return( !( *this == g2 ) ) ; }
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_asf_object_header
{
   /*----------------------------------------------------------------------*/
   str_GUID guid_object_id  ;
   wxUint64 ddw_object_size ;
   /*----------------------------------------------------------------------*/
} ;
/*-------------------------------------------------------------------------*/
struct str_asf_file_header
{
   /*----------------------------------------------------------------------*/
   str_asf_object_header object_header               ;
   wxUint32              dw_number_of_header_objects ;
   wxUint8               ___b_reserved_1             ; // 0x01
   wxUint8               ___b_reserved_2             ; // 0x02
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_asf_file_properties
{
   /*----------------------------------------------------------------------*/
   enum { FP_BROADCAST_FLAG = 0x01, FP_SEEKABLE_FLAG = 0x02 } ;
   /*----------------------------------------------------------------------*/
   str_GUID guid_file_id                   ;
   wxUint64 ___ddw_file_size               ;
   wxUint64 ___ddw_creation_date           ;
   wxUint64 ___ddw_data_packets_count      ;
   wxUint64 ddw_play_duration              ;
   wxUint64 ___ddw_send_duration           ;
   wxUint64 ddw_preroll                    ;
   wxUint32 dw_flags                       ; // Broadcast Flag  1 (LSB)
                                             // Seekable  Flag  1
                                             // Reserved       30
   wxUint32 ___dw_minimum_data_packet_size ;
   wxUint32 ___dw_maximum_data_packet_size ;
   wxUint32 dw_maximum_bitrate             ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_asf_stream_properties
{
   /*----------------------------------------------------------------------*/
   str_GUID guid_stream_type                   ;
   str_GUID guid_error_correction_type         ;
   wxUint64 ___ddw_time_offset                 ;
   wxUint32 dw_type_specific_data_length       ;
   wxUint32 ___dw_error_correction_data_length ;
   wxUint16 ___w_flags                         ; // Stream Number 7 (LSB)
                                                 // Reserved      8
                                                 // Encrypted Content Flag 1
   wxUint32 ___dw_reserved                     ;
   /*--( The audio part is wav )-------------------------------------------*/
/*
   Type-Specific    Data BYTE varies
   Error Correction Data BYTE varies
*/
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_asf_audio_specific_properties
{
   /*----------------------------------------------------------------------*/
   str_WAVEFORMATEX wfx ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_asf_video_specific_properties
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_encoded_image_width  ;
   wxUint32 dw_encoded_image_height ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_asf_codec_list
{
   /*----------------------------------------------------------------------*/
   str_GUID ___guid_reserved       ;
   wxUint32 dw_codec_entries_count ;
   /*----------------------------------------------------------------------*/
/*
   Codec Entries    See below    varies
*/
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_asf_codec_entries
{
   /*----------------------------------------------------------------------*/
   enum { CT_VIDEO = 0x0001, CT_AUDIO = 0x0002 } ;
   /*----------------------------------------------------------------------*/
   wxUint16 w_type ;
   /*----------------------------------------------------------------------*/
/*
   WORD  w_codec_name_length        ;
   WCHAR wc_codec_name              ; varies
   WORD  w_codec_description_length ;
   WCHAR wc_codec_description       ; varies
   WORD  w_codec_information_length ;
   BYTE  b_codec_information        ; varies
*/
   /*----------------------------------------------------------------------*/
} ;


/*-------------------------------------------------------------------------*/
struct str_asf_header_extension_object
{
   /*----------------------------------------------------------------------*/
   str_GUID ___guid_reserved              ;
   wxUint16 ___w_reserved                 ;
   wxUint32 dw_header_extension_data_size ;
   /*----------------------------------------------------------------------*/
/*
   Header extension data varies
*/
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_asf_metadata_library_object
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_record_count ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_asf_metadata_library_record_object
{
   /*----------------------------------------------------------------------*/
   wxUint16 ___w_language_list_index ;
   wxUint16 ___w_stream_number       ;
   wxUint16 w_name_length            ;
   wxUint16 w_data_type              ;
   wxUint32 dw_data_length           ;
   /*----------------------------------------------------------------------*/
/*
   NAME WCHAR varies
   DATA       varies
*/
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_asf_content_description
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_title_length       ;
   wxUint16 w_author_length      ;
   wxUint16 w_copyright_length   ;
   wxUint16 w_description_length ;
   wxUint16 w_rating_length      ;
   /*----------------------------------------------------------------------*/
/*
   Title       WCHAR Varies
   Author      WCHAR Varies
   Copyright   WCHAR Varies
   Description WCHAR Varies
   Rating      WCHAR Varies
*/
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_asf_extended_content_description
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_descriptors_count ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_asf_descriptor_val
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_descriptor_value_data_type ;
   wxUint16 w_descriptor_value_length    ;
// Descriptor Value See text varies
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*-------------------------------------------------------------------------*/
void CFileInit_asf::swap_le_data( str_GUID &guid )
{
   /*----------------------------------------------------------------------*/
   guid.Data1 = wxUINT32_SWAP_ALWAYS( guid.Data1 ) ;
   guid.Data2 = wxUINT16_SWAP_ALWAYS( guid.Data2 ) ;
   guid.Data3 = wxUINT16_SWAP_ALWAYS( guid.Data3 ) ;
   //guid.Data4 = wxUINT64_SWAP_ALWAYS( guid.Data4 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit_asf::swap_le_data( str_asf_object_header &aoh )
{
   /*----------------------------------------------------------------------*/
   swap_le_data( aoh.guid_object_id ) ;
   aoh.ddw_object_size = wxUINT64_SWAP_ALWAYS( aoh.ddw_object_size ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::read_le_data( str_GUID &guid )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( guid ), &guid ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   swap_le_data( guid ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::read_le_data( str_asf_object_header &aoh )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( aoh ), &aoh ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   swap_le_data( aoh ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::read_le_data( str_asf_file_header &afh )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( afh ), &afh ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   swap_le_data( afh.object_header ) ;
     afh.dw_number_of_header_objects
   = wxUINT32_SWAP_ALWAYS( afh.dw_number_of_header_objects ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::read_le_data( str_asf_file_properties &afp )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( afp ), &afp ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   swap_le_data( afp.guid_file_id ) ;
   afp.ddw_play_duration = wxUINT64_SWAP_ALWAYS( afp.ddw_play_duration ) ;
   afp.ddw_preroll       = wxUINT64_SWAP_ALWAYS( afp.ddw_preroll ) ;
   afp.dw_flags          = wxUINT32_SWAP_ALWAYS( afp.dw_flags ) ;
   afp.dw_flags          = wxUINT32_SWAP_ALWAYS( afp.dw_maximum_bitrate ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::read_le_data( str_asf_stream_properties &asp )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( asp ), &asp ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   swap_le_data( asp.guid_stream_type ) ;
   swap_le_data( asp.guid_error_correction_type ) ;
     asp.dw_type_specific_data_length
   = wxUINT32_SWAP_ALWAYS( asp.dw_type_specific_data_length ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::read_le_data( str_asf_audio_specific_properties &aap )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_le_data( aap.wfx ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::read_le_data( str_asf_video_specific_properties &avp )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( avp ), &avp ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
     avp.dw_encoded_image_width
   = wxUINT32_SWAP_ALWAYS( avp.dw_encoded_image_width ) ;
     avp.dw_encoded_image_height
   = wxUINT32_SWAP_ALWAYS( avp.dw_encoded_image_height ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::read_le_data( str_asf_codec_list &acl )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( acl ), &acl ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   //swap_le_data( acl.guid_reserved ) ;
     acl.dw_codec_entries_count
   = wxUINT32_SWAP_ALWAYS( acl.dw_codec_entries_count ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::read_le_data( str_asf_codec_entries &ace )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_le_data( ace.w_type ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::read_le_data( str_asf_header_extension_object &ahe )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( ahe ), &ahe ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   //swap_le_data( ahe.guid_reserved ) ;
     ahe.dw_header_extension_data_size
   = wxUINT32_SWAP_ALWAYS( ahe.dw_header_extension_data_size ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::read_le_data( str_asf_metadata_library_object &aml )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_le_data( aml.w_record_count ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::read_le_data( str_asf_metadata_library_record_object &amr)
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( amr ), &amr ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   amr.w_name_length  = wxUINT16_SWAP_ALWAYS( amr.w_name_length ) ;
   amr.w_data_type    = wxUINT16_SWAP_ALWAYS( amr.w_data_type ) ;
   amr.dw_data_length = wxUINT32_SWAP_ALWAYS( amr.dw_data_length ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::read_le_data( str_asf_content_description &acd )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( acd ), &acd ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   acd.w_title_length = wxUINT16_SWAP_ALWAYS( acd.w_title_length ) ;
   acd.w_author_length = wxUINT16_SWAP_ALWAYS( acd.w_author_length ) ;
   acd.w_copyright_length = wxUINT16_SWAP_ALWAYS( acd.w_copyright_length ) ;
     acd.w_description_length
   = wxUINT16_SWAP_ALWAYS( acd.w_description_length ) ;
   acd.w_rating_length = wxUINT16_SWAP_ALWAYS( acd.w_rating_length ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::read_le_data( str_asf_extended_content_description &aec )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_le_data( aec.w_descriptors_count ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::read_le_data( str_asf_descriptor_val &adv )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( adv ), &adv ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
     adv.w_descriptor_value_data_type
   = wxUINT16_SWAP_ALWAYS( adv.w_descriptor_value_data_type ) ;
     adv.w_descriptor_value_length
   = wxUINT16_SWAP_ALWAYS( adv.w_descriptor_value_length ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static const str_GUID st_ASF_Header_Object
= { 0x75B22630,0x668E,0x11CF, { 0xA6,0xD9,0x00,0xAA,0x00,0x62,0xCE,0x6C } } ;
static const str_GUID st_ASF_File_Properties_Object
= { 0x8CABDCA1,0xA947,0x11CF, { 0x8E,0xE4,0x00,0xC0,0x0C,0x20,0x53,0x65 } } ;
static const str_GUID st_ASF_Stream_Properties_Object
= { 0xB7DC0791,0xA9B7,0x11CF, { 0x8E,0xE6,0x00,0xC0,0x0C,0x20,0x53,0x65 } } ;
static const str_GUID st_ASF_Header_Extension_Object
= { 0x5FBF03B5,0xA92E,0x11CF, { 0x8E,0xE3,0x00,0xC0,0x0C,0x20,0x53,0x65 } } ;
static const str_GUID st_ASF_Metadata_Library_Object
= { 0x44231C94,0x9498,0x49D1, { 0xA1,0x41,0x1D,0x13,0x4E,0x45,0x70,0x54 } } ;
static const str_GUID st_ASF_Codec_List_Object
= { 0x86D15240,0x311D,0x11D0, { 0xA3,0xA4,0x00,0xA0,0xC9,0x03,0x48,0xF6 } } ;
static const str_GUID st_ASF_Content_Description_Object
= { 0x75B22633,0x668E,0x11CF, { 0xA6,0xD9,0x00,0xAA,0x00,0x62,0xCE,0x6C } } ;
static const str_GUID st_ASF_Extended_Content_Description_Object
= { 0xD2D0A440,0xE307,0x11D2, { 0x97,0xF0,0x00,0xA0,0xC9,0x5E,0xA8,0x50 } } ;
static const str_GUID st_ASF_Audio_Media
= { 0xF8699E40,0x5B4D,0x11CF, { 0xA8,0xFD,0x00,0x80,0x5F,0x5C,0x44,0x2B } } ;
static const str_GUID st_ASF_Video_Media
= { 0xBC19EFC0,0x5B4D,0x11CF, { 0xA8,0xFD,0x00,0x80,0x5F,0x5C,0x44,0x2B } } ;


/*-------------------------------------------------------------------------*/
static int st_asf_conv_tag_col( bool boo_video, const wxString &s_var )
{
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "WM/AlbumTitle" ) == 0 )
   {  return( boo_video ? COL_NB : COL_AUDTAG_ALBUM ) ; }
   /*----------------------------------------------------------------------*/
   if(    s_var.CmpNoCase( "WM/Composer" ) == 0
       || s_var.CmpNoCase( "ID3/TCOM" ) == 0
     )
   {  return( boo_video ? COL_NB : COL_AUDTAG_COMPOS ) ; }
   /*----------------------------------------------------------------------*/
   if(    s_var.CmpNoCase( "WM/GenreID" ) == 0
       || s_var.CmpNoCase( "WM/Genre" ) == 0
     )
   {  return( boo_video ? COL_VIDTAG_GENRE : COL_AUDTAG_GENRE ) ; }
   /*----------------------------------------------------------------------*/
   if(    s_var.CmpNoCase( "WM/EncodedBy" ) == 0
       || s_var.CmpNoCase( "WM/CodedBy" ) == 0
       || s_var.CmpNoCase( "ID3/TENC" ) == 0
     )
   {  return( boo_video ? COL_VIDTAG_ENGINEER : COL_AUDTAG_ENCOD_BY ) ; }
   /*----------------------------------------------------------------------*/
   if(    s_var.CmpNoCase( "WM/OriginalArtist" ) == 0
       || s_var.CmpNoCase( "ID3/TOPE" ) == 0
     )
   {  return( boo_video ? COL_VIDTAG_SOURCE : COL_AUDTAG_ORG_ART ) ; }
   /*----------------------------------------------------------------------*/
   if(    s_var.CmpNoCase( "Author" ) == 0
       || s_var.CmpNoCase( "ID3/TPE1" ) == 0
     )
   {  return( boo_video ? COL_VIDTAG_ARTIST : COL_AUDTAG_ARTIST ) ; }
   /*----------------------------------------------------------------------*/
   if(    s_var.CmpNoCase( "WM/Track" ) == 0
       || s_var.CmpNoCase( "WM/TrackNumber" ) == 0
     )
   {  return( boo_video ? COL_NB : COL_AUDTAG_TRACK ) ; }
   /*----------------------------------------------------------------------*/
   if(    s_var.CmpNoCase( "WM/PartOfSet" ) == 0
       || s_var.CmpNoCase( "ID3/TPOS" ) == 0
     )
   {  return( boo_video ? COL_NB : COL_AUDTAG_DISK ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "Title" ) == 0 )
   {  return( boo_video ? COL_VIDTAG_TITLE : COL_AUDTAG_TITLE ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "WM/Year" ) == 0 )
   {  return( boo_video ? COL_NB : COL_AUDTAG_YEAR ) ; }
   /*----------------------------------------------------------------------*/
   if(    s_var.CmpNoCase( "WM/URL" ) == 0
       || s_var.CmpNoCase( "UserWebURL" ) == 0
       || s_var.CmpNoCase( "ID3/WXXX" ) == 0
     )
   {  return( boo_video ? COL_NB : COL_AUDTAG_URL ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "Description" ) == 0 )
   {  return( boo_video ? COL_VIDTAG_COMMENT : COL_AUDTAG_COMMENT ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "ID3/TDAT" ) == 0 )
   {  return( boo_video ? COL_VIDTAG_CRE_DATE : COL_NB ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "ID3/TSSE" ) == 0 )
   {  return( boo_video ? COL_VIDTAG_SOFTWARE : COL_NB ) ; }
   /*----------------------------------------------------------------------*/
   return( COL_NB ) ;
   /*----------------------------------------------------------------------*/
}

/*--( String: WORD (size) + Non unicode data )-----------------------------*/
int CFileInit_asf::read_varchar_c( wxString &s )
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_size ;
   /*----------------------------------------------------------------------*/
   if( m_fa.read_le_data( w_size ) != 0 ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   return( m_fi.file_read_tb_c( w_size, s ) ) ;
   /*----------------------------------------------------------------------*/
}

/*--( String: WORD (size) + Unicode data )---------------------------------*/
int CFileInit_asf::read_varchar_wc( wxString &s )
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_size ;
   /*----------------------------------------------------------------------*/
   if( m_fa.read_le_data( w_size ) != 0 ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   return( m_fi.file_read_tb_wc_le( w_size * 2, s ) ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::read_var_name( wxString &s_var )
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_size ;

   /*-----------------------------------------------------------------------+
   ! Even if the data is unicode its size is expressed in bytes. Therefore  !
   ! "asf_read_varchar_wc" is not used.                                     !
   +-----------------------------------------------------------------------*/
   if( m_fa.read_le_data( w_size ) != 0 ) { return( -1 ) ; }
   /*--( Name of the variable )--------------------------------------------*/
   if( m_fi.file_read_tb_wc_le( w_size, s_var ) != 0 )
   {  return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::read_var_val( wxString &s_val )
{
   /*----------------------------------------------------------------------*/
   str_asf_descriptor_val val ;

   /*--( Type and size )---------------------------------------------------*/
   if( read_le_data( val ) != 0 ) { return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   switch( val.w_descriptor_value_data_type )
   {
      /*-------------------------------------------------------------------*/
      case 0x0000 : // Unicode string varies
         /*----------------------------------------------------------------*/
         if( m_fi.file_read_tb_wc_le( val.w_descriptor_value_length,
                                      s_val
                                    ) != 0
           )
         {  return( -2 ) ; }
         /*----------------------------------------------------------------*/
         break ;
      /*-------------------------------------------------------------------*/
      case 0x0001 : // BYTE array varies
         /*----------------------------------------------------------------*/
         if( m_fi.file_read_tb_c( val.w_descriptor_value_length, s_val ) != 0
           )
         {  return( -3 ) ; }
         /*----------------------------------------------------------------*/
         break ;
      /*-------------------------------------------------------------------*/
      case 0x0002 : // BOOL     32
      {  /*----------------------------------------------------------------*/
         wxUint32 dw_bool ;
         /*----------------------------------------------------------------*/
         if( m_fa.read_le_data( dw_bool ) != 0 ) { return( -4 ) ; }
         /*----------------------------------------------------------------*/
         s_val = ( dw_bool == 0 ? "False" : "True" ) ;
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      case 0x0003 : // wxUint32    32
      {  /*----------------------------------------------------------------*/
         wxUint32 dw ;
         /*----------------------------------------------------------------*/
         if( m_fa.read_le_data( dw ) != 0 ) { return( -5 ) ; }
         s_val.Printf( "%d", dw ) ;
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      case 0x0004 : // QWORD    64
      {  /*----------------------------------------------------------------*/
         wxUint64 ddw ;
         /*----------------------------------------------------------------*/
         if( m_fa.read_le_data( ddw ) != 0 ) { return( -6 ) ; }
         /*----------------------------------------------------------------*/
         s_val = ( *( wxLongLong * )&ddw ).ToString() ;
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      case 0x0005 : // WORD     16
      {  /*----------------------------------------------------------------*/
         wxUint16 w ;
         /*----------------------------------------------------------------*/
         if( m_fa.read_le_data( w ) != 0 ) { return( -7 ) ; }
         s_val.Printf( "%d", w ) ;
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::image_info( bool boo_video, size_t sz_size )
{
   /*----------------------------------------------------------------------*/
   char         tb_c_val[ 512 ]     ;
   wxFileOffset fo_offset_start     ;
   size_t       sz_size_to_skip = 0 ;
   wxString     s_mime              ;
   size_t       sz_num              ;
   int          i_start             ;

   /*--( Minimum requirement )---------------------------------------------*/
   if( sz_size <= 10 ) { return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   fo_offset_start = m_fa.get_offset() ;

   /*--( Don't need all )--------------------------------------------------*/
   if( sz_size > sizeof( tb_c_val ) )
   {  sz_size_to_skip = sz_size - sizeof( tb_c_val ) ;
      sz_size         = sizeof( tb_c_val ) ;
   }
   /*----------------------------------------------------------------------*/
   if(    m_fa.read_buffer( sz_size, tb_c_val ) != 0
       || (    sz_size_to_skip > 0
            && m_fa.skip_nb_byte( sz_size_to_skip ) != 0
          )
     )
   {  return( -2 ) ; }

   /*-----------------------------------------------------------------------+
   ! The data header is 5 bytes long:                                       !
   ! - 1 byte  : Picture Type (cf ID3/APIC)                                 !
   ! - 3 bytes : Size of "pure" image (on 3 bytes, yes)                     !
   ! - 1 byte  : Text encoding (unicode ...) (cf ID3/APIC)                  !
   +-----------------------------------------------------------------------*/
   sz_num = 5 ;

   /*--( Computation of the MIME type length: coded in UNICODE !!!! )------*/
   for( i_start = sz_num ;
        sz_num < sz_size && sr::pb_to_ui16( &tb_c_val[ sz_num ] ) != 0 ;
        sz_num += 2
      )
   {  ; }

   /*----------------------------------------------------------------------*/
   if( sz_num >= sz_size ) { return( -3 ) ; }

   /*--( Load it )---------------------------------------------------------*/
   if( sz_num > ( size_t )i_start )
   {
      /*-------------------------------------------------------------------*/
      if( sr::prepare_string( &tb_c_val[ i_start ], sz_num - i_start,
                              s_mime, sr::CONV_FROM_UTF16LE
                            ) > 0
        )
      {
         /*----------------------------------------------------------------*/
         if( boo_video )
         {  m_f.val_s( COL_VIDTAG_IMG_FORMAT ) = s_mime ; }
         else
         {  m_f.val_s( COL_AUDTAG_IMG_FORMAT ) = s_mime ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( Jump over the '\0\0' )--------------------------------------------*/
   sz_num += 2 ;

   /*--( The description will always been in "unicode" ! )-----------------*/
   while( sz_num < sz_size && sr::pb_to_ui16( &tb_c_val[ sz_num ] ) != 0 )
   {  sz_num += 2 ; }
   /*----------------------------------------------------------------------*/
   sz_num += 2 ;

   /*----------------------------------------------------------------------*/
   if( sz_num >= sz_size ) { return( -4 ) ; }

   /*----------------------------------------------------------------------*/
   if( m_fi.read_embedded_image_info( fo_offset_start + sz_num,
                                      sz_size + sz_size_to_skip - sz_num,
                                      s_mime
                                    ) != 0
     )
   {  return( -5 ) ; }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_asf::asf_read()
{
   /*----------------------------------------------------------------------*/
   str_asf_file_header                  file_header                         ;
   str_asf_object_header                object_header                       ;
   str_asf_file_properties              file_prop                           ;
   str_asf_stream_properties            stream_prop                         ;
   wxFileOffset                         fo_offset_metadata_lib      = -1    ;
   str_asf_content_description          content_desc                        ;
   wxFileOffset                         fo_offset_content_desc      = -1    ;
   str_asf_extended_content_description extended_desc                       ;
   wxFileOffset                         fo_offset_extended_content_desc = -1;
   str_asf_audio_specific_properties    audio_prop                          ;
   wxFileOffset                         fo_offset_audio_prop        = -1    ;
   str_asf_video_specific_properties    video_prop                          ;
   wxFileOffset                         fo_offset_video_prop        = -1    ;
   str_asf_codec_list                   codec_list                          ;
   wxFileOffset                         fo_offset_codec_list        = -1    ;
   wxString                             s_val                               ;
   int                                  i_object_nb                         ;
   wxFileOffset                         fo_offset_object_end                ;
   int                                  i_duration                  = -1    ;
   bool                                 boo_has_video                       ;
   int                                  i_col                               ;

   /*----------------------------------------------------------------------*/
   SR_SET_0( file_prop     ) ;
   SR_SET_0( stream_prop   ) ;
   SR_SET_0( content_desc  ) ;
   SR_SET_0( extended_desc ) ;
   SR_SET_0( audio_prop    ) ;
   SR_SET_0( video_prop    ) ;
   SR_SET_0( codec_list    ) ;

   /*----------------------------------------------------------------------*/
   if( read_le_data( file_header ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   if(    file_header.object_header.guid_object_id != st_ASF_Header_Object
       || file_header.object_header.ddw_object_size < sizeof( file_header )
     )
   {  return( -2 ) ; }

   /*----------------------------------------------------------------------*/
   fo_offset_object_end = m_fa.get_offset() ;

   /*--( Go through the objects in the header )----------------------------*/
   for( i_object_nb = file_header.dw_number_of_header_objects ;
        i_object_nb > 0 ;
        --i_object_nb
      )
   {
      /*-------------------------------------------------------------------*/
      if( read_le_data( object_header ) != 0 ) { return( -3 ) ; }
      /*-------------------------------------------------------------------*/
      fo_offset_object_end += object_header.ddw_object_size ;

      /*--( General properties )-------------------------------------------*/
      if( object_header.guid_object_id == st_ASF_File_Properties_Object )
      {
         /*----------------------------------------------------------------*/
         if(   object_header.ddw_object_size
             < sizeof( object_header ) + sizeof( file_prop )
           )
         {  return( -4 ) ; }
         /*----------------------------------------------------------------*/
         if( read_le_data( file_prop ) != 0 ) { return( -5 ) ; }
         /*----------------------------------------------------------------*/
         if( (   file_prop.dw_flags
               & str_asf_file_properties::FP_BROADCAST_FLAG
             ) == 0
           )
         {  i_duration = ( int )(   file_prop.ddw_play_duration / 10000000.0
                                  - file_prop.ddw_preroll       /     1000.0
                                ) ;
         }
         /*----------------------------------------------------------------*/
      }
      else /*--( Info "fixed" tags )---------------------------------------*/
      if( object_header.guid_object_id == st_ASF_Content_Description_Object )
      {
         /*----------------------------------------------------------------*/
         if(   object_header.ddw_object_size
             < sizeof( object_header ) + sizeof( content_desc )
           )
         {  return( -6 ) ; }
         /*----------------------------------------------------------------*/
         if( read_le_data( content_desc ) != 0 ) { return( -7 ) ; }
         /*----------------------------------------------------------------*/
         fo_offset_content_desc = m_fa.get_offset() ;
         /*----------------------------------------------------------------*/
      }
      else /*--( Infos "complementary" tags )------------------------------*/
      if(    object_header.guid_object_id
          == st_ASF_Extended_Content_Description_Object
        )
      {
         /*----------------------------------------------------------------*/
         if(   object_header.ddw_object_size
             < sizeof( object_header ) + sizeof( extended_desc )
           )
         {  return( -8 ) ; }
         /*----------------------------------------------------------------*/
         if( read_le_data( extended_desc ) != 0 ) { return( -9 ) ; }
         /*----------------------------------------------------------------*/
         fo_offset_extended_content_desc = m_fa.get_offset() ;
         /*----------------------------------------------------------------*/
      }
      else
         /*-----------------------------------------------------------------+
         ! Infos always ...                                                 !
         ! This time to looking for the "WM/Picture" images bigger          !
         ! than 64Kb ... stored here due to the size of the standard tags   !
         ! which is defined on two bytes ...                                !
         +-----------------------------------------------------------------*/
      if(    object_header.guid_object_id == st_ASF_Header_Extension_Object
          && fo_offset_metadata_lib < 0
        )
      {
         /*----------------------------------------------------------------*/
         str_asf_header_extension_object header_extension ;
         str_asf_object_header           he_data          ;
         size_t                          sz_size          ;
         size_t                          sz_size_to_skip  ;

         /*----------------------------------------------------------------*/
         if(   object_header.ddw_object_size
             < sizeof( object_header ) + sizeof( header_extension )
           )
         {  return( -10 ) ; }
         /*----------------------------------------------------------------*/
         if( read_le_data( header_extension ) != 0 ) { return( -11 ) ; }
         /*----------------------------------------------------------------*/
         sz_size = header_extension.dw_header_extension_data_size ;
         /*----------------------------------------------------------------*/
         while( sz_size != 0 && fo_offset_metadata_lib < 0 )
         {
            /*--( Read the header )----------------------------------------*/
            if( read_le_data( he_data ) != 0 ) { return( -12 ) ; }
            /*-------------------------------------------------------------*/
            sz_size -= sizeof( he_data ) ;

            /*--( And the data )-------------------------------------------*/
            if( he_data.guid_object_id == st_ASF_Metadata_Library_Object )
            {  /*----------------------------------------------------------*/
               if( sz_size > sizeof( str_asf_metadata_library_object ) )
               {  fo_offset_metadata_lib = m_fa.get_offset() ; }
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
            sz_size_to_skip = he_data.ddw_object_size - sizeof( he_data ) ;
            /*-------------------------------------------------------------*/
            if( m_fa.skip_nb_byte( sz_size_to_skip ) != 0 )
            {  return( -13 ) ; }
            /*-------------------------------------------------------------*/
            sz_size -= wxMin( sz_size, sz_size_to_skip ) ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      else
      if( object_header.guid_object_id == st_ASF_Stream_Properties_Object )
      {
         /*----------------------------------------------------------------*/
         if(   object_header.ddw_object_size
             < sizeof( object_header ) + sizeof( stream_prop )
           )
         {  return( -14 ) ; }
         /*----------------------------------------------------------------*/
         if( read_le_data( stream_prop ) != 0 ) { return( -15 ) ; }
         /*----------------------------------------------------------------*/
         if( stream_prop.guid_stream_type == st_ASF_Audio_Media )
         {
            /*-------------------------------------------------------------*/
            if(   stream_prop.dw_type_specific_data_length
                < sizeof( audio_prop )
              )
            {  return( -16 ) ; }
            /*-------------------------------------------------------------*/
            fo_offset_audio_prop = m_fa.get_offset() ;
            /*-------------------------------------------------------------*/
         }
         else
         if( stream_prop.guid_stream_type == st_ASF_Video_Media )
         {
            /*-------------------------------------------------------------*/
            if(   stream_prop.dw_type_specific_data_length
                < sizeof( video_prop )
              )
            {  return( -17 ) ; }
            /*-------------------------------------------------------------*/
            fo_offset_video_prop = m_fa.get_offset() ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      else
      if( object_header.guid_object_id == st_ASF_Codec_List_Object )
      {  /*----------------------------------------------------------------*/
         if(   object_header.ddw_object_size
             < sizeof( object_header ) + sizeof( codec_list )
           )
         {  return( -18 ) ; }
         /*----------------------------------------------------------------*/
         if( read_le_data( codec_list ) != 0 ) { return( -19 ) ; }
         /*----------------------------------------------------------------*/
         fo_offset_codec_list = m_fa.get_offset() ;
         /*----------------------------------------------------------------*/
      }

      /*--( Go to the end of the current object )--------------------------*/
      if( m_fa.set_offset( fo_offset_object_end ) != 0 )
      {  return( -20 ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   !                                                                        !
   ! Now the offsets are know, the tags can be filled ...                   !
   !                                                                        !
   ! Why this system (offset before data) ? :                               !
   ! 1) to know the file type : audio and/or video                          !
   ! 2) to gather the info in the the order I want and be able to make the  !
   !    computation and columns valuation without problem.                  !
   !    I am not sure of the in order of the information in the file        !
   !                                                                        !
   +-----------------------------------------------------------------------*/
   boo_has_video = ( fo_offset_video_prop >= 0 ) ;
   /*-----------------------------------------------------------------------+
   ! "Base" information                                                     !
   +-----------------------------------------------------------------------*/
   if( fo_offset_content_desc >= 0 )
   {
      /*-------------------------------------------------------------------*/
      if( m_fa.set_offset( fo_offset_content_desc ) != 0 ) { return( -21 ) ;}

      /*--( Title )--------------------------------------------------------*/
      if( m_fi.file_read_tb_wc_le( content_desc.w_title_length, s_val ) != 0)
      {  return( -22 ) ; }
      /*-------------------------------------------------------------------*/
      if( !s_val.empty() )
      {  i_col = ( boo_has_video ? COL_VIDTAG_TITLE : COL_AUDTAG_TITLE ) ;
         m_f.val_s( i_col ) = s_val ;
      }

      /*--( Author )-------------------------------------------------------*/
      if( m_fi.file_read_tb_wc_le( content_desc.w_author_length, s_val
                                 ) != 0
        )
      {  return( -23 ) ; }
      /*-------------------------------------------------------------------*/
      if( !s_val.empty() )
      {  i_col = ( boo_has_video ? COL_VIDTAG_ARTIST : COL_AUDTAG_ARTIST ) ;
         m_f.val_s( i_col ) = s_val ;
      }

      /*--( Copyright )----------------------------------------------------*/
      if( m_fi.file_read_tb_wc_le( content_desc.w_copyright_length, s_val
                                 ) != 0
        )
      {  return( -24 ) ; }
      /*-------------------------------------------------------------------*/
      if( !s_val.empty() )
      {    i_col
         = ( boo_has_video ? COL_VIDTAG_COPYRIGHT : COL_AUDTAG_COPYRIGHT ) ;
         m_f.val_s( i_col ) = s_val ;
      }

      /*--( Description )--------------------------------------------------*/
      if( m_fi.file_read_tb_wc_le( content_desc.w_description_length, s_val
                                 ) != 0
        )
      {  return( -25 ) ; }
      /*-------------------------------------------------------------------*/
      if( !s_val.empty() )
      {  i_col = ( boo_has_video ? COL_VIDTAG_COMMENT : COL_AUDTAG_COMMENT );
         m_f.val_s( i_col ) = s_val ;
      }

      /*--( Rating : for now it is not used )------------------------------*/
      /*
      if( file_read_tb_wc_sans_bom( content_desc.w_rating_length, s_val
                                  ) != 0
        )
      {  return( -26 ) ; }
      */
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! "Metadata lib" info, an image bigger than 64Kb can be found here       !
   +-----------------------------------------------------------------------*/
   if( fo_offset_metadata_lib >= 0 )
   {
      /*-------------------------------------------------------------------*/
      str_asf_metadata_library_object        md_lib     ;
      str_asf_metadata_library_record_object md_lib_rec ;
      int                                    i_nb_rec   ;
      wxString                               s_var      ;
      /*-------------------------------------------------------------------*/
      if( m_fa.set_offset( fo_offset_metadata_lib ) != 0 ) { return( -27 ) ;}
      /*-------------------------------------------------------------------*/
      if( read_le_data( md_lib ) != 0 ) { return( -28 ) ; }

      /*--( Go through the descriptions )----------------------------------*/
      for( i_nb_rec = md_lib.w_record_count ; i_nb_rec > 0 ; --i_nb_rec )
      {
         /*----------------------------------------------------------------*/
         if( read_le_data( md_lib_rec ) != 0 ) { return( -29 ) ; }

         /*----------------------------------------------------------------*/
         if( m_fi.file_read_tb_wc_le( md_lib_rec.w_name_length, s_var ) != 0)
         {  return( -30 ) ; }

         /*--( Special treatment for the images )--------------------------*/
         if(    m_f.get_image_offset() == -1
             && s_var.CmpNoCase( "WM/Picture" ) == 0
           )
         {  /*-------------------------------------------------------------*/
            if( image_info( boo_has_video, md_lib_rec.dw_data_length ) != 0 )
            {  return( -31 ) ; }
            /*-------------------------------------------------------------*/
         }
         else
         if( m_fa.skip_nb_byte( md_lib_rec.dw_data_length ) != 0 )
         {  return( -32 ) ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! "Complement tags" information                                          !
   +-----------------------------------------------------------------------*/
   if( fo_offset_extended_content_desc >= 0 )
   {
      /*-------------------------------------------------------------------*/
      int      i_nb_desc ;
      wxString s_var     ;
      /*-------------------------------------------------------------------*/
      if( m_fa.set_offset( fo_offset_extended_content_desc ) != 0 )
      {  return( -33 ) ; }

      /*--( Go through the descriptions )----------------------------------*/
      for( i_nb_desc = extended_desc.w_descriptors_count ;
           i_nb_desc > 0 ;
           --i_nb_desc
         )
      {  /*----------------------------------------------------------------*/
         if( read_var_name( s_var ) != 0 ) { return( -34 ) ; }

         /*--( Special treatment for the images )--------------------------*/
         if(    m_f.get_image_offset() == -1
             && s_var.CmpNoCase( "WM/Picture" ) == 0
           )
         {  /*-------------------------------------------------------------*/
            str_asf_descriptor_val val ;
            /*--( Type and size )------------------------------------------*/
            if( read_le_data( val ) != 0 ) { return( -35 ) ; }
            /*-------------------------------------------------------------*/
            if( image_info( boo_has_video, val.w_descriptor_value_length
                          ) != 0
              )
            {  return( -36 ) ; }
            /*-------------------------------------------------------------*/
            continue ;
            /*-------------------------------------------------------------*/
         }

         /*----------------------------------------------------------------*/
         if( read_var_val( s_val ) != 0 ) { return( -37 ) ; }
         /*----------------------------------------------------------------*/
         i_col = st_asf_conv_tag_col( boo_has_video, s_var ) ;
         /*--( Reserve the column even if the value is empty )-------------*/
         if( m_fi.reserve_col( i_col ) && !s_val.empty() )
         {
            /*-------------------------------------------------------------*/
            m_f.val_s( i_col ) = s_val ;
            /*-------------------------------------------------------------*/
            if( i_col == COL_AUDTAG_TRACK )
            {  m_fi.init_audio_num_and_nb_track( s_val ) ; }
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! Audio info                                                             !
   +-----------------------------------------------------------------------*/
   if( fo_offset_audio_prop >= 0 )
   {
      /*-------------------------------------------------------------------*/
      if( m_fa.set_offset( fo_offset_audio_prop ) != 0 ) { return( -38 ) ; }

      /*-------------------------------------------------------------------*/
      if( read_le_data( audio_prop ) != 0 ) { return( -39 ) ; }

      /*-------------------------------------------------------------------*/
      if( boo_has_video )
      {
         /*----------------------------------------------------------------*/
         m_fi.init_video_channel( audio_prop.wfx.nChannels ) ;
         m_fi.init_video_samprate( audio_prop.wfx.nSamplesPerSec ) ;
         /*----------------------------------------------------------------*/
      }
      else
      {  /*----------------------------------------------------------------*/
         m_fi.init_audio_channel( audio_prop.wfx.nChannels ) ;
         m_fi.init_audio_samprate( audio_prop.wfx.nSamplesPerSec ) ;
         m_fi.init_audio_bitrate( TBR_CBR,
                                  file_prop.dw_maximum_bitrate / 1000
                                ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! Video info                                                             !
   +-----------------------------------------------------------------------*/
   if( fo_offset_video_prop >= 0 )
   {
      /*-------------------------------------------------------------------*/
      if( m_fa.set_offset( fo_offset_video_prop ) != 0 ) { return( -40 ) ; }

      /*-------------------------------------------------------------------*/
      if( read_le_data( video_prop ) != 0 ) { return( -41 ) ; }
      /*-------------------------------------------------------------------*/
      m_fi.init_video_x_y( video_prop.dw_encoded_image_width ,
                           video_prop.dw_encoded_image_height
                         ) ;
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! Codec infos                                                            !
   +-----------------------------------------------------------------------*/
   if( fo_offset_codec_list >= 0 )
   {
      /*-------------------------------------------------------------------*/
      int                   i_nb_desc    ;
      str_asf_codec_entries codec_entry  ;
      wxString              s_name       ;
      wxString              s_desc       ;
      wxUint16              w_size       ;
      wxString              s_audio_info ;
      wxString              s_video_info ;

      /*-------------------------------------------------------------------*/
      if( m_fa.set_offset( fo_offset_codec_list ) != 0 ) { return( -42 ) ; }

      /*--( Go through the descriptions )----------------------------------*/
      for( i_nb_desc = codec_list.dw_codec_entries_count ;
           i_nb_desc > 0 ;
           --i_nb_desc
         )
      {
         /*--( Read the type )---------------------------------------------*/
         if( read_le_data( codec_entry ) != 0 ) { return( -43 ) ; }
         /*----------------------------------------------------------------*/
         if(    read_varchar_wc( s_name ) != 0
             || read_varchar_wc( s_desc ) != 0
           )
         {  return( -44 ) ; }
         /*--( Codec specific data )---------------------------------------*/
         if(    m_fa.read_le_data( w_size ) != 0
             || m_fa.skip_nb_byte( w_size ) != 0
           )
         {  return( -45 ) ; }

         /*--( The audio info are before the video ones )------------------*/
         switch( codec_entry.w_type )
         {
            /*-------------------------------------------------------------*/
            case str_asf_codec_entries::CT_AUDIO :
               /*----------------------------------------------------------*/
               m_fi.add_codec_info( s_name, s_audio_info ) ;
               break ;
               /*----------------------------------------------------------*/
            case str_asf_codec_entries::CT_VIDEO :
               /*----------------------------------------------------------*/
               m_fi.add_codec_info( s_name, s_video_info ) ;
               break ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      if( !s_audio_info.empty() || !s_video_info.empty() )
      {    m_f.val_s( boo_has_video ? COL_VIDEO_INFO : COL_AUDIO_INFO )
         = m_fi.concat_av_codecs( s_audio_info, s_video_info ) ;
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( i_duration >= 0 )
   {
      /*-------------------------------------------------------------------*/
      if( boo_has_video )
      {  m_fi.init_video_duration( i_duration ) ; }
      else
      {  m_fi.init_audio_duration( i_duration ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   m_fi.m_s_type_det = ( boo_has_video ? "wmv" : "wma" ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_asf()
{
   /*----------------------------------------------------------------------*/
   m_s_type_det = "wma/wmv" ;
   /*----------------------------------------------------------------------*/
   int i_ret = CFileInit_asf( *this ).asf_read() ;
   /*----------------------------------------------------------------------*/
   m_f.init_info_s_track() ;
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                       End of file CFileInit_asf.cpp                       !
+==========================================================================*/
