/*====================================+=====================================+
! File CFileInit_ape.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                       Audio file: ".ape", ".apl"                          !
!                                                                           !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! They can contain APE or id3v1 tags                                        !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "CFileInit.h"
#include "CFile.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
struct str_APETagFooterStruct ;
struct str_APETag_item_header ;
struct str_ape_header_common ;
struct str_ape_descriptor ;
struct str_ape_header ;
struct str_ape_header_old ;

/*--------------------------------------------------------------------------+
! Data/Treatment used during the loading                                    !
+--------------------------------------------------------------------------*/
class CFileInit_ape : public CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_ape( CFileInit &parent ) : CFileInit_type_base( parent )
      {  ; }
      /*-------------------------------------------------------------------*/
      int read_le_data( str_APETagFooterStruct &ape_fs ) ;
      int read_le_data( str_APETag_item_header &ape_ih ) ;
      int read_le_data( str_ape_header_common &ape_hc ) ;
      int read_le_data( str_ape_descriptor &ape_de ) ;
      int read_le_data( str_ape_header &ape_he ) ;
      int read_le_data( str_ape_header_old &ape_ho ) ;
      int apetag_read_var_val( wxString &s_var, wxString &s_val ) ;
      int init_apetag( wxFileOffset &fo_offset_end ) ;
      int ape_read( wxFileOffset fo_offset_end ) ;

   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*-------------------------------------------------------------------------*/
struct str_APETagFooterStruct
{
   /*----------------------------------------------------------------------*/
   char     tb_c_id[ 8 ]          ; // should equal 'APETAGEX'
   wxUint32 dw_version            ; // 1000 = version 1.0, 2000 = version 2.0
   wxUint32 dw_length             ; // tag size, incl footer, excl header
   wxUint32 dw_tagcount           ; // number of fields in the tag
   wxUint32 ___dw_flags           ; // tag flags (none currently defined)
   wxUint8  ___tb_b_reserved[ 8 ] ; // reserved for later use
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_APETag_item_header
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_value_size ;
   wxUint32 dw_flags      ; // 31: 1=tag contains header
                            // 30: 1=tag contains footer
                            // 29: 0=footer, 1=header
                            // 28...3: 0
                            // 2-1: 00 text UTF-8
                            //      01 binary
                            //      10 locator external info
                            //      11 reserved
                            // 0 : 1=read only
   /*----------------------------------------------------------------------*/
   // Follows tag name ending with a '\0'
   // and the value of dw_value_size bytes
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_ape_header_common
{
   /*----------------------------------------------------------------------*/
   char     tb_c_id[ 4 ] ; // should be equal to 'MAC '
   wxUint16 w_nVersion   ; // version number * 1000 (3.81 = 3810)
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_ape_descriptor
{
   /*----------------------------------------------------------------------*/
   wxUint16 ___w_dummy                   ; // had to add it ... strange ...
   wxUint32 dw_nDescriptorBytes          ; // number of descriptor bytes
   wxUint32 ___dw_nHeaderBytes           ; // nb of header APE_HEADER bytes
   wxUint32 ___dw_nSeekTableBytes        ; // nb of bytes of the seek table
   wxUint32 ___dw_nHeaderDataBytes       ;
   wxUint32 ___dw_nAPEFrameDataBytes     ;
   wxUint32 ___dw_nAPEFrameDataBytesHigh ;
   wxUint32 ___dw_nTerminatingDataBytes  ;
   wxUint8  ___tb_b_cFileMD5[ 16 ]       ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_ape_header
{
   /*----------------------------------------------------------------------*/
   wxUint16 ___w_nCompressionLevel ;
   wxUint16 ___w_nFormatFlags      ;
   wxUint32 dw_nBlocksPerFrame     ; // number of audio blocks in one frame
   wxUint32 dw_nFinalFrameBlocks   ; // number of audio blocks in final frame
   wxUint32 dw_nTotalFrames        ; // total number of frames
   wxUint16 ___w_nBitsPerSample    ; // bits per sample (typically 16)
   wxUint16 w_nChannels            ; // number of channels (1 or 2)
   wxUint32 dw_nSampleRate         ; // sample rate (typically 44100)
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_ape_header_old
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_nCompressionLevel     ;
   wxUint16 ___w_nFormatFlags       ;
   wxUint16 w_nChannels             ; // number of channels (1 or 2)
   wxUint32 dw_nSampleRate          ; // sample rate (typically 44100)
   wxUint32 ___dw_nHeaderBytes      ; // bytes after MAC header in WAV header
   wxUint32 ___dw_nTerminatingBytes ; // bytes after that raw data
   wxUint32 dw_nTotalFrames         ; // number of frames in the file
   wxUint32 dw_nFinalFrameBlocks    ; // number of samples in the final frame
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )

/*-------------------------------------------------------------------------*/
int CFileInit_ape::read_le_data( str_APETagFooterStruct &ape_fs )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( ape_fs ), &ape_fs ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   ape_fs.dw_version  = wxUINT32_SWAP_ALWAYS( ape_fs.dw_version  ) ;
   ape_fs.dw_length   = wxUINT32_SWAP_ALWAYS( ape_fs.dw_length   ) ;
   ape_fs.dw_tagcount = wxUINT32_SWAP_ALWAYS( ape_fs.dw_tagcount ) ;
   ape_fs.dw_flags    = wxUINT32_SWAP_ALWAYS( ape_fs.dw_flags    ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ape::read_le_data( str_APETag_item_header &ape_ih )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( ape_ih ), &ape_ih ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   ape_ih.dw_value_size = wxUINT32_SWAP_ALWAYS( ape_ih.dw_version ) ;
   ape_ih.dw_flags      = wxUINT32_SWAP_ALWAYS( ape_ih.dw_flags   ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ape::read_le_data( str_ape_header_common &ape_hc )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( ape_hc ), &ape_hc ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
   ape_hc.w_nVersion = wxUINT32_SWAP_ALWAYS( ape_hc.w_nVersion ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ape::read_le_data( str_ape_descriptor &ape_de )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( ape_de ), &ape_de ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
     ape_de.dw_nDescriptorBytes
   = wxUINT32_SWAP_ALWAYS( ape_de.dw_nDescriptorBytes ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ape::read_le_data( str_ape_header &ape_he )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( ape_he ), &ape_he ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
     ape_he.dw_nBlocksPerFrame
   = wxUINT32_SWAP_ALWAYS( ape_he.dw_nBlocksPerFrame ) ;
     ape_he.dw_nFinalFrameBlocks
   = wxUINT32_SWAP_ALWAYS( ape_he.dw_nFinalFrameBlocks ) ;
   ape_he.dw_nTotalFrames = wxUINT32_SWAP_ALWAYS( ape_he.dw_nTotalFrames ) ;
   ape_he.w_nChannels     = wxUINT16_SWAP_ALWAYS( ape_he.w_nChannels ) ;
   ape_he.dw_nSampleRate  = wxUINT32_SWAP_ALWAYS( ape_he.dw_nSampleRate ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ape::read_le_data( str_ape_header_old &ape_ho )
{
   /*----------------------------------------------------------------------*/
   if( m_fa.read_buffer( sizeof( ape_ho ), &ape_ho ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
#if wxBYTE_ORDER == wxBIG_ENDIAN
     ape_ho.w_nCompressionLevel
   = wxUINT16_SWAP_ALWAYS( ape_ho.w_nCompressionLevel ) ;
   ape_ho.w_nChannels    = wxUINT16_SWAP_ALWAYS( ape_ho.w_nChannels ) ;
   ape_ho.dw_nSampleRate = wxUINT32_SWAP_ALWAYS( ape_ho.dw_nSampleRate ) ;
     ape_ho.dw_nTotalFrames
   = wxUINT32_SWAP_ALWAYS( ape_ho.dw_nTotalFrames ) ;
     ape_ho.dw_nFinalFrameBlocks
   = wxUINT32_SWAP_ALWAYS( ape_ho.dw_nFinalFrameBlocks ) ;
#endif
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
static const char * const st_cp_c_apetagex = "APETAGEX" ;

/*-------------------------------------------------------------------------*/
static int st_apetag_conv_tag_col( const wxString &s_var )
{
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "Title" ) == 0 )
   {  return( COL_AUDTAG_TITLE ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "Album" ) == 0 )
   {  return( COL_AUDTAG_ALBUM ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "Artist" ) == 0 )
   {  return( COL_AUDTAG_ARTIST ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "Track" ) == 0 )
   {  return( COL_AUDTAG_TRACK ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "Part" ) == 0 ) // EasyTAG
   {  return( COL_AUDTAG_DISK ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "Copyright" ) == 0 )
   {  return( COL_AUDTAG_COPYRIGHT ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "Genre" ) == 0 )
   { return( COL_AUDTAG_GENRE ) ; }
   /*----------------------------------------------------------------------*/
   if(    s_var.CmpNoCase( "Publisher URL" ) == 0
       || s_var.CmpNoCase( "File URL"      ) == 0
       || s_var.CmpNoCase( "Copyright URL" ) == 0
       || s_var.CmpNoCase( "URL"           ) == 0 // TGF
       || s_var.CmpNoCase( "Related"       ) == 0 // EasyTAG
     )
   { return( COL_AUDTAG_URL ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "Comment" ) == 0 )
   {  return( COL_AUDTAG_COMMENT ) ; }
   /*----------------------------------------------------------------------*/
   if(    s_var.CmpNoCase( "Record Date" ) == 0
       || s_var.CmpNoCase( "Year"        ) == 0
     )
   {  return( COL_AUDTAG_YEAR ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "Composer" ) == 0 )
   {  return( COL_AUDTAG_COMPOS ) ; }
   /*----------------------------------------------------------------------*/
   if( s_var.CmpNoCase( "ORIGINALARTIST" ) == 0 )
   {  return( COL_AUDTAG_ORG_ART ) ; }
   /*----------------------------------------------------------------------*/
   if(    s_var.CmpNoCase( "ENCODEDBY" ) == 0
       || s_var.CmpNoCase( "Encoded By" ) == 0 // EasyTAG
     )
   {  return( COL_AUDTAG_ENCOD_BY ) ; }
   /*----------------------------------------------------------------------*/
   return( COL_NB ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ape::apetag_read_var_val( wxString &s_var, wxString &s_val )
{
   /*----------------------------------------------------------------------*/
   str_APETag_item_header item_header         ;
   char                   tb_c_var[ 32 ]      ;
   size_t                 sz_read             ;
   char                   *p_c_var_end = NULL ;
   int                    i_ret               ;

   /*----------------------------------------------------------------------*/
   s_var.clear() ;
   s_val.clear() ;

   /*----------------------------------------------------------------------*/
   if( read_le_data( item_header ) != 0 )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   sz_read = sizeof( tb_c_var ) ;
   /*--( The variable name )-----------------------------------------------*/
   while(    ( i_ret = m_fa.read_buffer_max( sz_read, tb_c_var ) ) == 0
          && ( p_c_var_end = ( char * )memchr( tb_c_var, '\0', sz_read )
             ) == NULL
          && sz_read == sizeof( tb_c_var )
        )
   {  s_var += wxString::From8BitData( tb_c_var, sz_read ) ; }

   /*----------------------------------------------------------------------*/
   if( i_ret   != 0                  ) { return( -1 ) ; }
   if( sz_read == 0 && s_var.empty() ) { return( -2 ) ; }

   /*--( Impossible case ... but: "No '\0' at the end of file ?" )---------*/
   if( p_c_var_end == NULL ) { p_c_var_end = tb_c_var + sz_read ; }

   /*----------------------------------------------------------------------*/
   s_var += wxString::From8BitData( tb_c_var, p_c_var_end - tb_c_var ) ;

   /*--( Go to the beginning of the next line )----------------------------*/
   if( m_fa.skip_nb_byte( - ( int )sz_read + ( p_c_var_end - tb_c_var ) + 1
                        ) != 0
     )
   {  return( -3 ) ; }

   /*--( If it's text the associated value is read )-----------------------*/
   if( ( item_header.dw_flags & 0x06 ) == 0x00 )
   {
      /*-------------------------------------------------------------------*/
      if( m_fi.file_read_tb_c( item_header.dw_value_size, s_val ) != 0 )
      {  return( -4 ) ; }
      /*-------------------------------------------------------------------*/
   }
   else /*-----------------------------------------------------------------*/
   if( m_fa.skip_nb_byte( item_header.dw_value_size ) != 0 )
   {  return( -5 ) ; }

   /*--( Skip an eventual footer )-----------------------------------------*/
   if(    ( item_header.dw_flags & 0x8000 ) == 0x8000
       && m_fa.skip_nb_byte( sizeof( item_header ) ) != 0
     )
   {  return( -6 ) ; }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ape::init_apetag( wxFileOffset &fo_offset_end )
{
   /*----------------------------------------------------------------------*/
   str_APETagFooterStruct ape_hf   ;
   int                    i_nb_tag ;
   wxString               s_var    ;
   wxString               s_val    ;
   int                    i_col    ;

   /*----------------------------------------------------------------------*/
   fo_offset_end = m_f.get_size().ToULong() ;

   /*--( Read the footer )-------------------------------------------------*/
   if(    m_fa.set_offset_from_end( sizeof( ape_hf ) ) != 0
       || read_le_data( ape_hf ) != 0
     )
   {  return( -1 ) ; }

   /*--( Ok ? )------------------------------------------------------------*/
   if( memcmp( ape_hf.tb_c_id, st_cp_c_apetagex, sizeof( ape_hf.tb_c_id )
             ) != 0
     )
   {  return( -2 ) ; }

   /*--( The tag beginning : dw_length takes footer into account )---------*/
   fo_offset_end -= ape_hf.dw_length ;

   /*--( Go to the beginning of the tag data )-----------------------------*/
   if( m_fa.set_offset( fo_offset_end ) != 0 )
   {  return( -3 ) ; }

   /*--( UTF8 for APE2, as it is for the others )--------------------------*/
   if( ape_hf.dw_version >= 2000 ) { m_fi.set_conv_from_utf8() ; }

   /*----------------------------------------------------------------------*/
   for( i_nb_tag = ape_hf.dw_tagcount ; i_nb_tag > 0 ; --i_nb_tag )
   {
      /*-------------------------------------------------------------------*/
      if( apetag_read_var_val( s_var, s_val ) != 0 )
      {  return( -4 ) ; }

      /*-------------------------------------------------------------------*/
      i_col = st_apetag_conv_tag_col( s_var ) ;
      /*--( The column is reserved even if it is empty )-------------------*/
      if( m_fi.reserve_col( i_col ) && !s_val.empty() )
      {
         /*----------------------------------------------------------------*/
         m_f.val_s( i_col ) = s_val ;
         /*----------------------------------------------------------------*/
         if( i_col == COL_AUDTAG_TRACK )
         {  m_fi.init_audio_num_and_nb_track( s_val ) ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( Then comes the tag header )---------------------------------------*/
   if(    m_fa.set_offset( fo_offset_end - sizeof( ape_hf ) ) != 0
       || read_le_data( ape_hf ) != 0
     )
   {  return( -5 ) ; }

   /*--( Data ok ? )-------------------------------------------------------*/
   if( memcmp( ape_hf.tb_c_id, st_cp_c_apetagex, sizeof( ape_hf.tb_c_id )
             ) == 0
     )
   {  fo_offset_end -= sizeof( ape_hf ) ; }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit_ape::ape_read( wxFileOffset fo_offset_end )
{
   /*----------------------------------------------------------------------*/
   str_ape_header_common header_common     ;
   int                   i_nb_total_blocks ;
   /*----------------------------------------------------------------------*/
   int                   i_samprate    = 0 ;
   int                   i_bitrate     = 0 ;
   int                   i_duration_ms = 0 ;
   int                   i_channel     = 0 ;

   /*----------------------------------------------------------------------*/
   if( m_fa.set_offset( 0 ) != 0 || read_le_data( header_common ) != 0 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   if( memcmp( header_common.tb_c_id, "MAC ", 4 ) != 0 )
   {  return( -2 ) ; }

   /*--( Following data depends on the version )---------------------------*/
   if( header_common.w_nVersion > 3980 )
   {
      /*-------------------------------------------------------------------*/
      str_ape_descriptor descriptor ;
      str_ape_header     header     ;

      /*-------------------------------------------------------------------*/
      if( read_le_data( descriptor ) != 0 )
      {  return( -3 ) ; }
      /*--( Descriptor can be larger than what it is ... )-----------------*/
      if(      descriptor.dw_nDescriptorBytes
             > sizeof( header_common ) + sizeof( descriptor )
          && m_fa.skip_nb_byte(   descriptor.dw_nDescriptorBytes
                                - sizeof( header_common )
                                - sizeof( descriptor )
                              ) != 0
        )
      {  return( -4 ) ; }
      /*-------------------------------------------------------------------*/
      if( read_le_data( header ) != 0 )
      {  return( -5 ) ; }

      /*-------------------------------------------------------------------*/
      i_channel  = header.w_nChannels    ;
      i_samprate = header.dw_nSampleRate ;
      /*-------------------------------------------------------------------*/
      i_nb_total_blocks = header.dw_nTotalFrames ;
      if( i_nb_total_blocks != 0 )
      {    i_nb_total_blocks
         =   ( i_nb_total_blocks - 1 ) * header.dw_nBlocksPerFrame
           + header.dw_nFinalFrameBlocks ;
      }
      /*-------------------------------------------------------------------*/
   }
   else
   {
      /*-------------------------------------------------------------------*/
      str_ape_header_old header_old          ;
      wxUint32           dw_blocks_per_frame ;

      /*-------------------------------------------------------------------*/
      if( read_le_data( header_old ) != 0 )
      {  return( -6 ) ; }
      /*-------------------------------------------------------------------*/
      i_channel  = header_old.w_nChannels    ;
      i_samprate = header_old.dw_nSampleRate ;

      /*-------------------------------------------------------------------*/
      if( header_common.w_nVersion >= 3950 )
      {  dw_blocks_per_frame = 73728 * 4 ; }
      else
      if(    header_common.w_nVersion >= 3900
          || (    header_common.w_nVersion >= 3800
               && header_old.w_nCompressionLevel == 4000
             )
        )
      {  dw_blocks_per_frame = 73728 ; }
      else
      {  dw_blocks_per_frame = 9216 ; }
      /*-------------------------------------------------------------------*/
      i_nb_total_blocks = header_old.dw_nTotalFrames ;
      if( i_nb_total_blocks != 0 )
      {    i_nb_total_blocks
         =   ( i_nb_total_blocks - 1 ) * dw_blocks_per_frame
           + header_old.dw_nFinalFrameBlocks ;
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( i_samprate != 0 )
   {  i_duration_ms = sr::round_0(   ( double )i_nb_total_blocks * 1000.0
                                   / ( double )i_samprate
                                 ) ;
   }
   /*--( Bitrate is average )----------------------------------------------*/
   if( i_duration_ms > 0 )
   {  i_bitrate = sr::round_0(   ( double )fo_offset_end * 8.0
                               / ( double )i_duration_ms
                             ) ;
   }
   /*----------------------------------------------------------------------*/
   m_f.val_s( COL_AUDIO_INFO ).Printf( "Monkey's Audio %d.%d",
                                       header_common.w_nVersion / 1000,
                                       header_common.w_nVersion % 1000
                                     ) ;

   /*----------------------------------------------------------------------*/
   if( i_bitrate > 0 )
   {  m_fi.init_audio_bitrate( TBR_ABR, i_bitrate ) ; }
   if( i_duration_ms >= 0 )
   {  m_fi.init_audio_duration( i_duration_ms / 1000 ) ; }
   if( i_samprate > 0 )
   {  m_fi.init_audio_samprate( i_samprate ) ; }
   if( i_channel > 0 )
   {  m_fi.init_audio_channel( i_channel ) ; }

   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CFileInit::init_ape()
{
   /*----------------------------------------------------------------------*/
   CFileInit_ape ape( *this ) ;
   wxFileOffset  fo_offset_end ;

   /*----------------------------------------------------------------------*/
   m_s_type_det = "ape" ;

   /*--( If no APE tag then try the id3v1 one )----------------------------*/
   if( ape.init_apetag( fo_offset_end ) != 0 )
   {  mp3_read_id3v1( fo_offset_end ) ; }

   /*----------------------------------------------------------------------*/
   int i_ret = ape.ape_read( fo_offset_end ) ;
   /*----------------------------------------------------------------------*/
   m_f.init_info_s_track() ;
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
!                                                                           !
! This function is directly used by other init modules ...                  !
!                                                                           !
+--------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
int CFileInit::ape_init_apetag( wxFileOffset &fo_offset_end )
{  return( CFileInit_ape( *this ).init_apetag( fo_offset_end ) ) ; }

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                      End of file CFileInit_ape.cpp                        !
+==========================================================================*/
