/*====================================+=====================================+
! File CFileInit_Base.cpp             ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "CApp.h"
#include "CFile.h"
#include "CFileInit.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
CFileInit::CFileInit( CFile &file_treated ) : m_f( file_treated )
{  ; }

/*-------------------------------------------------------------------------*/
CFileInit::~CFileInit()
{  ; }

/*-------------------------------------------------------------------------*/
void CFileInit::init_mini_name( const wxString &s_name )
{
   /*----------------------------------------------------------------------*/
   m_f.set_name( s_name ) ;
     m_f.val_s( COL_BASE_DOS_NAME )
   = m_f.get_name().GetShortPath().AfterLast( wxFileName::GetPathSeparator()
                                            ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_full( const wxString &s_path,
                           const wxString &s_name,
                           bool boo_is_dir
                         )
{
   /*----------------------------------------------------------------------*/
   wxDateTime dt_acc ;
   wxDateTime dt_mod ;
   wxDateTime dt_cre ;

   /*----------------------------------------------------------------------*/
   m_f.reset( boo_is_dir ) ;
   set_conv_from_8bit() ;
   reset_col_set() ;

   /*----------------------------------------------------------------------*/
   init_mini_name( s_name ) ;

   /*-----------------------------------------------------------------------+
   ! Timestamps                                                             !
   ! Under MSW, for directories, "wxFileName::GetTimes" checks              !
   ! "wxFileName::IsDir" result.                                            !
   ! Therefore a "conversion" to standard file is done in this case.        !
   +-----------------------------------------------------------------------*/
#ifdef __WXMSW__
   if( m_f.is_dir() )
   {  wxFileName::DirName( m_f.get_name().GetFullPath()
                         ).GetTimes( &dt_acc, &dt_mod, &dt_cre ) ;
   }
   else
#endif // __WXMSW__
   {  m_f.get_name().GetTimes( &dt_acc, &dt_mod, &dt_cre ) ; }

   /*----------------------------------------------------------------------*/
   if( dt_cre.IsValid() ) { init_date( COL_BASE_CRE_DATE, dt_cre ) ; }
   if( dt_acc.IsValid() ) { init_date( COL_BASE_ACC_DATE, dt_acc ) ; }
   if( dt_mod.IsValid() ) { init_date( COL_BASE_MOD_DATE, dt_mod ) ; }

   /*--( For directories, the total may have to be computed )--------------*/
   if( !m_f.is_dir() )
   {  m_f.set_size( m_f.get_name().GetSize() ) ; }
   else
   if( wxGetApp().M_boo_dir_size_fl.get() )
   {  m_f.set_size( wxDir::GetTotalSize( m_f.get_full_path() ) ) ; }
   else
   {  m_f.set_size( wxInvalidSize ) ; }

   /*--( The size might not be "readable" cf "wxFileName::GetSize()" )-----*/
   if( m_f.get_size() == wxInvalidSize )
   {  m_f.set_size( 0 ) ; }
   else
   {  m_f.init_s_size() ; }

   /*--( The dir/file attributes ... not very nice but it exists )---------*/
   init_attr() ;

   /*--( File description )------------------------------------------------*/
   if( m_f.is_dir() )
   {  m_f.val_s( COL_BASE_TYPE ) = _( "Directory" ) ; }
   else
   if( !m_f.get_name().HasExt() )
   {  m_f.val_s( COL_BASE_TYPE ) = _( "Unknown" ) ; }
   else
   {  m_f.val_s( COL_BASE_TYPE ) = get_type_desc() ; }

   /*-----------------------------------------------------------------------+
   ! Meta-data extraction if possible and wanted (option)                   !
   +-----------------------------------------------------------------------*/
   if(    !m_f.is_dir()
       && wxGetApp().M_boo_extr_meta.get()
       && m_fa.open_to_read( m_f.get_full_path() ) == 0
     )
   {
      /*--( To detect if any data has been found )-------------------------*/
      const size_t  co_sz_as_val_size_before = m_f.get_map_s_val().size() ;
      t_p_init_func p_func_ext = NULL ;
      t_p_init_func p_func_det = NULL ;
      int           i_ret_init = 0 ;

      /*--( First : init based on extension )------------------------------*/
      p_func_ext = get_init_func_ext() ;
      /*--------------------------------------------------------------------+
      ! The detection will be started for an unknown extension or if the    !
      ! init failed with no meta-data extracted and if the file detection   !
      ! has been activated in the options                                   !
      +--------------------------------------------------------------------*/
      if(    (    p_func_ext == NULL
               || (    ( i_ret_init = ( this->*p_func_ext )() ) != 0
                    && m_f.get_map_s_val().size() == co_sz_as_val_size_before
                  )
             )
          && wxGetApp().M_boo_file_type_detect.get()
        )
      {
         /*--( Then : init based on file content )-------------------------*/
         p_func_det = get_init_func_detect() ;
         /*-----------------------------------------------------------------+
         ! Useless to call the same function as the extension one           !
         ! The rewind of the file pointer is done because the detection has !
         ! read some data                                                   !
         +-----------------------------------------------------------------*/
         if(    p_func_det != NULL
             && p_func_det != p_func_ext
             && m_fa.set_offset( 0 ) == 0
           )
         {  i_ret_init = ( this->*p_func_det )() ; }
         /*----------------------------------------------------------------*/
      }

      /*--( Managed file ? )-----------------------------------------------*/
      if(    ( p_func_ext == NULL && p_func_det == NULL )
          || (    i_ret_init != 0
               && m_f.get_map_s_val().size() == co_sz_as_val_size_before
             )
        )
      {  m_f.val_s( COL_BASE_TYPE_DET ) = _( "unknown" ) ; }
      else /*--( All went fine )-------------------------------------------*/
      if( i_ret_init == 0 )
      {  m_f.val_s( COL_BASE_TYPE_DET ) = m_s_type_det ; }
      else /*--( init failed but some metadata have been extracted )-------*/
      if( m_f.get_map_s_val().size() != co_sz_as_val_size_before )
      {  m_f.val_s( COL_BASE_TYPE_DET ).Printf( _( "%s malformed" ),
                                                m_s_type_det
                                              ) ;
      }

      /*-------------------------------------------------------------------*/
      m_fa.close() ;
      /*-------------------------------------------------------------------*/
   }

   /*--( If given as parameter, the now useless path is removed )----------*/
   if( !s_path.empty() )
   {  m_f.get_name().MakeRelativeTo( s_path ) ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_date( int i_col, const wxDateTime &dt )
{
   /*----------------------------------------------------------------------*/
   m_f.val_dt( i_col ) = dt ;
   m_f.val_s( i_col )  = dt.Format( wxGetApp().get_disp_fmt_date() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_attr()
{
   /*----------------------------------------------------------------------*/
   if( !reserve_col( COL_BASE_ATTR ) ) { return ; }

   /*----------------------------------------------------------------------*/
   wxString s_attr     ;
   bool     boo_hidden ;

   /*----------------------------------------------------------------------*/
   boo_hidden = sr::is_file_attr_hidden( m_f.get_full_path() ) ;
   /*----------------------------------------------------------------------*/
   if( m_f.is_dir() )
   {  /*-------------------------------------------------------------------*/
      s_attr =   wxString( 'd' )
               + ( boo_hidden ? 'h' : '-' )
               + ( m_f.get_name().IsDirReadable() ? 'r' : '-' )
               + ( m_f.get_name().IsDirWritable() ? 'w' : '-' )
               + '-' ;
      /*-------------------------------------------------------------------*/
   }
   else
   {  /*-------------------------------------------------------------------*/
      s_attr =   wxString( '-' )
               + ( boo_hidden ? 'h' : '-' )
               + ( m_f.get_name().IsFileReadable()   ? 'r' : '-' )
               + ( m_f.get_name().IsFileWritable()   ? 'w' : '-' )
               + ( m_f.get_name().IsFileExecutable() ? 'x' : '-' ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   m_f.val_s( COL_BASE_ATTR ) = s_attr ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_img_x_y( int i_x, int i_y )
{
   /*----------------------------------------------------------------------*/
   if( reserve_col( COL_IMG_X ) )
   {  /*-------------------------------------------------------------------*/
      m_f.val_ll( COL_IMG_X ) = i_x ;
      m_f.val_s( COL_IMG_X ) << i_x ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( reserve_col( COL_IMG_Y ) )
   {  /*-------------------------------------------------------------------*/
      m_f.val_ll( COL_IMG_Y ) = i_y ;
      m_f.val_s( COL_IMG_Y ) << i_y ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_img_bpp( int i_bpp )
{
   /*----------------------------------------------------------------------*/
   if( reserve_col( COL_IMG_BPP ) )
   {  /*-------------------------------------------------------------------*/
      m_f.val_ll( COL_IMG_BPP ) = i_bpp ;
      m_f.val_s( COL_IMG_BPP ) << i_bpp ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_col_mp3_channel( int i_col, int i_channel )
{
   /*----------------------------------------------------------------------*/
   if( !reserve_col( i_col ) ) { return ; }
   /*----------------------------------------------------------------------*/
   switch( i_channel )
   {
      /*-------------------------------------------------------------------*/
      case 0 : m_f.val_s( i_col ) = "stereo"                ; break ;
      case 1 : m_f.val_s( i_col ) = "stereo (joint)"        ; break ;
      case 2 : m_f.val_s( i_col ) = "stereo (dual channel)" ; break ;
      case 3 : m_f.val_s( i_col ) = "mono"                  ; break ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_audio_channel( int i_channel )
{
   /*----------------------------------------------------------------------*/
   if( !reserve_col( COL_AUDIO_CHANNEL ) ) { return ; }
   /*----------------------------------------------------------------------*/
   switch( i_channel )
   {
      /*-------------------------------------------------------------------*/
      case 1 : m_f.val_s( COL_AUDIO_CHANNEL ) = "mono"   ; break ;
      case 2 : m_f.val_s( COL_AUDIO_CHANNEL ) = "stereo" ; break ;
      default :
         m_f.val_s( COL_AUDIO_CHANNEL ).Printf( "%d channels", i_channel ) ;
         break ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_audio_samprate( int i_samprate )
{
   /*----------------------------------------------------------------------*/
   if( !reserve_col( COL_AUDIO_SAMPRATE ) ) { return ; }
   /*----------------------------------------------------------------------*/
   m_f.val_ll( COL_AUDIO_SAMPRATE ) = i_samprate ;
   m_f.val_s( COL_AUDIO_SAMPRATE ) << i_samprate ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_audio_duration( int i_nb_sec )
{
   /*----------------------------------------------------------------------*/
   if( !reserve_col( COL_AUDIO_DURATION ) ) { return ; }
   /*----------------------------------------------------------------------*/
   m_f.set_duration( i_nb_sec ) ;
   /*----------------------------------------------------------------------*/
   m_f.val_ll( COL_AUDIO_DURATION ) = i_nb_sec ;
   m_f.val_s( COL_AUDIO_DURATION ) = sr::duration_to_string( i_nb_sec ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! In many cases the track information is defined as a string formatted:     !
! num / nb                                                                  !
! This function is in charge of splitting this string and fill the track    !
! id and track number with the right values.                                !
+--------------------------------------------------------------------------*/
void CFileInit::init_audio_num_and_nb_track( const wxString &s_track )
{
   /*----------------------------------------------------------------------*/
   CFile::t_s_val_cit it_s_val ;
   int                i_num    ;
   sr::wxString_cit   it       ;

   /*----------------------------------------------------------------------*/
   i_num = 0 ;
   it = sr::conv_str_to_int( s_track.begin(), s_track.end(), i_num ) ;
   /*----------------------------------------------------------------------*/
   if(    ( it == s_track.end() || *it == '/' )
       && reserve_col( COL_AUDTAG_TRACK_NUM )
     )
   {  /*-------------------------------------------------------------------*/
      m_f.val_ll( COL_AUDTAG_TRACK_NUM ) = i_num ;
      /*--( Is there a total track number that can be extracted ? )--------*/
      if( it != s_track.end() )
      {  /*----------------------------------------------------------------*/
         i_num = 0 ;
         sr::conv_str_to_int( ++it, s_track.end(), i_num ) ;
         if( reserve_col( COL_AUDTAG_TRACK_NB ) )
         {  m_f.val_ll( COL_AUDTAG_TRACK_NB ) = i_num ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_audio_bitrate( e_bitrate_type bitrate_type,
                                    int i_bitrate
                                  )
{
   /*----------------------------------------------------------------------*/
   if( !reserve_col( COL_AUDIO_BITRATE ) ) { return ; }
   /*----------------------------------------------------------------------*/
   m_f.val_ll( COL_AUDIO_BITRATE ) = i_bitrate ;
   switch( bitrate_type )
   {
      /*-------------------------------------------------------------------*/
      case TBR_VBR : case TBR_VBR_XING : case TBR_VBR_FH :
          m_f.val_s( COL_AUDIO_BITRATE ).Printf( "VBR%d", i_bitrate ) ;
          break ;
      /*-------------------------------------------------------------------*/
      default :
          m_f.val_s( COL_AUDIO_BITRATE ) << i_bitrate ;
          break ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_video_x_y( int i_x, int i_y )
{
   /*----------------------------------------------------------------------*/
   if( reserve_col( COL_VIDEO_X ) )
   {  /*-------------------------------------------------------------------*/
      m_f.val_ll( COL_VIDEO_X ) = i_x ;
      m_f.val_s( COL_VIDEO_X ) << i_x ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( reserve_col( COL_VIDEO_Y ) )
   {  /*-------------------------------------------------------------------*/
      m_f.val_ll( COL_VIDEO_Y ) = i_y ;
      m_f.val_s( COL_VIDEO_Y ) << i_y ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_video_fps( double do_fps )
{
   /*----------------------------------------------------------------------*/
   if( !reserve_col( COL_VIDEO_FPS ) ) { return ; }
   /*----------------------------------------------------------------------*/
   m_f.val_do( COL_VIDEO_FPS ) = do_fps ;
   m_f.val_s( COL_VIDEO_FPS ).Printf( "%.3lf", sr::round_001( do_fps ) ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_video_duration( int i_nb_sec )
{
   /*----------------------------------------------------------------------*/
   if( !reserve_col( COL_VIDEO_DURATION ) ) { return ; }
   /*----------------------------------------------------------------------*/
   m_f.set_duration( i_nb_sec ) ;
   /*----------------------------------------------------------------------*/
   m_f.val_ll( COL_VIDEO_DURATION ) = i_nb_sec ;
   m_f.val_s( COL_VIDEO_DURATION ) = sr::duration_to_string( i_nb_sec ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_video_channel( int i_channel )
{
   /*----------------------------------------------------------------------*/
   if( !reserve_col( COL_VIDEO_CHANNEL ) ) { return ; }
   /*----------------------------------------------------------------------*/
   switch( i_channel )
   {
      /*-------------------------------------------------------------------*/
      case 1 : m_f.val_s( COL_VIDEO_CHANNEL ) = "mono" ; break ;
      case 2 : m_f.val_s( COL_VIDEO_CHANNEL ) = "stereo" ; break ;
      default :
         m_f.val_s( COL_VIDEO_CHANNEL ).Printf( "%d channels", i_channel ) ;
         break ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_video_samprate( int i_samprate )
{
   /*----------------------------------------------------------------------*/
   if( !reserve_col( COL_VIDEO_SAMPRATE ) ) { return ; }
   /*----------------------------------------------------------------------*/
   m_f.val_ll( COL_VIDEO_SAMPRATE ) = i_samprate ;
   m_f.val_s( COL_VIDEO_SAMPRATE ) << i_samprate ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_exif_shutterspeedvalue( double do_val )
{
   /*-----------------------------------------------------------------------+
   ! The reservation has already been done in Exif or not necessary in the  !
   ! other calling functions ...                                            !
   +-----------------------------------------------------------------------*/
   //if( !reserve_col( COL_EXIF_SHUTTERSPEEDVALUE ) ) { return ; }
   /*--( INIFINITY is defined in "math.h" )--------------------------------*/
   if( do_val <= 0 || do_val == std::numeric_limits< double >::infinity() )
   { return ; }

   /*----------------------------------------------------------------------*/
   int i_res ;
   /*----------------------------------------------------------------------*/
   m_f.val_do( COL_EXIF_SHUTTERSPEEDVALUE ) = do_val ;
   i_res = sr::round_0( 1 / do_val ) ;
   /*----------------------------------------------------------------------*/
   if( i_res > 1 )
   {  m_f.val_s( COL_EXIF_SHUTTERSPEEDVALUE ).Printf( "1/%ds", i_res ) ; }
   else
   {  m_f.val_s( COL_EXIF_SHUTTERSPEEDVALUE ).Printf( "%0.2lfs",
                                                      sr::round_01( do_val )
                                                    ) ;
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_exif_focallength( double do_val )
{
   /*----------------------------------------------------------------------*/
   if( do_val == std::numeric_limits< double >::infinity() )
   {  return ; }
   /*----------------------------------------------------------------------*/
   m_f.val_do( COL_EXIF_FOCALLENGTH ) = do_val  ;
   m_f.val_s( COL_EXIF_FOCALLENGTH ).Printf( "%0.2lfmm",
                                             sr::round_01( do_val )
                                           ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_exif_focallength35mmfilm( long l_val )
{
   /*----------------------------------------------------------------------*/
   m_f.val_ll( COL_EXIF_FOCALLENGTHIN35MMFILM ) = l_val ;
   m_f.val_s( COL_EXIF_FOCALLENGTHIN35MMFILM ).Printf( "%ldmm", l_val ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_exif_exposuretime( unsigned long ul_nomin,
                                        unsigned long ul_denomin
                                      )
{
   /*----------------------------------------------------------------------*/
   double do_val = ( double )ul_nomin / ul_denomin ;
   /*----------------------------------------------------------------------*/
   if( do_val == std::numeric_limits< double >::infinity() )
   {  return ; }
   /*----------------------------------------------------------------------*/
   m_f.val_do( COL_EXIF_EXPOSURETIME ) = do_val ;
   /*----------------------------------------------------------------------*/
   if( ul_nomin == 0 )
   {  m_f.val_s( COL_EXIF_EXPOSURETIME ).Printf( "%lu/%lus",
                                                 ul_nomin, ul_denomin
                                                ) ;
   }
   else
   if( ul_denomin <= ul_nomin )
   {  m_f.val_s( COL_EXIF_EXPOSURETIME ).Printf( "%0.2lfs",
                                                 sr::round_01( do_val )
                                               ) ;
   }
   else
   if( ul_denomin % ul_nomin == 0 )
   {  m_f.val_s( COL_EXIF_EXPOSURETIME ).Printf( "1/%lus",
                                                 ul_denomin / ul_nomin
                                                ) ;
   }
   else
   {  m_f.val_s( COL_EXIF_EXPOSURETIME ).Printf( "1/%0.1lfs",
                               sr::round_1( ( double )ul_denomin / ul_nomin )
                                                ) ;
   }
   /*----------------------------------------------------------------------*/
}


/*-------------------------------------------------------------------------*/
void CFileInit::init_exif_rotation( e_iv_rotation iv_rotation )
{
   /*----------------------------------------------------------------------*/
   const char *p_c_text ;
   /*-----------------------------------------------------------------------+
   ! The text explains to which sides of the image correspond the           !
   ! "first line" and "first column"                                        !
   ! For example:                                                           !
   ! top-left => first line corresponds to the top of the image             !
   !             first column corresponds to the left of the image          !
   ! In this case the image is correctly oriented, nothing to do: 0 deg     !
   +------------------------------------------------------------------------+
   ! The "_HM" suffix stands for "Horizontal Mirror"                        !
   +-----------------------------------------------------------------------*/
   switch( iv_rotation )
   {
      /*-------------------------------------------------------------------*/
      case IV_ROTATION_0      : p_c_text = "top-left"     ; break ;
      case IV_ROTATION_0_HM   : p_c_text = "top-right"    ; break ;
      case IV_ROTATION_180    : p_c_text = "bottom-right" ; break ;
      case IV_ROTATION_180_HM : p_c_text = "bottom-left"  ; break ;
      case IV_ROTATION_90_HM  : p_c_text = "left-top"     ; break ;
      case IV_ROTATION_90     : p_c_text = "right-top"    ; break ;
      case IV_ROTATION_270_HM : p_c_text = "right-bottom" ; break ;
      case IV_ROTATION_270    : p_c_text = "left-bottom"  ; break ;
      /*-------------------------------------------------------------------*/
      default :
         p_c_text    = "undefined" ;
         iv_rotation = IV_ROTATION_0 ;
         break ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   m_f.set_img_rotation( iv_rotation ) ;
   m_f.val_s( COL_EXIF_ORIENTATION ) = p_c_text ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::init_doc_nb_files( int i_nb )
{
   /*----------------------------------------------------------------------*/
   if( i_nb > 0 && reserve_col( COL_DOC_FILECOUNT ) )
   {
      /*-------------------------------------------------------------------*/
      m_f.val_ll( COL_DOC_FILECOUNT ) = i_nb ;
      m_f.val_s( COL_DOC_FILECOUNT ) << i_nb ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*--( Many file types are "zipped" )---------------------------------------*/
void CFileInit::init_doc_zip_info( wxZipInputStream &zip_input )
{
   /*----------------------------------------------------------------------*/
   init_doc_nb_files( zip_input.GetTotalEntries() ) ;
   /*----------------------------------------------------------------------*/
   wxString s_comment = zip_input.GetComment() ;
   /*----------------------------------------------------------------------*/
   if(    reserve_col( COL_DOC_COMMENT )
       && prepare_string( s_comment ) > 0
     )
   {  m_f.val_s( COL_DOC_COMMENT ) = s_comment ; }
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Read an array of chars and convert it to a string based on current        !
! conversion                                                                !
+--------------------------------------------------------------------------*/
int CFileInit::file_read_tb_c( size_t sz_size, wxString &s )
{
   /*----------------------------------------------------------------------*/
   char   tb_c_val[ g_co_i_string_sz_max * 2 ] ;
   size_t sz_size_to_read  ;

   /*----------------------------------------------------------------------*/
   s.clear() ;
   /*----------------------------------------------------------------------*/
   if( sz_size == 0 ) { return( 0 ) ; }
   /*----------------------------------------------------------------------*/
   sz_size_to_read = wxMin( sz_size, sizeof( tb_c_val ) ) ;
   /*----------------------------------------------------------------------*/
   if(    m_fa.read_buffer( sz_size_to_read, tb_c_val ) != 0
       || (    sz_size_to_read < sz_size
            && m_fa.skip_nb_byte( sz_size - sz_size_to_read ) != 0
          )
     )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   prepare_string( tb_c_val, sz_size_to_read, s ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Read an array of wide chars and convert it to a string based on current   !
! conversion                                                                !
! "sz_size" is expressed in bytes                                           !
+--------------------------------------------------------------------------*/
int CFileInit::file_read_tb_wc_le( size_t sz_size, wxString &s )
{
   /*----------------------------------------------------------------------*/
   char   tb_c_val[ g_co_i_string_sz_max * 2 ] ;
   size_t sz_size_to_read ;

   /*----------------------------------------------------------------------*/
   s.clear() ;
   /*----------------------------------------------------------------------*/
   if( sz_size == 0 ) { return( 0 ) ; }
   /*----------------------------------------------------------------------*/
   sz_size_to_read = wxMin( sz_size, sizeof( tb_c_val ) ) ;
   /*--( Read and possibly skip )------------------------------------------*/
   if(    m_fa.read_buffer( sz_size_to_read, tb_c_val ) != 0
       || (    sz_size_to_read < sz_size
            && m_fa.skip_nb_byte( sz_size - sz_size_to_read ) != 0
          )
     )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   sr::prepare_string( tb_c_val, sz_size_to_read, s, sr::CONV_FROM_UTF16LE );
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Read info RIFF structured )------------------------------------------*/
int CFileInit::riff_read_serie_info( wxUint32 &dw_size_list,
                                  int ( *p_f_conv_tag_col )( wxUint32 dw_id )
                                    )
{
   /*----------------------------------------------------------------------*/
   str_chunk chunk ;
   int       i_col ;

   /*----------------------------------------------------------------------*/
   while( dw_size_list > sizeof( chunk ) )
   {
      /*-------------------------------------------------------------------*/
      if( m_fa.read_le_data( chunk ) != 0 )
      { return( -1 ) ; }
      /*-------------------------------------------------------------------*/
      dw_size_list -= sizeof( chunk ) ;

      /*--------------------------------------------------------------------+
      ! There might be some "alignment" problems. If needed a jump over the !
      ! gap is done. Generally only 1 byte (cf "That_Day.avi" )             !
      ! The tag names begin with an "I" except for an old one from the      !
      ! "wav"s: "DATE"                                                      !
      +--------------------------------------------------------------------*/
      while(    *( char * )&chunk.dw_id != 'I'
             && chunk.dw_id != SR_FOURCC( 'D','A','T','E' )
             && dw_size_list > 0
           )
      {  /*----------------------------------------------------------------*/
         if(    m_fa.skip_nb_byte( - ( ( int )sizeof( chunk ) - 1 ) ) != 0
             || m_fa.read_le_data( chunk ) != 0
           )
         {  return( -2 ) ; }
         /*----------------------------------------------------------------*/
         --dw_size_list ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      if( dw_size_list <= 0 ) { return( -3 ) ; }

      /*-------------------------------------------------------------------*/
      i_col = p_f_conv_tag_col( chunk.dw_id ) ;

      /*-------------------------------------------------------------------*/
      if( reserve_col( i_col ) )
      {  /*----------------------------------------------------------------*/
         wxString s_val ;
         /*----------------------------------------------------------------*/
         if( file_read_tb_c( chunk.dw_size, s_val ) == 0 && !s_val.empty() )
         {  m_f.val_s( i_col ) = s_val ; }
         /*----------------------------------------------------------------*/
      }
      else
      if( m_fa.skip_nb_byte( chunk.dw_size ) != 0 )
      {  return( -4 ) ; }
      /*-------------------------------------------------------------------*/
      dw_size_list -= chunk.dw_size ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! The current offset will be restored before returning                      !
! If the code type is unknown or the image analyse fails the return code    !
! will be 0. It will be non 0 if direct file access fails.                  !
+--------------------------------------------------------------------------*/
int CFileInit::read_embedded_image_info( wxFileOffset fo_offset,
                                         wxULongLong ull_size,
                                         const wxString &s_mime
                                       )
{
   /*----------------------------------------------------------------------*/
   wxFileOffset fo_offset_sav = m_fa.get_offset() ;
   wxString     s_local_mime                      ;
   int          i_image_init_ret_code             ;

   /*----------------------------------------------------------------------*/
   if( m_fa.set_offset( fo_offset ) != 0 ) { return( -1 ) ; }

   /*--( Make things easier and more permissive than the standards ... )---*/
   s_local_mime = s_mime.Lower() ;
   if( s_local_mime.StartsWith( "image/" ) )
   {  s_local_mime.Remove( 0, 6 ) ; }
   /*----------------------------------------------------------------------*/
   if( s_local_mime == "jpg" || s_local_mime == "jpeg" )
   {  i_image_init_ret_code = jpg_read( fo_offset, ull_size ) ; }
   else
   if( s_local_mime == "png" )
   {  i_image_init_ret_code = png_read( fo_offset, ull_size ) ; }
   else
   if( s_local_mime == "bmp" )
   {  i_image_init_ret_code = bmp_read( fo_offset, ull_size ) ; }
   else
   if( s_local_mime == "gif" )
   {  i_image_init_ret_code = gif_read( fo_offset, ull_size ) ; }
   else
   if( s_local_mime == "ico" || s_local_mime == "cur" )
   {  i_image_init_ret_code = ico_read( fo_offset, ull_size ) ; }
   else
   if( s_local_mime == "tif" || s_local_mime == "tiff" )
   {  i_image_init_ret_code = tiff_read_tiff( fo_offset, ull_size ) ; }
   else
   if( s_local_mime == "pcx" )
   {  i_image_init_ret_code = pcx_read( fo_offset, ull_size ) ; }
   else
   if( s_local_mime == "xpm" )
   {  i_image_init_ret_code = xpm_read( fo_offset, ull_size ) ; }
   else
   if( s_local_mime == "tga" )
   {  i_image_init_ret_code = tga_read( fo_offset, ull_size ) ; }
   else
   if( s_local_mime == "pnm" )
   {  i_image_init_ret_code = pnm_read( fo_offset, ull_size ) ; }
   else /*------------------------------------------------------------------+
        ! Some mime types may be misspelled, that's why the standard        !
        ! formats are tried.                                                !
        +------------------------------------------------------------------*/
   if(    jpg_read( fo_offset, ull_size ) == 0
       || (    m_fa.set_offset( fo_offset ) == 0
            && png_read( fo_offset, ull_size ) == 0
          )
     )
   {  i_image_init_ret_code = 0 ; }
   else
   {  i_image_init_ret_code = -1 ; }

   /*--( Go back to source )-----------------------------------------------*/
   if( m_fa.set_offset( fo_offset_sav ) != 0 ) { return( -2 ) ; }

   /*----------------------------------------------------------------------*/
   if( i_image_init_ret_code == 0 )
   {  m_f.set_image_offset( fo_offset ) ;
      m_f.set_image_size( ull_size ) ;
   }
   /*-----------------------------------------------------------------------+
   ! Can not analyse the image doesn't mean the treatment has to be stopped !
   +-----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFileInit::add_codec_info( const wxString &s_info, wxString &s_result
                              ) const
{
   /*----------------------------------------------------------------------*/
   sr::append_with_sep( s_info, ", ", s_result ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString CFileInit::concat_av_codecs( const wxString &s_audio,
                                      const wxString &s_video
                                    ) const
{
   /*----------------------------------------------------------------------*/
   if( !s_audio.empty() && !s_video.empty() )
   {  return( s_audio + " - " + s_video ) ; }
   /*----------------------------------------------------------------------*/
   if( !s_audio.empty() ) { return( s_audio ) ; }
   /*----------------------------------------------------------------------*/
   return( s_video ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                      End of file CFileInit_Base.cpp                       !
+==========================================================================*/
