/*====================================+=====================================+
! File CFile_ExpMod2.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!       Expression : evaluation of the standard modifiers "[],() ..."       !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <wx/tokenzr.h>
#include <wx/regex.h>
#include "CFile.h"
#include "CApp.h"
/*-------------------------------------------------------------------------*/



/*--------------------------------------------------------------------------+
! Substring extraction with separators: [from,nb,sep]                       !
! The substring is fully trimmed                                            !
!                                                                           !
! beg : start position of the substring, if negative it refers to the end.  !
!       If 0 then the complete string fully trimmed returns                 !
! nb  : optional, number of tab elements to extract                         !
!       By default nb is 1                                                  !
!       If <= 0 then its value is based on the number of elements:          !
!       (nb_tot - beg - nb) (0 means "until the end")                       !
! sep : optional, list of the characters used as separators                 !
!                                                                           !
! The two forms can be used: [,] and [,,] because of historical reasons.    !
! [from,nb,sep] appeared in version 3.00                                    !
! For the other modifiers an empty parameter ',' has to be specified.       !
!                                                                           !
! "eci.it_exp" must point to the opening bracket                            !
! "s_val" is in/out. On entry it contains the string to modify and on exit  !
! it contains the modified string.                                          !
!                                                                           !
+--------------------------------------------------------------------------*/
int CFile::exp_string_tab_elmt( t_exp_comp_info &eci,
                                wxString        &s_val
                              ) const
{
   /*----------------------------------------------------------------------*/
   int      i_elmt_beg = 0 ;
   int      i_elmt_nb  = 1 ;
   wxString s_sep = wxGetApp().M_s_chars_tab_sep.get() ;

   /*--( There must be an opening bracket )--------------------------------*/
   if( *eci.it_exp != '[' ) { return( -1 ) ; }
   /*--( Jump over it )----------------------------------------------------*/
   ++eci.it_exp ;

   /*--( Then extract the start index )------------------------------------*/
   extr_integer( eci, i_elmt_beg ) ;
   if( eci.it_exp == eci.it_exp_end ) { return( -2 ) ; }

   /*-----------------------------------------------------------------------+
   ! Then come two optional parameters:                                     !
   ! number of elements and separators string                               !
   +-----------------------------------------------------------------------*/
   if( *eci.it_exp == ',' )
   {
      /*-------------------------------------------------------------------*/
      sr::next_non_space( eci.it_exp_end, ++eci.it_exp ) ;

      /*--( A variable as parameter ? )------------------------------------*/
      if( *eci.it_exp == '%' )
      {
         /*----------------------------------------------------------------*/
         wxString        s_param   ;
         t_exp_comp_info eci_param ;
         int             i_param   ;
         /*--( Extraction of the param )-----------------------------------*/
         extr_compute_variable( eci, s_param ) ;
         eci_param.boo_first_sub_exp = true ;
         eci_param.it_exp = s_param.begin() ;
         eci_param.it_exp_end = s_param.end() ;
         /*--( Its type will determine the syntax used )-------------------*/
         if( extr_integer( eci_param, i_param ) == s_param.end() )
         {  i_elmt_nb = i_param ; }
         else
         {  s_sep = s_param ; }
         /*----------------------------------------------------------------*/
      }
      else
      if( !sr::is_string_delimiter( *eci.it_exp ) )
      {  /*----------------------------------------------------------------*/
         extr_integer( eci, i_elmt_nb ) ;
         if( eci.it_exp == eci.it_exp_end ) { return( -3 ) ; }
         /*--( A coma can be present before the separators string )--------*/
         if( *eci.it_exp == ',' )
         {  sr::next_non_space( eci.it_exp_end, ++eci.it_exp ) ;
            if( eci.it_exp == eci.it_exp_end ) { return( -4 ) ; }
         }
         /*----------------------------------------------------------------*/
      }
      /*--( The separators list )------------------------------------------*/
      if( sr::is_string_delimiter( *eci.it_exp ) )
      {  extr_quoted_string( eci, s_sep ) ;
         if( eci.it_exp == eci.it_exp_end ) { return( -5 ) ; }
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( The bracket has to be closed )------------------------------------*/
   if( *eci.it_exp != ']' ) { return( -6 ) ; }

   /*--( 0 as position means complete string fully trimmed )---------------*/
   if( i_elmt_beg == 0 )
   {  s_val.Trim( false ).Trim() ; return( 0 ) ; }

   /*--( Split with all tokens even the empty ones )-----------------------*/
   wxStringTokenizer tokenizer( s_val, s_sep, wxTOKEN_RET_EMPTY_ALL ) ;
   size_t sz_from ;
   size_t sz_to   ;

   /*--( Recompute "real" indexes )----------------------------------------*/
   if( i_elmt_beg < 0 )
   {  i_elmt_beg += tokenizer.CountTokens() + 1 ; }
   if( i_elmt_nb <= 0 )
   {  i_elmt_nb += tokenizer.CountTokens() - i_elmt_beg + 1 ; }
   /*--( Out of limits is not an error, it just returns an empty result )--*/
   if( i_elmt_beg > ( int )tokenizer.CountTokens() || i_elmt_nb <= 0 )
   {  s_val.clear() ; return( 0 ) ; }

   /*--( Minimal control )-------------------------------------------------*/
   if( i_elmt_beg <= 0 ) { return( -7 ) ; }

   /*--( Computing the position of the first character to extract )--------*/
   for( ; i_elmt_beg > 1 && tokenizer.HasMoreTokens() ; --i_elmt_beg  )
   {  tokenizer.GetNextToken() ; }
   sz_from = tokenizer.GetPosition() ;
   /*--( Computing the position of the last character to extract )---------*/
   for( ; i_elmt_nb > 0 && tokenizer.HasMoreTokens() ; --i_elmt_nb )
   {  tokenizer.GetNextToken() ; }
   sz_to = tokenizer.GetPosition() ;
   if( tokenizer.HasMoreTokens() ) { --sz_to ; }

   /*--( Only keep what is needed )----------------------------------------*/
   s_val = s_val.Mid( sz_from, sz_to - sz_from ).Trim( false ).Trim() ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Replacement modifier following the syntax: (from,to,beg,nb,case)          !
!                                                                           !
! from: string to replace                                                   !
! to  : optional, replacement string                                        !
!       if not specified the string to replace is deleted                   !
! beg : optional, starting occurrence (< 0 look backwards)                  !
! nb  : optional, number of occurrences to treat (< 0 => all)               !
! case: optional, search case sensitive ? (0=false (default), else true)    !
!                                                                           !
! "eci.it_exp" points after the "(" (maybe even after the first param).     !
! "s_val" is in/out. On entry it contains the string to modify and on exit  !
! it contains the modified string.                                          !
!                                                                           !
! The replace and substring operators uses the same delimiters "()". As the !
! first param can be a variable the the calling function has "extracted"    !
! the first parameter to determine which one to call.                       !
! To avoid a re-analyse of this first parameter is given in "s_param".      !
!                                                                           !
+--------------------------------------------------------------------------*/
int CFile::exp_replace( t_exp_comp_info &eci,
                        const wxString  &s_param,
                        wxString        &s_val
                      ) const
{
   /*----------------------------------------------------------------------*/
   wxString s_from           ;
   wxString s_to             ;
   int      i_start_occ =  1 ;
   int      i_nb_rep    = -1 ;
   int      i_case_sens =  0 ;

   /*----------------------------------------------------------------------*/
   if( !s_param.empty() )
   {  s_from = s_param ; }
   else
   {  /*-------------------------------------------------------------------*/
      if( !sr::is_string_delimiter( *eci.it_exp ) ) { return( -1 ) ; }
      /*--( The "from" string is mandatory )-------------------------------*/
      extr_quoted_string( eci, s_from ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   if( eci.it_exp == eci.it_exp_end ) { return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   if( *eci.it_exp == ',' )
   {  /*--( Then comes the replacement string )----------------------------*/
      ++eci.it_exp ;
      extr_quoted_string( eci, s_to ) ;
      if( eci.it_exp == eci.it_exp_end ) { return( -3 ) ; }
      /*--( First occurrence )---------------------------------------------*/
      if( *eci.it_exp == ',' )
      {  /*----------------------------------------------------------------*/
         ++eci.it_exp ;
         extr_integer( eci, i_start_occ ) ;
         if( eci.it_exp == eci.it_exp_end ) { return( -4 ) ; }
         /*--( Number of replacements )------------------------------------*/
         if( *eci.it_exp == ',' )
         {  /*-------------------------------------------------------------*/
            ++eci.it_exp ;
            extr_integer( eci, i_nb_rep ) ;
            if( eci.it_exp == eci.it_exp_end ) { return( -5 ) ; }
            /*--( Case sensitive search ? )--------------------------------*/
            if( *eci.it_exp == ',' )
            {  ++eci.it_exp ;
               extr_integer( eci, i_case_sens ) ;
               if( eci.it_exp == eci.it_exp_end ) { return( -6 ) ; }
            }
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   sr::replace_string( s_val, s_from, s_to, ( i_case_sens != 0 ),
                       i_start_occ, i_nb_rep
                     ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Extraction of a substring by position following the syntax: (beg,nb)      !
!                                                                           !
! beg : starting index, if < 0 then the extraction is done from the end     !
! nb  : optional, number of characters to extract                           !
!       By default nb is "until the end of the string"                      !
!       If <= 0 then its value is based on the end of the string:           !
!       (length - beg - nb) (0 meaning "until the end")                     !
!                                                                           !
! "eci.it_exp" points after the "(" (maybe even after the first param).     !
! "s_val" is in/out. On entry it contains the string to modify and on exit  !
! it contains the modified string.                                          !
!                                                                           !
! For an explanation of the "s_param" cf "exp_replace".                     !
!                                                                           !
+--------------------------------------------------------------------------*/
int CFile::exp_substr( t_exp_comp_info &eci,
                       const wxString  &s_param,
                       wxString        &s_val
                     ) const
{
   /*----------------------------------------------------------------------*/
   int  i_beg = 0            ;
   int  i_nb  = s_val.size() ;
   bool boo_param_nb_present ;

   /*----------------------------------------------------------------------*/
   if( s_param.empty() )
   {  extr_integer( eci, i_beg ) ; }
   else
   {  /*-------------------------------------------------------------------*/
      t_exp_comp_info eci_param ;
      /*-------------------------------------------------------------------*/
      eci_param.boo_first_sub_exp = true ;
      extr_integer( eci_param, i_beg ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( eci.it_exp == eci.it_exp_end ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   if( ( boo_param_nb_present = ( *eci.it_exp == ',' ) ) == true )
   {  /*-------------------------------------------------------------------*/
      ++eci.it_exp ;
      extr_integer( eci, i_nb ) ;
      if( eci.it_exp == eci.it_exp_end ) { return( -2 ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*--( Start negative => from the right of the string )------------------*/
   if( i_beg < 0 ) { i_beg += s_val.size() + 1 ; }
   /*--( Out of limit ? Useless to go further )----------------------------*/
   if( i_beg <= 0 ) { s_val.clear() ; return( 0 ) ; }
   /*----------------------------------------------------------------------*/
   if( i_beg > ( int )s_val.size() )
   {  i_beg = s_val.size() ; }
   else
   {  --i_beg ; }
   /*-----------------------------------------------------------------------+
   ! Negative value => string length minus nb                               !
   ! 0 means: "until the end"                                               !
   +-----------------------------------------------------------------------*/
   if( boo_param_nb_present && i_nb <= 0 )
   {  /*-------------------------------------------------------------------*/
      i_nb += s_val.size() - i_beg ;
      /*-------------------------------------------------------------------*/
      if( i_nb < 0 )
      {  s_val.clear() ; return( 0 ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   s_val = s_val.substr( i_beg, i_nb ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Apply a regular expression following the syntax: s/format/replacement/gi  !
!                                                                           !
! "eci.it_exp" points to the "s"                                            !
! "s_val" is in/out. On entry it contains the string to modify and on exit  !
! it contains the modified string.                                          !
!                                                                           !
+--------------------------------------------------------------------------*/
int CFile::exp_regex( t_exp_comp_info &eci, wxString &s_val ) const
{
   /*----------------------------------------------------------------------*/
   wxLogNull        no_log           ; // Don't want syntax error message
   sr::wxString_cit it_begin_sav = eci.it_exp ;
   wxString  tb_s_exp[ 2 ]           ;
   int       i_exp_num               ;
   int       i_flags  = wxRE_DEFAULT ;
   int       i_nb_rep = 1            ;

   /*--( Minimal size and controls )---------------------------------------*/
   if( *eci.it_exp != 's' || eci.it_exp_end - eci.it_exp < 5 )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   ++eci.it_exp ;
   if( *eci.it_exp != '/' )
   {  return( -2 ) ; }

   /*-----------------------------------------------------------------------+
   ! The string is supposed to contain two parts:                           !
   ! - The search string (regex)                                            !
   ! - The replacement format (fmt)                                         !
   +-----------------------------------------------------------------------*/
   for( i_exp_num = 0 ; i_exp_num < 2 ; ++i_exp_num )
   {
      /*-------------------------------------------------------------------*/
      for( ++eci.it_exp ;
              eci.it_exp != eci.it_exp_end
           && (    *eci.it_exp != '/'
                || (    eci.it_exp > it_begin_sav
                     && *( eci.it_exp - 1 ) == '\\'
                   )
              ) ;
           ++eci.it_exp
         )
      {
         /*--( Escaped characters => the "escape" is copied )--------------*/
         if( *eci.it_exp == '\\' && eci.it_exp + 1 < eci.it_exp_end )
         {  tb_s_exp[ i_exp_num ] += *eci.it_exp++ ; }
         /*----------------------------------------------------------------*/
         tb_s_exp[ i_exp_num ] += *eci.it_exp ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      if( eci.it_exp == eci.it_exp_end || *eci.it_exp != '/' )
      {  return( -3 ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*--( Search for the flags )--------------------------------------------*/
   for( ++eci.it_exp ;
           eci.it_exp != eci.it_exp_end
        && ( *eci.it_exp == 'i' || *eci.it_exp == 'g' ) ;
        ++eci.it_exp
      )
   {
      /*-------------------------------------------------------------------*/
      switch( ( *eci.it_exp ).GetValue() )
      {  case 'i' : i_flags |= wxRE_ICASE ; break ;
         case 'g' : i_nb_rep = 0          ; break ;
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( The next non space should be a ')' )------------------------------*/
   sr::next_non_space( eci.it_exp_end, eci.it_exp ) ;

   /*--( Go using "ARE" regular expression flavour )-----------------------*/
   wxRegEx regex( tb_s_exp[ 0 ], i_flags | wxRE_ADVANCED ) ;
   if( !regex.IsValid() ) { return( -4 ) ; }
   /*----------------------------------------------------------------------*/
   regex.Replace( &s_val, tb_s_exp[ 1 ], i_nb_rep ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Apply a "tr" standard operator following the syntax: tr/from/to/cds       !
!                                                                           !
! "from" and "to" can specify character intervals like 'a-z', '0-3' etc.    !
!                                                                           !
! "eci.it_exp" points to the "t" of "tr"                                    !
! "s_val" is in/out. On entry it contains the string to modify and on exit  !
! it contains the modified string.                                          !
!                                                                           !
+--------------------------------------------------------------------------*/
int CFile::exp_tr( t_exp_comp_info &eci, wxString &s_val ) const
{
   /*----------------------------------------------------------------------*/
   sr::wxString_cit it_begin_sav = eci.it_exp ;
   wxString  tb_s_exp[ 2 ] ;
   int       i_exp_num     ;
   bool      boo_complement = false ;
   bool      boo_delete     = false ;
   bool      boo_squeeze    = false ;

   /*--( Minimal size and controls )---------------------------------------*/
   if(    eci.it_exp_end - eci.it_exp < 5
       || *eci.it_exp != 't'
       || *( eci.it_exp + 1 ) != 'r'
     )
   {  return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   eci.it_exp += 2 ;
   if( *eci.it_exp != '/' )
   {  return( -2 ) ; }

   /*-----------------------------------------------------------------------+
   ! The string is supposed to contain two parts:                           !
   ! - The "from" string                                                    !
   ! - The "to" string                                                      !
   +-----------------------------------------------------------------------*/
   for( i_exp_num = 0 ; i_exp_num < 2 ; ++i_exp_num )
   {
      /*-------------------------------------------------------------------*/
      for( ++eci.it_exp ;
              eci.it_exp != eci.it_exp_end
           && (    *eci.it_exp != '/'
                || (    eci.it_exp > it_begin_sav
                     && *( eci.it_exp - 1 ) == '\\'
                   )
              ) ;
           ++eci.it_exp
         )
      {
         /*--( Escaped characters => the "escape" is skipped )-------------*/
         if( *eci.it_exp == '\\' && eci.it_exp + 1 < eci.it_exp_end )
         {  ++eci.it_exp ; }
         /*----------------------------------------------------------------*/
         tb_s_exp[ i_exp_num ] += *eci.it_exp ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      if( eci.it_exp == eci.it_exp_end || *eci.it_exp != '/' )
      {  return( -3 ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*--( Search for the flags )--------------------------------------------*/
   for( ++eci.it_exp ;
           eci.it_exp != eci.it_exp_end
        && (    *eci.it_exp == 'c'
             || *eci.it_exp == 'd'
             || *eci.it_exp == 's'
           ) ;
        ++eci.it_exp
      )
   {
      /*-------------------------------------------------------------------*/
      switch( ( *eci.it_exp ).GetValue() )
      {  case 'c' : boo_complement = true ; break ;
         case 'd' : boo_delete     = true ; break ;
         case 's' : boo_squeeze    = true ; break ;
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( The next non space should be a ')' )------------------------------*/
   sr::next_non_space( eci.it_exp_end, eci.it_exp ) ;

   /*--( "to" can't be empty is "from" is not )----------------------------*/
   if( tb_s_exp[ 1 ].empty() && !tb_s_exp[ 0 ].empty() )
   {  return( -4 ) ; }

   /*--( If "complement" is asked then "to" can't be empty )---------------*/
   if( boo_complement && tb_s_exp[ 0 ].empty() )
   {  return( -5 ) ; }

   /*----------------------------------------------------------------------*/
   wxString         s_res  ;
   sr::wxString_cit it_in  ;
   sr::wxString_cit it     ;
   wxUniChar        c_repl ;
   /*-----------------------------------------------------------------------+
   ! These index variables will contain the "position" of a character. The  !
   ! intervals length are counted too. Ie for "a-j" "j" is at index 9.      !
   +-----------------------------------------------------------------------*/
   int i_ind_found ;
   int i_ind       ;
   /*----------------------------------------------------------------------*/
   s_res.reserve( s_val.size() ) ;

   /*----------------------------------------------------------------------*/
   for( it_in = s_val.begin() ; it_in != s_val.end() ; ++it_in )
   {
      /*--( Search for this character in "from" )--------------------------*/
      for( it = tb_s_exp[ 0 ].begin(), i_ind_found = -1, i_ind = 0 ;
           it != tb_s_exp[ 0 ].end() && i_ind_found < 0 ;
           ++it
         )
      {  /*--( Range description ? Example : A-Z )-------------------------*/
         if(    *it == '-'
             && it > tb_s_exp[ 0 ].begin()
             && it < tb_s_exp[ 0 ].end() - 1
           )
         {  /*--( Characters delimiting the interval )---------------------*/
            wxUniChar c_a = *( it - 1 ) ;
            wxUniChar c_b = *( it + 1 ) ;
            /*--( Current char part of the interval ? )--------------------*/
            if( *it_in >= c_a && *it_in <= c_b )
            {    i_ind_found
               = i_ind - 1 + ( *it_in ).GetValue() - c_a.GetValue() ;
            }
            else /*--( No, skip the interval and so the nb of its elts )---*/
            {  ++it ;
               i_ind += c_b - c_a + 1 ;
            }
            /*-------------------------------------------------------------*/
         }
         else /*--( "Pure" equality )--------------------------------------*/
         if( *it == *it_in )
         {  i_ind_found = i_ind ; }
         else
         {  ++i_ind ; }
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      c_repl = 0 ;

      /*--------------------------------------------------------------------+
      ! If the character has not been found it is simply copied.            !
      ! This rule is inverted if "complement" asked.                        !
      +--------------------------------------------------------------------*/
      if(    ( i_ind_found <  0 && !boo_complement )
          || ( i_ind_found >= 0 &&  boo_complement )
        )
      {  c_repl = *it_in ; }
      else /*---------------------------------------------------------------+
           ! In "complement" mode if it has not been found it has to be     !
           ! replaced ... the last char of the "to" is used.                !
           ! Note that in "complement" mode "to" can't be empty (cf upper). !
           +---------------------------------------------------------------*/
      if( boo_complement && i_ind_found < 0 )
      {  c_repl = *tb_s_exp[ 1 ].rbegin() ; }
      else
      {  /*-----------------------------------------------------------------+
         ! The character has been found in "from".                          !
         ! Its replacement character has to be searched in "to".            !
         ! To find it "i_ind_found" is decreased to 0.                      !
         +-----------------------------------------------------------------*/
         for( it = tb_s_exp[ 1 ].begin() ;
              it != tb_s_exp[ 1 ].end() && c_repl == 0 ;
              ++it
            )
         {  /*--( Range description ? Example : A-Z )----------------------*/
            if(    *it == '-'
                && it > tb_s_exp[ 1 ].begin()
                && it < tb_s_exp[ 1 ].end() - 1
              )
            {  /*--( Characters delimiting the interval )------------------*/
               wxUniChar c_a = *( it - 1 ) ;
               wxUniChar c_b = *( it + 1 ) ;

               /*--( Index is part of the interval ? )---------------------*/
               if( i_ind_found < c_b - c_a )
               {  c_repl = wxUniChar( c_a.GetValue() + i_ind_found + 1 ) ; }
               else
               {  ++it ;
                  i_ind_found -= c_b - c_a + 1 ;
               }
               /*----------------------------------------------------------*/
            }
            else /*--( At the end ? )--------------------------------------*/
            if( i_ind_found-- == 0 )
            {  c_repl = *it ; }
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }

      /*--------------------------------------------------------------------+
      ! For characters found in "from" without correspondence in "to" the   !
      ! last element of "to" is copied.                                     !
      ! "d" disables this behaviour.                                        !
      +--------------------------------------------------------------------*/
      if( c_repl == 0 && !boo_delete )
      {  c_repl = *tb_s_exp[ 1 ].rbegin() ; }

      /*--------------------------------------------------------------------+
      ! Character finally added if it satisfies the "non duplicate" rule    !
      ! (if asked).                                                         !
      +--------------------------------------------------------------------*/
      if(    c_repl != 0
          && ( !boo_squeeze || s_res.empty() || c_repl != *s_res.rbegin() )
        )
      {  s_res += c_repl ; }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   s_val = s_res ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                       End of File CFile_ExpMod2.cpp                       !
+==========================================================================*/
