/*====================================+=====================================+
! File sr_string.cpp                  ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <wx/convauto.h>
#include "CApp.h"
#include "common/sr_lib.h"
#include "common/sr_string.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
void sr::next_non_space( sr::wxString_cit it_end, sr::wxString_cit &it )
{
   /*----------------------------------------------------------------------*/
   while( it != it_end && sr::bool_isspace( *it ) ) { ++it ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
bool sr::ic_is_lower( const wxUniChar &a, const wxUniChar &b )
{
   /*--( Same characters ? )-----------------------------------------------*/
   if( a == b ) { return( false ) ; }

   /*--( Then compare on the upper-case version )--------------------------*/
   const wxUniChar a_up = wxToupper( a ) ;
   const wxUniChar b_up = wxToupper( b ) ;
   /*----------------------------------------------------------------------*/
   if( a_up == b_up ) { return( false ) ; }

   /*--( Then compare the non accented upper-case version )----------------*/
   const wxUniChar a_up_na = sr::unaccent( a_up ) ;
   const wxUniChar b_up_na = sr::unaccent( b_up ) ;
   /*----------------------------------------------------------------------*/
   if( a_up_na == b_up_na ) { return( a_up < b_up ) ; }

   /*--( Digits and characters at the end )--------------------------------*/
   const bool boo_b_alnum = sr::bool_isalnum( b_up_na ) ;
   if( sr::bool_isalnum( a_up_na ) != boo_b_alnum )
   {  return( boo_b_alnum ) ; }

   /*--( Else on the non accented version => accented 'e' before 'd' )-----*/
   return( a_up_na < b_up_na ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int sr::ic_comp( sr::wxString_cit it1, sr::wxString_cit it2, int i_nb )
{
   /*----------------------------------------------------------------------*/
   for( ; i_nb > 0 ; --i_nb, ++it1, ++it2 )
   {  if( !sr::ic_equal( *it1, *it2 ) )
      {  return( sr::ic_is_lower( *it1, *it2 ) ? -1 : +1 ) ; }
   }
   /*--( Arrived there the strings are identical )-------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int sr::ic_comp( const wxString &s1, const wxString &s2 )
{
   /*----------------------------------------------------------------------*/
   sr::wxString_cit it1 = s1.begin() ;
   sr::wxString_cit it2 = s2.begin() ;
   int i_nb = wxMin( s1.size(), s2.size() ) ;
   /*----------------------------------------------------------------------*/
   for( ; i_nb > 0 ; --i_nb, ++it1, ++it2 )
   {
      /*-------------------------------------------------------------------*/
      if( !sr::ic_equal( *it1, *it2 ) )
      {  return( sr::ic_is_lower( *it1, *it2 ) ? -1 : +1 ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*--( Finally the longest string will "win" )---------------------------*/
   return( s1.size() - s2.size() ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Natural string comparison : numbers in strings and case insensitive )*/
int sr::natural_comp( const wxString &s1, const wxString &s2 )
{
   /*----------------------------------------------------------------------*/
   sr::wxString_cit it1 = s1.begin()  ;
   int              i_nb1 = s1.size() ;
   sr::wxString_cit it2 = s2.begin()  ;
   int              i_nb2 = s2.size() ;
   /*----------------------------------------------------------------------*/
   while( i_nb1 > 0 && i_nb2 > 0 )
   {
      /*--( If the strings contain numbers then compare them )-------------*/
      if( sr::bool_isdigit( *it1 ) && sr::bool_isdigit( *it2 ) )
      {
         /*----------------------------------------------------------------*/
         sr::wxString_cit it1_sav = it1 ;
         sr::wxString_cit it2_sav = it2 ;
         /*--( Jump over the '0' )-----------------------------------------*/
         while( i_nb1 > 0 && *it1 == '0' ) { ++it1 ; --i_nb1 ; }
         while( i_nb2 > 0 && *it2 == '0' ) { ++it2 ; --i_nb2 ; }
         /*--( Skip same digits )------------------------------------------*/
         while(    i_nb1 > 0 && i_nb2 > 0
                && *it1 == *it2 && sr::bool_isdigit( *it1 )
              )
         {  ++it1 ; --i_nb1 ; ++it2 ; --i_nb2 ; }
         /*--( First different chars )-------------------------------------*/
         bool boo_num1 = ( i_nb1 > 0 && sr::bool_isdigit( *it1 ) ) ;
         bool boo_num2 = ( i_nb2 > 0 && sr::bool_isdigit( *it2 ) ) ;
         /*----------------------------------------------------------------*/
         if( !boo_num1 && !boo_num2 )
         {  /*--( The values are identical only the number 0 is different )*/
            int i_nb_0_diff = ( it1 - it1_sav ) - ( it2 - it2_sav ) ;
            /*--( The one preceded by the highest nb of 0 is "lower" )-----*/
            if( i_nb_0_diff != 0 ) { return( - i_nb_0_diff ) ; }
            /*--( Same numbers. The rests have to be compared )------------*/
         }
         else /*--( One is a number and not the other ? )------------------*/
         if( boo_num1 != boo_num2 )
         {  return( boo_num1 ? +1 : -1 ) ; }
         else
         {
            /*--------------------------------------------------------------+
            ! The two characters are different digits.                      !
            ! The comparison of the number of digits constituting the       !
            ! rest of the numbers will tell the "highest".                  !
            +--------------------------------------------------------------*/
            int i_nb_num1 = 0 ;
            int i_nb_num2 = 0 ;
            /*--( Number of following digits )-----------------------------*/
            while( i_nb1 > 0 && sr::bool_isdigit( *( it1 + i_nb_num1 ) ) )
            {  ++i_nb_num1 ; --i_nb1 ; }
            while( i_nb2 > 0 && sr::bool_isdigit( *( it2 + i_nb_num2 ) ) )
            {  ++i_nb_num2 ; --i_nb2 ; }
            /*--( Not the same number of digits ? the longest wins )-------*/
            if( i_nb_num1 != i_nb_num2 )
            {  return( i_nb_num1 - i_nb_num2 ) ; }
            /*--------------------------------------------------------------+
            ! Same thing ? Then compare the digits that made the difference !
            +--------------------------------------------------------------*/
            return( *it1 - *it2 ) ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }
      else /*--( The comparison is case insensitive )----------------------*/
      if( !sr::ic_equal( *it1, *it2 ) )
      {  return( sr::ic_is_lower( *it1, *it2 ) ? -1 : +1 ) ; }
      else
      {  --i_nb1 ; ++it1 ; --i_nb2 ; ++it2 ; }
      /*-------------------------------------------------------------------*/
   }
   /*--( Finally the longest string "wins" )-------------------------------*/
   return( s1.size() - s2.size() ) ;
   /*----------------------------------------------------------------------*/
}

/*--( "Computation" of the non accented version of a character )-----------*/
wxUniChar sr::unaccent( const wxUniChar &c )
{
   /*--( Ascii means non accented ... )------------------------------------*/
   if( c.IsAscii() ) { return( c ) ; }
   /*----------------------------------------------------------------------*/
   switch( c.GetValue() )
   {
      /*-------------------------------------------------------------------*/
      case 0x01E2 :
      case 0x01FC :
         return( 0x00C6 ) ; // E in A
      case 0x00C0 : // '' : // 0xC0 192
      case 0x00C1 : // '' : // 0xC1 193
      case 0x00C2 : // '' : // 0xC2 194
      case 0x00C3 : // '' : // 0xC3 195
      case 0x00C4 : // '' : // 0xC4 196
      case 0x00C5 : // '' : // 0xC5 197
      case 0x0100 :
      case 0x0102 :
      case 0x0104 :
      case 0x01CD :
      case 0x01DE :
      case 0x01E0 :
      case 0x01FA :
      case 0x0200 :
      case 0x0202 :
      case 0x0226 :
         return( 'A' ) ;
      case 0x00C7 : // '' : // 0xC7 199
      case 0x0106 :
      case 0x0108 :
      case 0x010A :
      case 0x010C :
      case 0x0187 :
         return( 'C' ) ;
      case 0x010E :
      case 0x0110 :
         return( 'D' ) ;
      case 0x00C8 : // '' : // 0xC8 200
      case 0x00C9 : // '' : // 0xC9 201
      case 0x00CA : // '' : // 0xCA 202
      case 0x00CB : // '' : // 0xCB 203
      case 0x0112 :
      case 0x0114 :
      case 0x0116 :
      case 0x0118 :
      case 0x011A :
      case 0x0204 :
      case 0x0206 :
      case 0x0228 :
         return( 'E' ) ;
      case 0x011C :
      case 0x011E :
      case 0x0120 :
      case 0x0122 :
      case 0x01E4 :
      case 0x01E6 :
      case 0x01F4 :
         return( 'G' ) ;
      case 0x0124 :
      case 0x0126 :
      case 0x021E :
         return( 'H' ) ;
      case 0x00CC : // '' : // 0xCC 204
      case 0x00CD : // '' : // 0xCD 205
      case 0x00CE : // '' : // 0xCE 206
      case 0x00CF : // '' : // 0xCF 207
      case 0x0128 :
      case 0x012A :
      case 0x012C :
      case 0x012E :
      case 0x0130 :
      case 0x01CF :
      case 0x0208 :
      case 0x020A :
         return( 'I' ) ;
      case 0x0134 :
         return( 'J' ) ;
      case 0x0136 :
      case 0x01E8 :
         return( 'K' ) ;
      case 0x0139 :
      case 0x013B :
      case 0x013D :
      case 0x013F :
      case 0x0141 :
         return( 'L' ) ;
      case 0x00D1 : // '' : // 0xD1 209
      case 0x0143 :
      case 0x0145 :
      case 0x0147 :
      case 0x014A :
      case 0x01F8 :
         return( 'N' ) ;
      case 0x00D2 : // '' : // 0xD2 210
      case 0x00D3 : // '' : // 0xD3 211
      case 0x00D4 : // '' : // 0xD4 212
      case 0x00D5 : // '' : // 0xD5 213
      case 0x00D6 : // '' : // 0xD6 214
      case 0x014C :
      case 0x014E :
      case 0x0150 :
      case 0x01A0 :
      case 0x01D1 :
      case 0x01FE :
      case 0x020C :
      case 0x020E :
      case 0x022A :
      case 0x022C :
      case 0x022E :
      case 0x0230 :
         return( 'O' ) ;
      case 0x01EA :
      case 0x01EC :
         return( 'Q' ) ;
      case 0x0154 :
      case 0x0156 :
      case 0x0158 :
         return( 'R' ) ;
      case 0x008A : // '' : // 0x8A 138
      case 0x015A :
      case 0x015C :
      case 0x015E :
      case 0x0160 :
         return( 'S' ) ;
      case 0x0162 :
      case 0x0164 :
      case 0x0166 :
      case 0x021A :
         return( 'T' ) ;
      case 0x00D9 : // '' : // 0xD9 217
      case 0x00DA : // '' : // 0xDA 218
      case 0x00DB : // '' : // 0xDB 219
      case 0x00DC : // '' : // 0xDC 220
      case 0x0168 :
      case 0x016A :
      case 0x016C :
      case 0x016E :
      case 0x0170 :
      case 0x0172 :
      case 0x01AF :
      case 0x01D3 :
      case 0x01D5 :
      case 0x01D7 :
      case 0x01D9 :
      case 0x01DB :
      case 0x0214 :
      case 0x0216 :
         return( 'U' ) ;
      case 0x00DD : // '' : // 0xDD 221
      case 0x009F : // '' : // 0x9F 159
      case 0x0176 :
      case 0x0178 :
      case 0x0232 :
         return( 'Y' ) ;
      case 0x0174 :
         return( 'W' ) ;
      case 0x008E : // '' : // 0x8E 142
      case 0x0179 :
      case 0x017B :
      case 0x017D :
      case 0x0224 :
         return( 'Z' ) ;
      case 0x01E3 :
      case 0x01FD :
         return( 0x00E6 ) ; // e in a
      case 0x00E0 : // '' : // 0xE0 224
      case 0x00E1 : // '' : // 0xE1 225
      case 0x00E2 : // '' : // 0xE2 226
      case 0x00E3 : // '' : // 0xE3 227
      case 0x00E4 : // '' : // 0xE4 228
      case 0x00E5 : // '' : // 0xE5 229
      case 0x0101 :
      case 0x0103 :
      case 0x0105 :
      case 0x01CE :
      case 0x01DF :
      case 0x01E1 :
      case 0x01FB :
      case 0x0201 :
      case 0x0203 :
      case 0x0227 :
         return( 'a' ) ;
      case 0x00E7 : // '' : // 0xE7 231
      case 0x0107 :
      case 0x0109 :
      case 0x010B :
      case 0x010D :
      case 0x0188 :
         return( 'c' ) ;
      case 0x010F :
      case 0x0111 :
         return( 'd' ) ;
      case 0x00E8 : // '' : // 0xE8 232
      case 0x00E9 : // '' : // 0xE9 233
      case 0x00EA : // '' : // 0xEA 234
      case 0x00EB : // '' : // 0xEB 235
      case 0x0113 :
      case 0x0115 :
      case 0x0117 :
      case 0x0119 :
      case 0x011B :
      case 0x0205 :
      case 0x0207 :
      case 0x0229 :
         return( 'e' ) ;
      case 0x011D :
      case 0x011F :
      case 0x0121 :
      case 0x0123 :
      case 0x01E5 :
      case 0x01E7 :
      case 0x01F5 :
         return( 'g' ) ;
      case 0x0125 :
      case 0x0127 :
      case 0x021F :
         return( 'h' ) ;
      case 0x00EC : // '' : // 0xEC 236
      case 0x00ED : // '' : // 0xED 237
      case 0x00EE : // '' : // 0xEE 238
      case 0x00EF : // '' : // 0xEF 239
      case 0x0129 :
      case 0x012B :
      case 0x012D :
      case 0x012F :
      case 0x01D0 :
      case 0x0209 :
      case 0x020B :
         return( 'i' ) ;
      case 0x0135 :
      case 0x01F0 :
         return( 'j' ) ;
      case 0x0137 :
      case 0x0138 :
      case 0x01E9 :
         return( 'k' ) ;
      case 0x013A :
      case 0x013C :
      case 0x013E :
      case 0x0140 :
      case 0x0142 :
         return( 'l' ) ;
      case 0x00F1 : // '' : // 0xF1 241
      case 0x0144 :
      case 0x0146 :
      case 0x0148 :
      case 0x0149 :
      case 0x014B :
      case 0x01F9 :
         return( 'n' ) ;
      case 0x00F2 : // '' : // 0xF2 242
      case 0x00F3 : // '' : // 0xF3 243
      case 0x00F4 : // '' : // 0xF4 244
      case 0x00F5 : // '' : // 0xF5 245
      case 0x00F6 : // '' : // 0xF6 246
      case 0x014D :
      case 0x014F :
      case 0x0151 :
      case 0x01A1 :
      case 0x01D2 :
      case 0x01FF :
      case 0x020D :
      case 0x020F :
      case 0x022B :
      case 0x022D :
      case 0x022F :
      case 0x0231 :
         return( 'o' ) ;
      case 0x01EB :
      case 0x01ED :
         return( 'q' ) ;
      case 0x0155 :
      case 0x0157 :
      case 0x0159 :
         return( 'r' ) ;
      case 0x009A : // '' : // 0x9A 154
      case 0x015B :
      case 0x015D :
      case 0x015F :
      case 0x0161 :
         return( 's' ) ;
      case 0x0163 :
      case 0x0165 :
      case 0x0167 :
      case 0x021B :
         return( 't' ) ;
      case 0x00F9 : // '' : // 0xF9 249
      case 0x00FA : // '' : // 0xFA 250
      case 0x00FB : // '' : // 0xFB 251
      case 0x00FC : // '' : // 0xFC 252
      case 0x0169 :
      case 0x016B :
      case 0x016D :
      case 0x016F :
      case 0x0171 :
      case 0x0173 :
      case 0x01B0 :
      case 0x01D4 :
      case 0x01D6 :
      case 0x01D8 :
      case 0x01DA :
      case 0x01DC :
      case 0x0215 :
      case 0x0217 :
         return( 'u' ) ;
      case 0x00FD : // '' : // 0xFD 253
      case 0x00FF : // '' : // 0xFF 255
      case 0x0177 :
      case 0x0233 :
         return( 'y' ) ;
      case 0x0175 :
         return( 'w' ) ;
      case 0x009E : // '' : // 0x9E 158
      case 0x017A :
      case 0x017C :
      case 0x017E :
      case 0x0225 :
         return( 'z' ) ;
      /*-------------------------------------------------------------------*/
   } ;

   /*----------------------------------------------------------------------*/
   return( c ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString sr::unaccent( const wxString &s )
{
   /*----------------------------------------------------------------------*/
   wxString         s_res ;
   sr::wxString_cit it_s  ;
   /*----------------------------------------------------------------------*/
   s_res.reserve( s.size() ) ;
   /*----------------------------------------------------------------------*/
   for( it_s = s.begin() ; it_s != s.end() ; ++it_s )
   {  s_res += sr::unaccent( *it_s ) ; }
   /*----------------------------------------------------------------------*/
   return( s_res ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Replace %XX (X=hexadecimal character) substrings with the corresponding   !
! value.                                                                    !
! The conversion is done via "wxConvAuto"                                   !
! Example: bb%41cc => bbAcc                                                 !
+--------------------------------------------------------------------------*/
wxString sr::replace_percent_xx( const wxString &s, e_cls cls )
{
   /*----------------------------------------------------------------------*/
   wxCharBuffer   cb_from   ;
   wxMemoryBuffer mb_to     ;
   size_t         sz_length ;
   size_t         sz_num    ;
   char           c_xnum1   ;
   char           c_xnum2   ;

   /*----------------------------------------------------------------------*/
   cb_from   = s.mb_str()        ;
   sz_length = cb_from.length()  ;
   mb_to.SetBufSize( sz_length ) ;
   /*----------------------------------------------------------------------*/
   for( sz_num = 0 ; sz_num < sz_length ; )
   {
      /*-------------------------------------------------------------------*/
      if(    cb_from[ sz_num ] == '%'
          && sz_length - sz_num >= 3
          && isxdigit( c_xnum1 = cb_from[ sz_num + 1 ] )
          && isxdigit( c_xnum2 = cb_from[ sz_num + 2 ] )
        )
      {  mb_to.AppendByte(   ( sr::xchar_value( c_xnum1 ) << 4 )
                           | sr::xchar_value( c_xnum2 )
                         ) ;
         sz_num += 3 ;
      }
      else
      {  mb_to.AppendByte( cb_from[ sz_num++ ] ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*--( If something has been done, the size has been reduced )-----------*/
   if( mb_to.GetDataLen() == cb_from.length() )
   {  return( s ) ; }

   /*--( The conversion may have introduce control characters )------------*/
   wxString s_res( ( char * )mb_to.GetData(), wxConvAuto(),
                   mb_to.GetDataLen()
                 ) ;
   if( cls == sr::CLS_DO_CLEAN_STRING )
   {  sr::clean_string( s_res ) ; }
   /*----------------------------------------------------------------------*/
   return( s_res ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! During file transfers between non UTF8 to UTF8 file systems (MSW to       !
! Linux for example) characters interpretation may be wrongly done.         !
! Finally what can be seen is utf8 sequences represented as their ascii     !
! values.                                                                   !
! For example, for a "e grave": "Upper-case A tilda" and a "copyright" sign !
+--------------------------------------------------------------------------*/
wxString sr::convert_ascii_utf8( const wxString &s, e_cls cls )
{
   /*----------------------------------------------------------------------*/
   wxScopedCharBuffer scb_au( s.To8BitData() ) ;
   /*--( If the conversion doesn't work the returned string is empty )-----*/
   wxString s_res( wxString::FromUTF8( scb_au.data(), scb_au.length() ) ) ;
   /*--( In case of problem the source is returned )-----------------------*/
   if( s_res.empty() ) { return( s ) ; }
   /*--( The conversion may have introduce control characters (not sure) )-*/
   if( cls == sr::CLS_DO_CLEAN_STRING )
   {  sr::clean_string( s_res ) ; }
   /*----------------------------------------------------------------------*/
   return( s_res ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Keep only one occurrence of each character in a string                    !
! ie: abbac => abc                                                          !
+--------------------------------------------------------------------------*/
wxString sr::keep_one_of( const wxString &s )
{
   /*----------------------------------------------------------------------*/
   wxString         s_res ;
   sr::wxString_cit it_s  ;
   /*----------------------------------------------------------------------*/
   s_res.reserve( s.size() ) ;
   /*----------------------------------------------------------------------*/
   for( it_s = s.begin() ; it_s != s.end() ; ++it_s )
   {
      /*-------------------------------------------------------------------*/
      if( s_res.find( *it_s ) == wxString::npos ) { s_res += *it_s ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( s_res ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Replacement of forbidden characters: OS dependent                         !
+--------------------------------------------------------------------------*/
void sr::replace_filename_forbidden_chars( wxString &s_name,
                                           const wxString &s_by
                                         )
{
   /*----------------------------------------------------------------------*/
   const wxString  &s_forbidden = wxFileName::GetForbiddenChars() ;
   const wxUniChar c_sep        = wxFileName::GetPathSeparator()  ;
   /*----------------------------------------------------------------------*/
   wxString        s_vol              ;
   wxString        s_path             ;
   sr::wxString_it it_path            ;
   bool            boo_update = false ;
   /*----------------------------------------------------------------------*/
   if( s_name.empty() ) { return ; }
   /*----------------------------------------------------------------------*/
   if( s_by.empty() ) { wxFAIL ; return ; }

   /*----------------------------------------------------------------------*/
   wxFileName::SplitVolume( s_name, &s_vol, &s_path ) ;

   /*----------------------------------------------------------------------*/
   for( it_path = s_path.begin() ; it_path != s_path.end() ; ++it_path )
   {
      /*-------------------------------------------------------------------*/
      if(    *it_path != c_sep
          && s_forbidden.find( *it_path ) != wxString::npos
        )
      {  *it_path = s_by[ 0 ] ; boo_update = true ; }
      /*-------------------------------------------------------------------*/
   }

   /*--( A modification has been done ? )----------------------------------*/
   if( boo_update )
   {
      /*-------------------------------------------------------------------*/
      if( s_vol.empty() )
      {  s_name.swap( s_path ) ; }
      else
      {  s_name = s_vol + wxFileName::GetVolumeSeparator() + s_path ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Delete characters : CASE SENSITIVE OPERATION                              !
+--------------------------------------------------------------------------*/
void sr::delete_chars( wxString &s, const wxString &s_char_list )
{
   /*----------------------------------------------------------------------*/
   sr::wxString_it it = s.begin() ;
   /*----------------------------------------------------------------------*/
   while( it != s.end() )
   {
      /*-------------------------------------------------------------------*/
      if( s_char_list.find( *it ) == wxString::npos )
      {  ++it ; }
      else
      {  it = s.erase( it ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Some characters can not be repeated: many times the same character        !
! CASE SENSITIVE OPERATION                                                  !
+--------------------------------------------------------------------------*/
void sr::delete_repeat_chars( wxString &s, const wxString &s_char_list )
{
   /*----------------------------------------------------------------------*/
   sr::wxString_it it = s.begin() ;
   wxUniChar       c_current      ;
   /*----------------------------------------------------------------------*/
   while( it != s.end() )
   {
      /*-------------------------------------------------------------------*/
      c_current = *it++ ;
      /*-------------------------------------------------------------------*/
      if( s_char_list.find( c_current ) != wxString::npos )
      {
         /*----------------------------------------------------------------*/
         while( it != s.end() && *it == c_current )
         {  it = s.erase( it ) ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString sr::lpad_ll_with_0( const wxLongLong &ll, int i_pad )
{
   /*----------------------------------------------------------------------*/
   wxString s_val = ll.ToString() ;
   /*----------------------------------------------------------------------*/
   if( i_pad > ( int )s_val.length() )
   {  s_val.Pad( i_pad - s_val.length(), '0', false ) ; }
   /*----------------------------------------------------------------------*/
   return( s_val ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString::size_type sr::ic_search_beg( const wxString &s,
                                       const wxString &s_what,
                                       wxString::size_type pos
                                     )
{
   /*----------------------------------------------------------------------*/
   if( pos + s_what.size() > s.size() || s_what.empty() )
   {  return( wxString::npos ) ; }
   /*----------------------------------------------------------------------*/
   sr::wxString_cit it     = s.begin() + pos         ;
   sr::wxString_cit it_end = s.end() - s_what.size() ;
   /*----------------------------------------------------------------------*/
   for( ; it <= it_end ; ++it )
   {
      /*-------------------------------------------------------------------*/
      if( sr::ic_comp( it, s_what.begin(), s_what.size() ) == 0 )
      {  return( it - s.begin() ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( wxString::npos ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Search for a string from the end )-----------------------------------*/
wxString::size_type sr::ic_search_end( const wxString &s,
                                       const wxString &s_what,
                                       wxString::size_type pos
                                     )
{
   /*----------------------------------------------------------------------*/
   int i_pos_max = s.size() - s_what.size() ;
   /*----------------------------------------------------------------------*/
   if( i_pos_max < 0 || pos >= s.size() || s_what.empty() )
   {  return( wxString::npos ) ; }
   /*--( Starting point )--------------------------------------------------*/
   sr::wxString_cit it = s.begin() + wxMin( ( int )pos, i_pos_max ) ;
   /*----------------------------------------------------------------------*/
   for( ; it >= s.begin() ; --it )
   {
      /*-------------------------------------------------------------------*/
      if( sr::ic_comp( it, s_what.begin(), s_what.size() ) == 0 )
      {  return( it - s.begin() ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( wxString::npos ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
bool sr::has_BOM16( const char *p_c, int i_len )
{
   /*----------------------------------------------------------------------*/
   return(    i_len >= 2
           && (    ( p_c[0] == '\xFE' && p_c[1] == '\xFF' ) // Big-endian
                || ( p_c[0] == '\xFF' && p_c[1] == '\xFE' ) // Litle-endian
              )
         ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Try to detect and convert an UTF16 character buffer )----------------*/
void sr::utf16_to_string( const char *p_c, int i_len, wxString &s )
{
   /*----------------------------------------------------------------------*/
   s.clear() ;
   /*----------------------------------------------------------------------*/
   if( i_len < 2 ) { return ; }
   /*----------------------------------------------------------------------*/
   if( sr::has_BOM16( p_c, i_len ) )
   {  s = wxString( p_c, wxConvAuto(), i_len ) ; }
   else
   if( p_c[ 0 ] == '\0' && p_c[ 1 ] != '\0' )
   {  s = wxString( p_c, wxMBConvUTF16BE(), i_len ) ; }
   else
   if( p_c[ 0 ] != '\0' && p_c[ 1 ] == '\0' )
   {  s = wxString( p_c, wxMBConvUTF16LE(), i_len ) ; }
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Extraction from the "it" position of an integer value.                    !
! If no integer value is detected the destination is not modified           !
+--------------------------------------------------------------------------*/
sr::wxString_cit sr::conv_str_to_int( sr::wxString_cit it,
                                      sr::wxString_cit it_end,
                                      int &i_val
                                    )
{
   /*----------------------------------------------------------------------*/
   bool boo_negative = false ;
   /*----------------------------------------------------------------------*/
   sr::next_non_space( it_end, it ) ;

   /*--( Is there a sign ? )-----------------------------------------------*/
   if( it == it_end ) { return( it ) ; }
   switch( ( unsigned int )*it )
   {  /*-------------------------------------------------------------------*/
      case '-' : boo_negative = true ;
      case '+' :
         ++it ;
         i_val = 0 ;
         break ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   sr::next_non_space( it_end, it ) ;
   /*----------------------------------------------------------------------*/
   if( !sr::bool_isdigit( *it ) ) { return( it ) ; }
   /*----------------------------------------------------------------------*/
   i_val = 0 ;
   do
   {  i_val = i_val * 10 + sr::char_value( *it ) ; ++it ;
   } while( it != it_end && sr::bool_isdigit( *it ) ) ;
   /*----------------------------------------------------------------------*/
   sr::next_non_space( it_end, it ) ;
   /*----------------------------------------------------------------------*/
   if( boo_negative ) { i_val *= -1 ; }
   /*----------------------------------------------------------------------*/
   return( it ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Extraction from the "it" position of a "duration" string:                 !
! [+-][hour h][min m][sec s]                                                !
! The computed value is a signed number of seconds.                         !
! If no value is present the destination is not modified.                   !
+--------------------------------------------------------------------------*/
sr::wxString_cit sr::conv_time_to_nb_sec( sr::wxString_cit it,
                                          sr::wxString_cit it_end,
                                          int &i_nb_sec
                                        )
{
   /*----------------------------------------------------------------------*/
   bool boo_negative = false ;
   int  boo_h        = false ;
   int  boo_m        = false ;
   int  boo_s        = false ;
   int  i_val                ;
   bool boo_end      = false ;

   /*----------------------------------------------------------------------*/
   sr::next_non_space( it_end, it ) ;

   /*--( Is there a sign ? )-----------------------------------------------*/
   if( it == it_end ) { return( it ) ; }
   switch( ( *it ).GetValue() )
   {  /*-------------------------------------------------------------------*/
      case '-' : boo_negative = true ;
      case '+' :
         ++it ;
         i_nb_sec = 0 ;
         break ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   sr::next_non_space( it_end, it ) ;

   /*----------------------------------------------------------------------*/
   if( !sr::bool_isdigit( *it ) ) { return( it ) ; }
   /*----------------------------------------------------------------------*/
   i_nb_sec = 0 ;
   /*-----------------------------------------------------------------------+
   ! Extract the hour/minute/second                                         !
   ! They have to be defined in this order but maybe not all of them        !
   +-----------------------------------------------------------------------*/
   do
   {
      /*--( Extraction of the numeric value )------------------------------*/
      for( i_val = 0 ; it != it_end && sr::bool_isdigit( *it ) ; ++it )
      {  i_val = i_val * 10 + sr::char_value( *it ) ; }
      /*--------------------------------------------------------------------+
      ! Control of the order of the information                             !
      +--------------------------------------------------------------------*/
      if(    it == it_end
          || ( *it == 'h' && ( boo_h || boo_m || boo_s ) )
          || ( *it == 'm' && ( boo_m || boo_s ) )
          || ( *it == 's' && ( boo_s ) )
        )
      {  boo_end = true ; }
      else
      switch( ( *it ).GetValue() )
      {  case 'h' : i_nb_sec += 3600 * i_val ; boo_h = true ; break ;
         case 'm' : i_nb_sec +=   60 * i_val ; boo_m = true ; break ;
         case 's' : i_nb_sec +=        i_val ; boo_s = true ; break ;
         default : i_nb_sec += i_val ; boo_end = true ; break ;
      }
      /*-------------------------------------------------------------------*/
      if( !boo_end ) { ++it ; }
      /*-------------------------------------------------------------------*/
   } while( !boo_end ) ;

   /*----------------------------------------------------------------------*/
   sr::next_non_space( it_end, it ) ;
   /*----------------------------------------------------------------------*/
   if( boo_negative ) { i_nb_sec *= -1 ; }
   /*----------------------------------------------------------------------*/
   return( it ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! i_start_occ: Starting occurrence (< 0 means backwards)                    !
! i_nb_rep: Number of replacement to do (< 0 means all after i_start_occ)   !
+--------------------------------------------------------------------------*/
void sr::replace_string( wxString &s,
                         const wxString &s_from, const wxString &s_to,
                         bool boo_case_sens, int i_start_occ, int i_nb_rep
                       )
{
   /*----------------------------------------------------------------------*/
   if( s_from.empty() || i_start_occ == 0 || i_nb_rep == 0 ) { return ; }

   /*----------------------------------------------------------------------*/
   int i_pos ;
   /*-----------------------------------------------------------------------+
   ! Find the first occurrence to replace. The "0" case has been eliminated !
   +-----------------------------------------------------------------------*/
   if( i_start_occ > 0 )
   {  /*-------------------------------------------------------------------*/
      i_pos = 0 ;
      /*-------------------------------------------------------------------*/
      while(    i_start_occ != 0
             && (    (    boo_case_sens
                       && ( i_pos = s.find( s_from, i_pos )
                          ) != ( int )wxString::npos
                     )
                  || (    !boo_case_sens
                       && ( i_pos = sr::ic_search_beg( s, s_from, i_pos )
                          ) != ( int )wxString::npos
                     )
                )
           )
      {  if( --i_start_occ != 0 ) { i_pos += s_from.size() ; } }
      /*-------------------------------------------------------------------*/
   }
   else
   {  /*--( From the end )-------------------------------------------------*/
      i_pos = s.size() - 1 ;
      /*-------------------------------------------------------------------*/
      while(    i_start_occ != 0
             && (    (     boo_case_sens
                        && ( i_pos = s.rfind( s_from, i_pos )
                           ) != ( int )wxString::npos
                     )
                  || (     !boo_case_sens
                        && ( i_pos = sr::ic_search_end( s, s_from, i_pos )
                           ) != ( int )wxString::npos
                     )
                )
           )
      {  if( ++i_start_occ != 0 ) { i_pos -= s_from.size() ; } }
      /*-------------------------------------------------------------------*/
   }

   /*--( Replace what has been asked. Negative => all remaining occ )------*/
   while( i_nb_rep != 0 && i_pos != ( int )wxString::npos )
   {
      /*-------------------------------------------------------------------*/
      s.replace( i_pos, s_from.size(), s_to ) ;
      /*-------------------------------------------------------------------*/
      if( --i_nb_rep != 0 )
      {  /*--( Skip the replacement string )-------------------------------*/
         i_pos += s_to.size() ;
         /*----------------------------------------------------------------*/
         if( boo_case_sens )
         {  i_pos = s.find( s_from, i_pos ) ; }
         else
         {  i_pos = sr::ic_search_beg( s, s_from, i_pos ) ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Replacement of the characters in "s_from" by the one at the same place    !
! in "s_to".                                                                !
! If the "to" is smaller than "from" then the last "to" character is used.  !
! CASE SENSITIVE OPERATION                                                  !
+--------------------------------------------------------------------------*/
void sr::replace_chars( wxString &s,
                        const wxString &s_from, const wxString &s_to
                      )
{
   /*----------------------------------------------------------------------*/
   sr::wxString_it it                          ;
   const int       co_i_lg_to = s_to.size()    ;
   const wxUniChar c_last_to  = *s_to.rbegin() ;
   int             i_pos                       ;
   /*----------------------------------------------------------------------*/
   for( it = s.begin() ; it != s.end() ; ++it )
   {  /*-------------------------------------------------------------------*/
      if( ( i_pos = s_from.find( *it ) ) != ( int )wxString::npos )
      {  /*----------------------------------------------------------------*/
         *it = ( i_pos < co_i_lg_to ? s_to[ i_pos ] : c_last_to ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! CASE SENSITIVE OPERATION                                                  !
+--------------------------------------------------------------------------*/
void sr::replace_chars( wxString &s,
                        const wxUniChar &c_from, const wxUniChar &c_to
                      )
{
   /*----------------------------------------------------------------------*/
   sr::wxString_it it ;
   /*----------------------------------------------------------------------*/
   for( it = s.begin() ; it != s.end() ; ++it )
   {
      /*-------------------------------------------------------------------*/
      if( *it == c_from ) { *it = c_to ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*--( Returns the length of the cleaned string )---------------------------*/
int sr::clean_string( wxString &s )
{
   /*-----------------------------------------------------------------------+
   ! Working on the original can make the exe crash ... so a copy is made   !
   +-----------------------------------------------------------------------*/
   wxString         s_out ;
   sr::wxString_cit it    ;
   /*----------------------------------------------------------------------*/
   s_out.reserve( s.size() ) ;

   /*----------------------------------------------------------------------*/
   for( it = s.begin() ;
        it != s.end() && s_out.size() < ( size_t )g_co_i_string_sz_max ;
        ++it
      )
   {
      /*--------------------------------------------------------------------+
      ! Characters 0x7F..0x9f are seen as control characters. They are      !
      ! valid in some encodings. So only the test 0x00..0x1F is used.       !
      +--------------------------------------------------------------------*/
      if( *it >= 0x20 )
      {  s_out += *it ; }
      else
      {  /*-----------------------------------------------------------------+
         ! Replace control chars with spaces except for the CRLF sequence   !
         ! which is replaced only with one.                                 !
         +-----------------------------------------------------------------*/
         if( *it != '\n' || it == s.begin() || *( it - 1 ) != '\r' )
         {  s_out += ' ' ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( Copy the trimmed result )-----------------------------------------*/
   s = s_out.Trim() ;
   /*----------------------------------------------------------------------*/
   return( s.size() ) ;
   /*----------------------------------------------------------------------*/
}

/*--( It returns the length of the prepared string )-----------------------*/
int sr::prepare_string( const char *p_c_val, int i_len, wxString &s,
                        sr::e_conv_type conv_type_local
                      )
{
   /*----------------------------------------------------------------------*/
   s.clear() ;
   /*-----------------------------------------------------------------------+
   ! The "From" functions will "insert" '\0' if they are before their       !
   ! length param (i_len).                                                  !
   ! "strlen" is not used because the array may not be null terminated.     !
   !                                                                        !
   ! The "i_len" is supposed to be expressed in number of bytes.            !
   +-----------------------------------------------------------------------*/
   if(    conv_type_local == sr::CONV_FROM_UTF16LE
       || conv_type_local == sr::CONV_FROM_UTF16BE
       || conv_type_local == sr::CONV_FROM_UTF16LEBE
     )
   {  /*-------------------------------------------------------------------*/
      int i_val_len ;
      for( i_val_len = 0 ;
              i_val_len < i_len
           && *( wxUint16 * )( p_c_val + i_val_len ) != 0 ;
           i_val_len += 2
         )
      {  ; }
      /*-------------------------------------------------------------------*/
      i_len = i_val_len ;
      /*-------------------------------------------------------------------*/
   }
   else
   {  /*-------------------------------------------------------------------*/
      int i_val_len ;
      for( i_val_len = 0 ;
           i_val_len < i_len && p_c_val[ i_val_len ] != '\0' ;
           ++i_val_len
         )
      {  ; }
      /*-------------------------------------------------------------------*/
      i_len = i_val_len ;
      /*-------------------------------------------------------------------*/
   }

   /*--( Minimal control for speed and avoid crash in WIDE conversion )----*/
   if( i_len <= ( conv_type_local == sr::CONV_FROM_UTF16LEBE ? 2 : 0 ) )
   {  return( 0 ) ; }

   /*----------------------------------------------------------------------*/
   switch( conv_type_local )
   {
      /*-------------------------------------------------------------------*/
      case sr::CONV_FROM_8BIT :
         s = wxString::From8BitData( p_c_val, i_len ) ;
         break ;
      case sr::CONV_FROM_UTF8 :
         /*----------------------------------------------------------------*/
         s = wxString::FromUTF8( p_c_val, i_len ) ;
         if( s.empty() )
         {  s = wxString::From8BitData( p_c_val, i_len ) ; }
         /*----------------------------------------------------------------*/
         break ;
      /*--( Force BOM little endian )--------------------------------------*/
      case sr::CONV_FROM_UTF16LE :
         /*--( The number of char must be even ... )-----------------------*/
         s = wxString( p_c_val, wxMBConvUTF16LE(), i_len / 2 * 2 ) ;
         /*----------------------------------------------------------------*/
         break ;
      /*--( Force BOM big endian )-----------------------------------------*/
      case sr::CONV_FROM_UTF16BE :
         /*--( The number of char must be even ... )-----------------------*/
         s = wxString( p_c_val, wxMBConvUTF16BE(), i_len / 2 * 2 ) ;
         /*----------------------------------------------------------------*/
         break ;
      /*--( The string is supposed to start with the BOM )-----------------*/
      case sr::CONV_FROM_UTF16LEBE :
         /*--( The number of char must be even and greater than 2 (BOM) )--*/
         s = wxString( p_c_val, wxConvAuto(), i_len / 2 * 2 ) ;
         /*----------------------------------------------------------------*/
         break ;
      /*-------------------------------------------------------------------*/
      case sr::CONV_FROM_MAC :
         s = wxString( p_c_val, wxCSConv( wxFONTENCODING_MACROMAN ), i_len );
         break ;
      /*-------------------------------------------------------------------*/
      default :
         wxFAIL ;
         break ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( sr::clean_string( s ) ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Function used by the case change functions to find "words" in a string.   !
! It will "analyse/change" the string given as first parameter.             !
! It returns true if the word found belongs to the forced case words list.  !
! In this case the string word is changed to the expected cased one.        !
+--------------------------------------------------------------------------*/
static bool st_next_force_case_word( wxString &s,
                                     bool boo_jump_force_case_word,
                                     int &i_beg, int &i_nb
                                   )
{
   /*----------------------------------------------------------------------*/
   int i_word ;
   int i_end  ;

   /*--( Jump over the separator, if any )---------------------------------*/
   if(   i_beg > 0
      || wxGetApp().M_s_chars_word_sep.get().find( s[ 0 ] ) != wxString::npos
     )
   {  ++i_beg ; }

   /*--( Extraction of the coordinates of the next word )------------------*/
   i_end = s.find_first_of( wxGetApp().M_s_chars_word_sep.get(), i_beg ) ;
   if( i_end == ( int )wxString::npos )
   {  i_end = s.size() ; }
   i_nb = i_end - i_beg ;

   /*--( This word is in our list ? )--------------------------------------*/
   i_word = wxGetApp().M_as_force_case_word.Index( s.substr( i_beg, i_nb ),
                                                   false
                                                 ) ;
   if( i_word != wxNOT_FOUND )
   {
      /*--( The word has to be in defined case )---------------------------*/
      s.replace( i_beg, i_nb, wxGetApp().M_as_force_case_word[ i_word ] ) ;
      if( boo_jump_force_case_word )
      {  i_beg += i_nb ; i_nb = 0 ; }
      /*-------------------------------------------------------------------*/
      return( true ) ;
      /*-------------------------------------------------------------------*/
   }

   /*--( "Unknown" word found )--------------------------------------------*/
   return( false ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! "fwc": stands for "Force case word" for the words defined that have to    !
! respect a particular case.                                                !
! Only the case of the words that have been asked for are changed.          !
+--------------------------------------------------------------------------*/
void sr::fcw_update( wxString &s )
{
   /*----------------------------------------------------------------------*/
   int i_beg = 0 ;
   int i_nb      ;

   /*----------------------------------------------------------------------*/
   while( i_beg < ( int )s.size() )
   {
      /*-------------------------------------------------------------------*/
      st_next_force_case_word( s, true, i_beg, i_nb ) ;
      i_beg += i_nb ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static void st_upper( sr::wxString_it it, int i_nb )
{
   /*----------------------------------------------------------------------*/
   for( ; i_nb > 0 ; ++it, --i_nb ) { *it = wxToupper( *it ) ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
static void st_lower( sr::wxString_it it, int i_nb )
{
   /*----------------------------------------------------------------------*/
   for( ; i_nb > 0 ; ++it, --i_nb ) { *it = wxTolower( *it ) ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void sr::fcw_upper( wxString &s, e_fcw fcw_force )
{
   /*----------------------------------------------------------------------*/
   if( fcw_force == FCW_DONT_FORCE_CASE )
   {  st_upper( s.begin(), s.size() ) ; }
   else
   {  /*-------------------------------------------------------------------*/
      int i_beg = 0 ;
      int i_nb      ;
      /*-------------------------------------------------------------------*/
      while( i_beg < ( int )s.size() )
      {
         /*----------------------------------------------------------------*/
         st_next_force_case_word( s, true, i_beg, i_nb ) ;
         st_upper( s.begin() + i_beg, i_nb ) ;
         i_beg += i_nb ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void sr::fcw_upper_first_word_letter( wxString &s, e_fcw fcw_force )
{
   /*----------------------------------------------------------------------*/
   int             i_beg = 0 ;
   int             i_nb      ;
   sr::wxString_it it        ;

   /*----------------------------------------------------------------------*/
   while( i_beg < ( int )s.size() )
   {
      /*--( The order is important. The function must be called !! )-------*/
      if(    st_next_force_case_word( s, false, i_beg, i_nb )
          && fcw_force == FCW_DO_FORCE_CASE
        )
      {  i_beg += i_nb ; i_nb = 0 ; }
      /*-------------------------------------------------------------------*/
      if( i_nb > 0 )
      {  /*----------------------------------------------------------------*/
         it  = s.begin() + i_beg ;
         *it = wxToupper( *it ) ;
         ++it ; --i_nb ; ++i_beg ;
         /*----------------------------------------------------------------*/
         st_lower( it, i_nb ) ;
         i_beg += i_nb ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void sr::fcw_lower( wxString &s, e_fcw fcw_force )
{
   /*----------------------------------------------------------------------*/
   if( fcw_force == FCW_DONT_FORCE_CASE )
   {  st_lower( s.begin(), s.size() ) ; }
   else
   {  /*-------------------------------------------------------------------*/
      int i_beg = 0 ;
      int i_nb      ;
      /*-------------------------------------------------------------------*/
      while( i_beg < ( int )s.size() )
      {
         /*----------------------------------------------------------------*/
         st_next_force_case_word( s, true, i_beg, i_nb ) ;
         st_lower( s.begin() + i_beg, i_nb ) ;
         i_beg += i_nb ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void sr::fcw_upper_first_letter_lower_rest( wxString &s, e_fcw fcw_force )
{
   /*----------------------------------------------------------------------*/
   sr::wxString_it it ;

   /*----------------------------------------------------------------------*/
   if( s.empty() ) { return ; }

   /*----------------------------------------------------------------------*/
   if( fcw_force == FCW_DONT_FORCE_CASE )
   {  /*-------------------------------------------------------------------*/
      it  = s.begin() ;
      *it = wxToupper( *it ) ;
      st_lower( ++it, s.size() - 1 ) ;
      /*-------------------------------------------------------------------*/
   }
   else
   {  /*-------------------------------------------------------------------*/
      int i_beg = 0 ;
      int i_nb      ;
      /*-------------------------------------------------------------------*/
      while( i_beg < ( int )s.size() )
      {
         /*----------------------------------------------------------------*/
         st_next_force_case_word( s, true, i_beg, i_nb ) ;
         /*----------------------------------------------------------------*/
         it = s.begin() + i_beg ;
         if( i_beg == 0 && i_nb > 0 )
         {  *it = wxToupper( *it ) ;
            ++it ; ++i_beg ; --i_nb ;
         }
         /*----------------------------------------------------------------*/
         st_lower( it, i_nb ) ;
         i_beg += i_nb ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void sr::prepend_with_sep( const wxString &s_toadd, const wxString &s_sep,
                           wxString &s_result
                         )
{
   /*----------------------------------------------------------------------*/
   if( s_toadd.empty() ) { return ; }
   /*----------------------------------------------------------------------*/
   if( !s_result.empty() )
   {  s_result.insert( 0, s_sep ) ; }
   /*----------------------------------------------------------------------*/
   s_result.insert( 0, s_toadd ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void sr::append_with_sep( const wxString &s_toadd, const wxString &s_sep,
                          wxString &s_result
                        )
{
   /*----------------------------------------------------------------------*/
   if( s_toadd.empty() ) { return ; }
   /*----------------------------------------------------------------------*/
   if( !s_result.empty() )
   {  s_result.append( s_sep ) ; }
   /*----------------------------------------------------------------------*/
   s_result.append( s_toadd ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString sr::conv_to_hex( void *p_v, size_t sz )
{
   /*----------------------------------------------------------------------*/
   unsigned char *p_b = ( unsigned char * )p_v ;
   wxString      s ;
   /*----------------------------------------------------------------------*/
   for( ; sz > 0 ; --sz )
   {  s += wxString::Format( "%02X", *p_b++ ) ; }
   /*----------------------------------------------------------------------*/
   return( s ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Dump byte buffer )---------------------------------------------------*/
wxString sr::byte_to_string( wxUint8 b )
{  return( wxString::Format( isprint( b ) ? "%c" : "%0x02X", b ) ) ; }

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                          End of file sr_string.cpp                        !
+==========================================================================*/
