/*====================================+=====================================+
! File CShortcut.cpp                  ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!     Management of Windows file shortcuts (Desktop, Quick Launch ...)      !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#ifdef __WXMSW__
/*-------------------------------------------------------------------------*/
#include "common/msw/CShortcut.h"
#include <shlobj.h>
/*-------------------------------------------------------------------------*/



/*--------------------------------------------------------------------------+
! I don't use SHGetSpecialFolderPath because it is "only" present in        !
! shell32.dll starting version 4.71                                         !
+--------------------------------------------------------------------------*/
static int st_compute_path( CShortcut::e_shortcut scut, wxString &s_path )
{
   /*----------------------------------------------------------------------*/
   LPMALLOC      pMalloc  = NULL ;
   LPSHELLFOLDER pDesktop = NULL ;
   LPITEMIDLIST  pidl     = NULL ;
   STRRET        str_strret      ;
   int           i_ret    = 0    ;
   int           i_csidl         ;
   HRESULT       hres            ;

   /*----------------------------------------------------------------------*/
   switch( scut )
   {  /*-------------------------------------------------------------------*/
      case CShortcut::SCUT_SENDTO  : i_csidl = CSIDL_SENDTO ; break ;
      case CShortcut::SCUT_QUICK   : i_csidl = CSIDL_APPDATA ; break ;
      case CShortcut::SCUT_DESKTOP : i_csidl = CSIDL_DESKTOPDIRECTORY; break;
      case CShortcut::SCUT_PROGRAMS: i_csidl = CSIDL_PROGRAMS ; break ;
      case CShortcut::SCUT_START   : i_csidl = CSIDL_STARTMENU ; break ;
      default : return( -2 ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   hres = SHGetMalloc( &pMalloc ) ;
   if( FAILED( hres ) ) { i_ret = -3 ; goto func_end ; }
   /*----------------------------------------------------------------------*/
   hres = SHGetDesktopFolder( &pDesktop ) ;
   if( FAILED( hres ) ) { i_ret = -4 ; goto func_end ; }
   /*----------------------------------------------------------------------*/
   hres = SHGetSpecialFolderLocation( NULL, i_csidl, &pidl ) ;
   if( FAILED( hres ) ) { i_ret = -5 ; goto func_end ; }
   /*--( Finally get the name .... )---------------------------------------*/
   hres = pDesktop->GetDisplayNameOf( pidl, SHGDN_FORPARSING, &str_strret ) ;
   if( FAILED( hres ) ) { i_ret = -6 ; goto func_end ; }

   /*--( What is its format ? )--------------------------------------------*/
   switch( str_strret.uType )
   {
      /*--( Standard string )----------------------------------------------*/
      case STRRET_CSTR :
         s_path = wxString::From8BitData( str_strret.cStr ) ;
         break ;
      /*--( Standard string with an offset )-------------------------------*/
      case STRRET_OFFSET :
           s_path
         = wxString::From8BitData( ( char * )pidl + str_strret.uOffset ) ;
         break ;
      /*--( Unicode string ... )-------------------------------------------*/
      case STRRET_WSTR :
         /*----------------------------------------------------------------*/
         s_path = wxString( str_strret.pOleStr, wxMBConvUTF16LE() ) ;
         /*--( Memory was allocated by the system, it has to be freed )----*/
         pMalloc->Free( str_strret.pOleStr ) ;
         /*----------------------------------------------------------------*/
         break ;
      /*-------------------------------------------------------------------*/
      default : i_ret = -7 ; goto func_end ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   if( s_path.Last() != '\\' )
   {  s_path += '\\' ; }
   /*--( Complement ... )--------------------------------------------------*/
   if( scut == CShortcut::SCUT_QUICK )
   {  s_path += "Microsoft\\Internet Explorer\\Quick Launch\\" ; }
   /*----------------------------------------------------------------------*/
func_end :
   /*----------------------------------------------------------------------*/
   if( pMalloc != NULL )
   {
      /*-------------------------------------------------------------------*/
      if( pidl     != NULL ) { pMalloc->Free( pidl ) ; }
      if( pDesktop != NULL ) { pDesktop->Release()   ; }
      pMalloc->Release() ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CShortcut::init_name( e_shortcut scut, const wxString &s_base_name )
{
   /*----------------------------------------------------------------------*/
   wxString s_path ;
   int      i_ret  ;
   /*----------------------------------------------------------------------*/
   if( s_base_name.empty() ) { return( -10 ) ; }
   /*----------------------------------------------------------------------*/
   if( ( i_ret = st_compute_path( scut, s_path ) ) != 0 )
   {  return( i_ret ) ; }
   /*----------------------------------------------------------------------*/
   m_s_name = s_path + s_base_name ;
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*--( CoInitialize must have been called )---------------------------------*/
int CShortcut::load()
{
   /*----------------------------------------------------------------------*/
   IShellLink   *p_sl = NULL ;
   IPersistFile *p_pf = NULL ;
   HRESULT      hres         ;
   int          i_ret = 0    ;

   /*----------------------------------------------------------------------*/
   if( m_s_name.empty() ) { i_ret = -1 ; goto func_end ; }

   /*--( Clean data ... )--------------------------------------------------*/
   m_s_target.clear() ;
   m_s_desc.clear()   ;

   /*-----------------------------------------------------------------------+
   ! Create an IShellLink object and get a pointer to the IShellLink        !
   ! interface (returned from CoCreateInstance).                            !
   +-----------------------------------------------------------------------*/
   hres = CoCreateInstance( CLSID_ShellLink, NULL, CLSCTX_INPROC_SERVER,
                            IID_IShellLink, ( void ** )&p_sl
                          ) ;
   /*----------------------------------------------------------------------*/
   if( FAILED( hres ) ) { i_ret = -2 ; goto func_end ; }

   /*-----------------------------------------------------------------------+
   ! Query IShellLink for the IPersistFile interface to save the shortcut   !
   ! in persistent storage                                                  !
   +-----------------------------------------------------------------------*/
   hres = p_sl->QueryInterface( IID_IPersistFile, ( void ** )&p_pf ) ;
   if( FAILED( hres ) ) { i_ret = -3 ; goto func_end ; }

   /*----------------------------------------------------------------------*/
   wxChar          tb_wc_tmp[ MAX_PATH + 1 ] ;
   WIN32_FIND_DATA ffd                       ;

   /*--( Loading shortcut information )------------------------------------*/
   hres = p_pf->Load( m_s_name.wx_str(), STGM_READ ) ;
   if( FAILED( hres ) ) { i_ret = -4 ; goto func_end ; }

   /*--( Get the associated path )-----------------------------------------*/
   hres = p_sl->GetPath( tb_wc_tmp, WXSIZEOF( tb_wc_tmp ), &ffd, 0 ) ;
   if( FAILED( hres ) ) { i_ret = -6 ; goto func_end ; }
   m_s_target = tb_wc_tmp ;

   /*--( Get the description )---------------------------------------------*/
   hres = p_sl->GetDescription( tb_wc_tmp, MAX_PATH ) ;
   if( FAILED( hres ) ) { i_ret = -7 ; goto func_end ; }
   m_s_desc = tb_wc_tmp ;
   /*----------------------------------------------------------------------*/
func_end :
   /*----------------------------------------------------------------------*/
   if( p_pf != NULL ) { p_pf->Release() ; }
   if( p_sl != NULL ) { p_sl->Release() ; }
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*--( CoInitialize must have been called )---------------------------------*/
int CShortcut::create_update()
{
   /*----------------------------------------------------------------------*/
   IShellLink   *p_sl = NULL ;
   IPersistFile *p_pf = NULL ;
   HRESULT      hres         ;
   int          i_ret = 0    ;

   /*----------------------------------------------------------------------*/
   if( m_s_name.empty() )   { i_ret = -1 ; goto func_end ; }
   /*----------------------------------------------------------------------*/
   if( m_s_target.empty() ) { i_ret = -2 ; goto func_end ; }

   /*-----------------------------------------------------------------------+
   ! Create an IShellLink object and get a pointer to the IShellLink        !
   ! interface (returned from CoCreateInstance).                            !
   +-----------------------------------------------------------------------*/
   hres = CoCreateInstance( CLSID_ShellLink, NULL, CLSCTX_INPROC_SERVER,
                            IID_IShellLink, (void **)&p_sl
                          ) ;
   /*----------------------------------------------------------------------*/
   if( FAILED( hres ) ) { i_ret = -3 ; goto func_end ; }

   /*-----------------------------------------------------------------------+
   ! Query IShellLink for the IPersistFile interface to save the shortcut   !
   ! in persistent storage                                                  !
   +-----------------------------------------------------------------------*/
   hres = p_sl->QueryInterface( IID_IPersistFile, ( void ** )&p_pf ) ;
   if( FAILED( hres ) ) { i_ret = -4 ; goto func_end ; }

   /*--( Load of the shortcut to not loose the associated info )-----------*/
   hres = p_pf->Load( m_s_name.wx_str(), STGM_READ ) ;
   if( FAILED( hres ) )
   {  /*--( The description has to be forced )-----------------------------*/
      hres = p_sl->SetDescription( m_s_desc.wx_str() ) ;
      if( FAILED( hres ) ) { i_ret = -5 ; goto func_end ; }
      /*-------------------------------------------------------------------*/
   }

   /*--( Set the target )--------------------------------------------------*/
   hres = p_sl->SetPath( m_s_target.wx_str() ) ;
   if( FAILED( hres ) ) { i_ret = -6 ; goto func_end ; }

   /*----------------------------------------------------------------------*/
   hres = p_pf->Save( m_s_name.wx_str(), TRUE ) ;
   if( FAILED( hres ) ) { i_ret = -7 ; goto func_end ; }
   /*----------------------------------------------------------------------*/
func_end :
   /*----------------------------------------------------------------------*/
   if( p_pf != NULL ) { p_pf->Release() ; }
   if( p_sl != NULL ) { p_sl->Release() ; }
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*--( CoInitialize must have been called )---------------------------------*/
int CShortcut::remove()
{
   /*----------------------------------------------------------------------*/
   if( m_s_name.empty() ) { return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   if( DeleteFile( m_s_name.wx_str() ) == 0 )
   {  return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
#endif // __WXMSW__
/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                        End of file CShortcut.cpp                          !
+==========================================================================*/
