/*====================================+=====================================+
! File CChecksum.cpp                  ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!               Checksum computation: MD5, SHA1 & CRC32                     !
!                                                                           !
+-------+-------------------------------------------------------------------+
! Notes !                                                                   !
+-------+                                                                   !
! This code is based on information found at these addresses:               !
!                                                                           !
! MD5:   http://en.wikipedia.org/wiki/Md5                                   !
! SHA1:  http://en.wikipedia.org/wiki/Sha1                                  !
! CRC32: http://www.codeproject.com/KB/recipes/crc32.aspx                   !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <wx/stopwatch.h>
#include "CApp.h"
#include "CFileAccess.h"
#include "common/CChecksum.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
bool CChecksumMD5::m_st_boo_init_done = false ;
wxUint32 CChecksumMD5::m_st_tb_dw_r[ 64 ]     ;
wxUint32 CChecksumMD5::m_st_tb_dw_k[ 64 ]     ;

/*-------------------------------------------------------------------------*/
bool CChecksumCRC32::m_st_boo_init_done = false ;
wxUint32 CChecksumCRC32::m_st_tb_crc32[ 256 ] ;

/*-------------------------------------------------------------------------*/
void CChecksum::result_to_string( wxString &s_result ) const
{
   /*----------------------------------------------------------------------*/
   int i_pos ;
   /*----------------------------------------------------------------------*/
   s_result.clear() ;
   s_result.reserve( m_i_result_len * 2 ) ;
   /*----------------------------------------------------------------------*/
   for( i_pos = 0 ; i_pos < m_i_result_len ; ++i_pos )
   {  s_result += wxString::Format( "%02x", m_tb_uc_result[ i_pos ] ) ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CChecksum::update( const wxFileName &filename,
                       t_fnc_progress_info p_fnc_progress
                     )
{
   /*----------------------------------------------------------------------*/
   CFileAccess   fa                       ;
   unsigned char tb_uc_buffer[ 128 * 64 ] ;
   size_t        sz_buffer_len            ;
   wxLongLong    ll_last_time             ;
   wxULongLong   ull_file_size            ;
   int           i_ret                    ;

   /*----------------------------------------------------------------------*/
   if( fa.open_to_read( filename ) != 0 )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
   ll_last_time  = wxGetLocalTimeMillis() ;
   ull_file_size = filename.GetSize()     ;
   /*----------------------------------------------------------------------*/
   sz_buffer_len = sizeof( tb_uc_buffer ) ;
   while(    fa.read_buffer_max( sz_buffer_len, tb_uc_buffer ) == 0
          && sz_buffer_len > 0
        )
   {
      /*-------------------------------------------------------------------*/
      add_data( tb_uc_buffer, sz_buffer_len ) ;

      /*--( Information ? (shouldn't be here if the file size is 0 ) )-----*/
      if(    p_fnc_progress != NULL
          && wxGetLocalTimeMillis() - ll_last_time > 1500
        )
      {
         /*----------------------------------------------------------------*/
         p_fnc_progress(
             wxString::Format( _( "Computing %s checksum (%d%%) of %s ..." ),
                               m_co_p_c_checksum_name,
                               ( int )(   fa.get_offset() * 100.0
                                        / ull_file_size.ToDouble()
                                      ),
                               filename.GetFullPath()
                             )
                       ) ;
         /*----------------------------------------------------------------*/
         ll_last_time = wxGetLocalTimeMillis() ;
         /*----------------------------------------------------------------*/

      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   i_ret = ( sz_buffer_len != 0 ? -2 : 0 ) ;
   /*----------------------------------------------------------------------*/
   fa.close() ;
   /*----------------------------------------------------------------------*/
   return( i_ret ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CChecksum::calc( const unsigned char *p_uc_buffer, size_t sz_buffer_len,
                      unsigned char *p_uc_result
                    )
{
   /*----------------------------------------------------------------------*/
   init() ;
   add_data( p_uc_buffer, sz_buffer_len ) ;
   final() ;
   /*----------------------------------------------------------------------*/
   copy_result( p_uc_result ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CChecksum::calc( const char *p_c_buffer, unsigned char *p_uc_result )
{
   /*----------------------------------------------------------------------*/
   calc( ( unsigned char * )p_c_buffer, strlen( p_c_buffer ), p_uc_result ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxString CChecksum::calc( const wxString &s_in )
{
   /*----------------------------------------------------------------------*/
   wxCharBuffer chb_in( s_in.mb_str() ) ;
   /*----------------------------------------------------------------------*/
   init() ;
   add_data( ( unsigned char * )chb_in.data(), chb_in.length() ) ;
   final() ;
   /*----------------------------------------------------------------------*/
   wxString s_result ;
   result_to_string( s_result ) ;
   /*----------------------------------------------------------------------*/
   return( s_result ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CChecksum::calc( const wxFileName &filename, wxString &s_result,
                     t_fnc_progress_info p_fnc_progress
                   )
{
   /*----------------------------------------------------------------------*/
   init() ;
   if( update( filename, p_fnc_progress ) != 0 ) { return( -1 ) ; }
   final() ;
   /*----------------------------------------------------------------------*/
   result_to_string( s_result ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CChecksumMD5::init()
{
   /*--( Done only once: during first instantiation )----------------------*/
   if( !m_st_boo_init_done )
   {
      /*-------------------------------------------------------------------*/
      wxUint32 tb_dw_init_r[ 4 ][ 4 ]
             = { { 7, 12, 17, 22 }, { 5,  9, 14, 20 },
                 { 4, 11, 16, 23 }, { 6, 10, 15, 21 }
               } ;
      int i_num ;
      int i_rep ;

      /*-------------------------------------------------------------------*/
      for( i_num = 0 ; i_num < 4 ; ++i_num )
      {  /*----------------------------------------------------------------*/
         for( i_rep = 0 ; i_rep < 4 ; ++i_rep )
         {  /*-------------------------------------------------------------*/
            memcpy( &m_st_tb_dw_r[ i_num * 16 + i_rep * 4 ],
                    tb_dw_init_r[ i_num ], sizeof( tb_dw_init_r[ 0 ] )
                  ) ;
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      for( i_num = 0 ; i_num < 64 ; ++i_num )
      {    m_st_tb_dw_k[ i_num ]
         = ( wxUint32 )floor(   fabs( sin( ( double )i_num + 1 ) )
                              * 4294967296.0
                            ) ;
      }
      /*-------------------------------------------------------------------*/
      m_st_boo_init_done = true ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   m_i_result_len = 0 ;
   /*----------------------------------------------------------------------*/
   m_ull_total_size           = 0              ;
   m_i_not_treated_buffer_len = 0              ;
   m_dw_h0 = wxUINT32_SWAP_ON_BE( 0x67452301 ) ;
   m_dw_h1 = wxUINT32_SWAP_ON_BE( 0xEFCDAB89 ) ;
   m_dw_h2 = wxUINT32_SWAP_ON_BE( 0x98BADCFE ) ;
   m_dw_h3 = wxUINT32_SWAP_ON_BE( 0x10325476 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( The buffer is supposed to be 64 bytes long )-------------------------*/
void CChecksumMD5::add_data_64bytes( const unsigned char *p_uc_buffer,
                                     CChecksumMD5::e_sltt sltt
                                   )
{
   /*----------------------------------------------------------------------*/
   wxUint32 *p_dw = ( wxUint32 * )p_uc_buffer ;
   wxUint32 dw_a  = m_dw_h0 ;
   wxUint32 dw_b  = m_dw_h1 ;
   wxUint32 dw_c  = m_dw_h2 ;
   wxUint32 dw_d  = m_dw_h3 ;
   wxUint32 dw_f            ;
   wxUint32 dw_g            ;
   wxUint32 dw_temp         ;
   wxUint32 dw_num          ;

   /*----------------------------------------------------------------------*/
   for( dw_num = 0 ; dw_num < 64 ; ++dw_num )
   {
      /*-------------------------------------------------------------------*/
      if( dw_num < 16 )
      {  dw_f = ( ( dw_b & dw_c ) | ( ~dw_b & dw_d ) ) ;
         dw_g = dw_num ;
      }
      else
      if( dw_num < 32 )
      {  dw_f = ( ( dw_d & dw_b ) | ( ~dw_d & dw_c ) ) ;
         dw_g = ( 5 * dw_num + 1 ) % 16 ;
      }
      else
      if( dw_num < 48 )
      {  dw_f = ( dw_b ^ dw_c ^ dw_d ) ;
         dw_g = ( 3 * dw_num + 5 ) % 16 ;
      }
      else
      {  dw_f = ( dw_c ^ ( dw_b | ~dw_d ) ) ;
         dw_g = ( 7 * dw_num ) % 16 ;
      }
      /*-------------------------------------------------------------------*/
      dw_temp = dw_d ;
      dw_d    = dw_c ;
      dw_c    = dw_b ;
      dw_b   += leftrotate(   dw_a + dw_f + m_st_tb_dw_k[ dw_num ]
                            + wxINT32_SWAP_ON_BE( p_dw[ dw_g ] ),
                            m_st_tb_dw_r[ dw_num ]
                          ) ;
      dw_a    = dw_temp ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   m_dw_h0 += dw_a ;
   m_dw_h1 += dw_b ;
   m_dw_h2 += dw_c ;
   m_dw_h3 += dw_d ;
   /*----------------------------------------------------------------------*/
   if( sltt == DO_SUM_LENGTH_TO_TOTAL )
   {  m_ull_total_size += 64 ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CChecksumMD5::add_data( const unsigned char *p_uc_buffer,
                             size_t sz_buffer_len
                           )
{
   /*--( Something left ? )------------------------------------------------*/
   if( m_i_not_treated_buffer_len > 0 )
   {
      /*-------------------------------------------------------------------*/
      size_t sz_size_to_add
           = wxMin( sz_buffer_len, 64 - m_i_not_treated_buffer_len ) ;

      /*--( Fill the buffer up ... )---------------------------------------*/
      memcpy( &m_tb_uc_not_treated_buffer[ m_i_not_treated_buffer_len ],
              p_uc_buffer, sz_size_to_add
            ) ;
      m_i_not_treated_buffer_len += sz_size_to_add ;
      p_uc_buffer   += sz_size_to_add ;
      sz_buffer_len -= sz_size_to_add ;

      /*--( And if filled, treat it )--------------------------------------*/
      if( m_i_not_treated_buffer_len == 64 )
      {
         /*----------------------------------------------------------------*/
         add_data_64bytes( m_tb_uc_not_treated_buffer ) ;
         m_i_not_treated_buffer_len = 0 ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( Treat the full 64 bytes parts )-----------------------------------*/
   while( sz_buffer_len >= 64 )
   {
      /*-------------------------------------------------------------------*/
      add_data_64bytes( p_uc_buffer ) ;
      p_uc_buffer   += 64 ;
      sz_buffer_len -= 64 ;
      /*-------------------------------------------------------------------*/
   }

   /*--( Keep the remaining bytes )----------------------------------------*/
   if( sz_buffer_len > 0 )
   {
      /*-------------------------------------------------------------------*/
      memcpy( m_tb_uc_not_treated_buffer, p_uc_buffer, sz_buffer_len ) ;
      m_i_not_treated_buffer_len = sz_buffer_len ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CChecksumMD5::final()
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_nb_bits ;

   /*--( This will be the "final" data length )----------------------------*/
   m_ull_total_size += m_i_not_treated_buffer_len ;
   /*--( Add the last "1" bit. Surely there is buffer space )--------------*/
   m_tb_uc_not_treated_buffer[ m_i_not_treated_buffer_len++ ] = 0x80 ;

   /*--( Enough place to add the length ? )--------------------------------*/
   if( m_i_not_treated_buffer_len >= 56 )
   {
      /*--( No, fill this buffer and add it )------------------------------*/
      memset( &m_tb_uc_not_treated_buffer[ m_i_not_treated_buffer_len ],
              '\0', 64 - m_i_not_treated_buffer_len
            ) ;
      /*--( Add this one without incrementing the total length )-----------*/
      add_data_64bytes( m_tb_uc_not_treated_buffer,
                        DONT_SUM_LENGTH_TO_TOTAL
                      ) ;
      m_i_not_treated_buffer_len = 0 ;
      /*-------------------------------------------------------------------*/
   }

   /*--( Filling up and leaving enough place for the size )----------------*/
   memset( &m_tb_uc_not_treated_buffer[ m_i_not_treated_buffer_len ],
           '\0', 56 - m_i_not_treated_buffer_len
         ) ;

   /*--( Original length in bits (not bytes) )-----------------------------*/
   m_ull_total_size *= 8 ;
   /*--( Size stored in little endian )------------------------------------*/
   dw_nb_bits = wxUINT32_SWAP_ON_BE( m_ull_total_size.GetLo() ) ;
   memcpy( &m_tb_uc_not_treated_buffer[ 56 ], &dw_nb_bits, 4 )  ;
   dw_nb_bits = wxUINT32_SWAP_ON_BE( m_ull_total_size.GetHi() ) ;
   memcpy( &m_tb_uc_not_treated_buffer[ 60 ], &dw_nb_bits, 4 )  ;

   /*--( Add it )----------------------------------------------------------*/
   add_data_64bytes( m_tb_uc_not_treated_buffer, DONT_SUM_LENGTH_TO_TOTAL ) ;

   /*--( And store the binary result )-------------------------------------*/
   wxUint32 *p_dw_result = ( wxUint32 * )m_tb_uc_result ;
   p_dw_result[ 0 ] = wxUINT32_SWAP_ON_BE( m_dw_h0 ) ;
   p_dw_result[ 1 ] = wxUINT32_SWAP_ON_BE( m_dw_h1 ) ;
   p_dw_result[ 2 ] = wxUINT32_SWAP_ON_BE( m_dw_h2 ) ;
   p_dw_result[ 3 ] = wxUINT32_SWAP_ON_BE( m_dw_h3 ) ;
   m_i_result_len = 16 ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CChecksumSHA1::init()
{
   /*----------------------------------------------------------------------*/
   m_i_result_len = 0 ;
   /*----------------------------------------------------------------------*/
   m_ull_total_size           = 0              ;
   m_i_not_treated_buffer_len = 0              ;
   m_dw_h0 = wxUINT32_SWAP_ON_BE( 0x67452301 ) ;
   m_dw_h1 = wxUINT32_SWAP_ON_BE( 0xEFCDAB89 ) ;
   m_dw_h2 = wxUINT32_SWAP_ON_BE( 0x98BADCFE ) ;
   m_dw_h3 = wxUINT32_SWAP_ON_BE( 0x10325476 ) ;
   m_dw_h4 = wxUINT32_SWAP_ON_BE( 0xC3D2E1F0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( The buffer is supposed to be 64 bytes long )-------------------------*/
void CChecksumSHA1::add_data_64bytes( const unsigned char *p_uc_buffer,
                                      CChecksumMD5::e_sltt sltt
                                    )
{
   /*----------------------------------------------------------------------*/
   wxUint32 *p_dw_src = ( wxUint32 * )p_uc_buffer ;
   wxUint32 tb_dw_ext[ 80 ] ;
   wxUint32 dw_a = m_dw_h0  ;
   wxUint32 dw_b = m_dw_h1  ;
   wxUint32 dw_c = m_dw_h2  ;
   wxUint32 dw_d = m_dw_h3  ;
   wxUint32 dw_e = m_dw_h4  ;
   wxUint32 dw_f            ;
   wxUint32 dw_k            ;
   wxUint32 dw_temp         ;
   wxUint32 dw_num          ;

   /*--( Init of the base part of the array )------------------------------*/
   for( dw_num = 0 ; dw_num < 16 ; ++dw_num )
   {  tb_dw_ext[ dw_num ] = wxUINT32_SWAP_ON_LE( p_dw_src[ dw_num ] ) ; }
   /*--( Init of the extension part of the array )-------------------------*/
   for( dw_num = 16 ; dw_num < 80 ; ++dw_num )
   {  tb_dw_ext[ dw_num ] = leftrotate(   tb_dw_ext[ dw_num -  3 ]
                                        ^ tb_dw_ext[ dw_num -  8 ]
                                        ^ tb_dw_ext[ dw_num - 14 ]
                                        ^ tb_dw_ext[ dw_num - 16 ],
                                        1
                                      ) ;
   }

   /*----------------------------------------------------------------------*/
   for( dw_num = 0 ; dw_num < 80 ; ++dw_num )
   {
      /*-------------------------------------------------------------------*/
      if( dw_num < 20 )
      {  dw_f = ( ( dw_b & dw_c ) | ( ~dw_b & dw_d ) ) ;
         dw_k = wxUINT32_SWAP_ON_BE( 0x5A827999 ) ;
      }
      else
      if( dw_num < 40 )
      {  dw_f = ( dw_b ^ dw_c ^ dw_d ) ;
         dw_k = wxUINT32_SWAP_ON_BE( 0x6ED9EBA1 ) ;
      }
      else
      if( dw_num < 60 )
      {  dw_f = ( ( dw_b & dw_c ) | ( dw_b & dw_d ) | ( dw_c & dw_d ) ) ;
         dw_k = wxUINT32_SWAP_ON_BE( 0x8F1BBCDC ) ;
      }
      else
      {  dw_f = ( dw_b ^ dw_c ^ dw_d ) ;
         dw_k = wxUINT32_SWAP_ON_BE( 0xCA62C1D6 ) ;
      }
      /*-------------------------------------------------------------------*/
      dw_temp =   leftrotate( dw_a, 5 )
                + dw_f + dw_e + dw_k + tb_dw_ext[ dw_num ] ;
      dw_e    = dw_d ;
      dw_d    = dw_c ;
      dw_c    = leftrotate( dw_b, 30 ) ;
      dw_b    = dw_a ;
      dw_a    = dw_temp ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   m_dw_h0 += dw_a ;
   m_dw_h1 += dw_b ;
   m_dw_h2 += dw_c ;
   m_dw_h3 += dw_d ;
   m_dw_h4 += dw_e ;
   /*----------------------------------------------------------------------*/
   if( sltt == DO_SUM_LENGTH_TO_TOTAL )
   {  m_ull_total_size += 64 ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CChecksumSHA1::final()
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_nb_bits ;

   /*--( This will be the "final" data length )----------------------------*/
   m_ull_total_size += m_i_not_treated_buffer_len ;
   /*--( Add the last "1" bit. Can't be out of buffer place )--------------*/
   m_tb_uc_not_treated_buffer[ m_i_not_treated_buffer_len++ ] = 0x80 ;

   /*--( Enough place to add the length ? )--------------------------------*/
   if( m_i_not_treated_buffer_len >= 56 )
   {
      /*--( No, fill this buffer and add it )------------------------------*/
      memset( &m_tb_uc_not_treated_buffer[ m_i_not_treated_buffer_len ],
              '\0', 64 - m_i_not_treated_buffer_len
            ) ;
      /*--( Add this one without incrementing the total length )-----------*/
      add_data_64bytes( m_tb_uc_not_treated_buffer,
                        DONT_SUM_LENGTH_TO_TOTAL
                      ) ;
      m_i_not_treated_buffer_len = 0 ;
      /*-------------------------------------------------------------------*/
   }

   /*--( Filling up and leaving enough place for the size )----------------*/
   memset( &m_tb_uc_not_treated_buffer[ m_i_not_treated_buffer_len ],
           '\0', 56 - m_i_not_treated_buffer_len
         ) ;

   /*--( Original length in bits (not bytes) )-----------------------------*/
   m_ull_total_size *= 8 ;
   /*--( Size stored in big endian )---------------------------------------*/
   dw_nb_bits = wxUINT32_SWAP_ON_LE( m_ull_total_size.GetHi() ) ;
   memcpy( &m_tb_uc_not_treated_buffer[ 56 ], &dw_nb_bits, 4 )  ;
   dw_nb_bits = wxUINT32_SWAP_ON_LE( m_ull_total_size.GetLo() ) ;
   memcpy( &m_tb_uc_not_treated_buffer[ 60 ], &dw_nb_bits, 4 )  ;

   /*--( Add it )----------------------------------------------------------*/
   add_data_64bytes( m_tb_uc_not_treated_buffer, DONT_SUM_LENGTH_TO_TOTAL ) ;

   /*--( And store the binary result )-------------------------------------*/
   wxUint32 *p_dw_result = ( wxUint32 * )m_tb_uc_result ;
   p_dw_result[ 0 ] = wxUINT32_SWAP_ON_LE( m_dw_h0 ) ;
   p_dw_result[ 1 ] = wxUINT32_SWAP_ON_LE( m_dw_h1 ) ;
   p_dw_result[ 2 ] = wxUINT32_SWAP_ON_LE( m_dw_h2 ) ;
   p_dw_result[ 3 ] = wxUINT32_SWAP_ON_LE( m_dw_h3 ) ;
   p_dw_result[ 4 ] = wxUINT32_SWAP_ON_LE( m_dw_h4 ) ;
   m_i_result_len = 20 ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
inline void CChecksumCRC32::crc32_byte( const unsigned char uc_byte )
{
   /*----------------------------------------------------------------------*/
     m_dw_result
   = (   ( m_dw_result >> 8 )
       ^ m_st_tb_crc32[ uc_byte ^ ( m_dw_result & 0xFF ) ]
     ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CChecksumCRC32::init()
{
   /*----------------------------------------------------------------------*/
   if( !m_st_boo_init_done )
   {
      /*--( Official polynomial used by CRC32 in PKZip )-------------------*/
      const wxUint32 co_dw_polynomial = wxUINT32_SWAP_ON_BE( 0xEDB88320 ) ;
      wxUint32 dw_crc ;
      int i_num_dw    ;
      int i_num_bit     ;
      /*-------------------------------------------------------------------*/
      for( i_num_dw = 0 ; i_num_dw < 256 ; ++i_num_dw )
      {
         /*----------------------------------------------------------------*/
         dw_crc = i_num_dw ;
         /*----------------------------------------------------------------*/
         for( i_num_bit = 8 ; i_num_bit > 0 ; --i_num_bit )
         {
            /*-------------------------------------------------------------*/
            if( ( dw_crc & 1 ) != 0 )
            {  dw_crc = ( ( dw_crc >> 1 ) ^ co_dw_polynomial ) ; }
            else
            {  dw_crc >>= 1 ; }
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         m_st_tb_crc32[ i_num_dw ] = dw_crc ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      m_st_boo_init_done = true ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   m_i_result_len = 0          ;
   m_dw_result    = 0xFFFFFFFF ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CChecksumCRC32::add_data( const unsigned char *p_uc_buffer,
                                   size_t sz_buffer_len
                                 )
{
   /*----------------------------------------------------------------------*/
   while( sz_buffer_len-- > 0 ) { crc32_byte( *p_uc_buffer++ ) ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CChecksumCRC32::final()
{
   /*----------------------------------------------------------------------*/
   m_dw_result = wxUINT32_SWAP_ON_LE( ~m_dw_result ) ;
   /*--( And store the text result )---------------------------------------*/
   wxUint32 *p_dw_result = ( wxUint32 * )m_tb_uc_result ;
   p_dw_result[ 0 ] = m_dw_result ;
   m_i_result_len = 4 ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                         End of file CChecksum.cpp                         !
+==========================================================================*/
