/*====================================+=====================================+
! File CArInputStream.cpp             ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!      Minimalistic implementation of the wxInputStream for AR files        !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include "common/sr_lib.h"
#include "common/CArInputStream.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
CArEntry::CArEntry( const wxString &s_name,
                    const wxDateTime & WXUNUSED( dt ),
                    wxFileOffset fo_size
                  )
         : m_s_name( s_name ),
           m_fo_size( fo_size ),
           m_fo_offset( -1 )
{  ; }

/*-------------------------------------------------------------------------*/
CArEntry::~CArEntry()
{  ; }

/*-------------------------------------------------------------------------*/
CArEntry::CArEntry( const CArEntry &e )
         : wxArchiveEntry(),
           m_s_name( e.m_s_name ),
           m_fo_size( e.m_fo_size ),
           m_fo_offset( e.m_fo_offset )
{  ; }

/*-------------------------------------------------------------------------*/
CArEntry &CArEntry::operator=( const CArEntry &e )
{
   /*----------------------------------------------------------------------*/
   if( &e != this )
   {
      /*-------------------------------------------------------------------*/
      m_s_name    = e.m_s_name    ;
      m_fo_size   = e.m_fo_size   ;
      m_fo_offset = e.m_fo_offset ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( *this ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CArEntry::read_header( wxInputStream *p_in )
{
   /*----------------------------------------------------------------------*/
   str_ar_file_header fh ;

   /*----------------------------------------------------------------------*/
   if(     p_in->Read( &fh, sizeof( fh ) ).LastRead() != sizeof( fh )
       || fh.tb_c_magic[ 0 ] != '\140'
       || fh.tb_c_magic[ 1 ] != '\n'
     )
   {  return( -1 ) ; }

   /*----------------------------------------------------------------------*/
#define CONV_TB_C_D2S( t ) \
        wxString::From8BitData( t, sizeof( t ) ).Trim( false ).Trim()
   /*----------------------------------------------------------------------*/
   m_s_name    = CONV_TB_C_D2S( fh.tb_c_filename )       ;
   m_fo_size   = wxAtol( CONV_TB_C_D2S( fh.tb_c_size ) ) ;
   m_fo_offset = p_in->TellI()                           ;
   /*----------------------------------------------------------------------*/
#undef CONV_TB_C_STR
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
CArInputStream::CArInputStream( wxInputStream &stream, wxMBConv &conv )
               :  wxArchiveInputStream( stream, conv )
{  Init() ; }

/*-------------------------------------------------------------------------*/
CArInputStream::CArInputStream( wxInputStream *stream, wxMBConv &conv )
               :  wxArchiveInputStream( stream, conv )
{  Init() ; }

/*-------------------------------------------------------------------------*/
CArInputStream::~CArInputStream()
{  ; }

/*-------------------------------------------------------------------------*/
void CArInputStream::Init()
{
   /*----------------------------------------------------------------------*/
   char tb_c_header[ 8 ] ;

   /*----------------------------------------------------------------------*/
   m_fo_entry_pos    = wxInvalidOffset ;
   m_fo_entry_offset = 0 ;
   m_fo_entry_size   = wxInvalidOffset ;

   /*--( Check the file header )-------------------------------------------*/
   if(    m_parent_i_stream->Read( tb_c_header, sizeof( tb_c_header )
                                 ).LastRead() == sizeof( tb_c_header )
       && memcmp( tb_c_header, "!<arch>\n", sizeof( tb_c_header ) ) != 0
     )
   {  m_lasterror = wxSTREAM_READ_ERROR ; }
   else
   {  m_lasterror = m_parent_i_stream->GetLastError() ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxDECLARE_SCOPED_PTR( CArEntry, CPtrArEntry )
wxDEFINE_SCOPED_PTR( CArEntry, CPtrArEntry )

/*-------------------------------------------------------------------------*/
CArEntry *CArInputStream::GetNextEntry()
{
   /*----------------------------------------------------------------------*/
   if( !CloseEntry() ) { return( NULL ) ; }

   /*----------------------------------------------------------------------*/
   CPtrArEntry ptr_entry( new CArEntry ) ;

   /*----------------------------------------------------------------------*/
   if( ptr_entry->read_header( m_parent_i_stream ) != 0 )
   {  return( NULL ) ; }
   /*----------------------------------------------------------------------*/
   m_fo_entry_pos    = 0 ;
   m_fo_entry_offset = ptr_entry->m_fo_offset ;
   m_fo_entry_size   = ptr_entry->m_fo_size ;

   /*----------------------------------------------------------------------*/
   return( ptr_entry.release() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
bool CArInputStream::CloseEntry()
{
   /*----------------------------------------------------------------------*/
   if( m_lasterror == wxSTREAM_READ_ERROR ) { return( false ) ; }
   if( !IsOpened() ) { return( true ) ; }

   /*----------------------------------------------------------------------*/
   wxFileOffset fo_to_skip = m_fo_entry_size - m_fo_entry_pos ;

   /*--( Data can't end on an odd address (the rest has an even size) )----*/
   if( m_fo_entry_size % 2 != 0 )
   {  ++fo_to_skip ; }

   /*--( Skip the rest of the current entry's data )-----------------------*/
   if( fo_to_skip == 0 )
   {  ; }
   else
   if( m_parent_i_stream->IsSeekable() )
   {
      /*-------------------------------------------------------------------*/
      if( m_parent_i_stream->SeekI( fo_to_skip, wxFromCurrent
                                  ) == wxInvalidOffset
        )
      {  return( false ) ; }
      /*-------------------------------------------------------------------*/
   }
   else
   {  /*--( So, read to simulate the skip )--------------------------------*/
      char   tb_c_dummy[ BUFSIZ ] ;
      size_t sz_read ;
      /*-------------------------------------------------------------------*/
      for( ; fo_to_skip > 0 ; fo_to_skip -= sz_read )
      {
         /*----------------------------------------------------------------*/
         sz_read = wxMin( fo_to_skip, ( int )sizeof( tb_c_dummy ) ) ;
         /*----------------------------------------------------------------*/
         if(    m_parent_i_stream->Read( tb_c_dummy, sz_read ).LastRead()
             != sz_read
           )
         {  return( false ) ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*---------------------------------------------------------------------*/
   m_fo_entry_pos     = wxInvalidOffset ;
   m_fo_entry_offset += m_fo_entry_size ;
   m_fo_entry_size    = wxInvalidOffset ;
   m_lasterror        = m_parent_i_stream->GetLastError() ;

   /*---------------------------------------------------------------------*/
   return( IsOk() ) ;
   /*---------------------------------------------------------------------*/
}

/*------------------------------------------------------------------------*/
bool CArInputStream::OpenEntry( CArEntry &entry )
{
   /*---------------------------------------------------------------------*/
   if(    GetLastError() != wxSTREAM_READ_ERROR
       && m_parent_i_stream->IsSeekable()
       && m_parent_i_stream->SeekI( entry.GetOffset() ) == entry.GetOffset()
     )
   {
      /*------------------------------------------------------------------*/
      m_fo_entry_offset = entry.GetOffset() ;
      m_fo_entry_size   = entry.m_fo_size ;
      m_fo_entry_pos    = 0 ;
      m_lasterror       = wxSTREAM_NO_ERROR ;
      /*------------------------------------------------------------------*/
      return( true ) ;
      /*------------------------------------------------------------------*/
   }
   /*---------------------------------------------------------------------*/
   m_lasterror = wxSTREAM_READ_ERROR ;
   return( false ) ;
   /*---------------------------------------------------------------------*/
}

/*------------------------------------------------------------------------*/
bool CArInputStream::OpenEntry( wxArchiveEntry &entry )
{
   /*---------------------------------------------------------------------*/
   CArEntry *p_entry = wxStaticCast( &entry, CArEntry ) ;
   return( p_entry != NULL ? OpenEntry( *p_entry ) : false ) ;
   /*---------------------------------------------------------------------*/
}

/*------------------------------------------------------------------------*/
size_t CArInputStream::OnSysRead( void *p_buffer, size_t sz_size )
{
   /*---------------------------------------------------------------------*/
   if( !IsOpened() ) { m_lasterror = wxSTREAM_READ_ERROR ;  }
   if( !IsOk() || sz_size == 0 ) { return( 0 ) ; }

   /*---------------------------------------------------------------------*/
   if( m_fo_entry_pos >= m_fo_entry_size )
   {  sz_size = 0 ; }
   else
   if( m_fo_entry_pos + ( wxFileOffset )sz_size > m_fo_entry_size )
   {  sz_size = m_fo_entry_size - m_fo_entry_pos ; }

   /*---------------------------------------------------------------------*/
   size_t sz_read = m_parent_i_stream->Read( p_buffer, sz_size).LastRead() ;
   m_fo_entry_pos += sz_read ;

   /*---------------------------------------------------------------------*/
   if( m_fo_entry_pos >= m_fo_entry_size )
   {  m_lasterror = wxSTREAM_EOF ; }
   else
   if( !m_parent_i_stream->IsOk())
   {  m_lasterror = wxSTREAM_READ_ERROR ; }
   /*---------------------------------------------------------------------*/
   return( sz_read ) ;
   /*---------------------------------------------------------------------*/
}

/*------------------------------------------------------------------------*/
wxFileOffset CArInputStream::OnSysSeek( wxFileOffset fo_pos,
                                        wxSeekMode mode
                                      )
{
   /*---------------------------------------------------------------------*/
   if( !IsOpened() )
   {  m_lasterror = wxSTREAM_READ_ERROR ; }
   if( !IsOk() ) { return( wxInvalidOffset ) ; }
   /*---------------------------------------------------------------------*/
   switch( mode )
   {
      /*------------------------------------------------------------------*/
      case wxFromStart   : break ;
      case wxFromCurrent : fo_pos += m_fo_entry_pos  ; break ;
      case wxFromEnd     : fo_pos += m_fo_entry_size ; break ;
      /*------------------------------------------------------------------*/
   }

   /*---------------------------------------------------------------------*/
   if(    fo_pos < 0
       || m_parent_i_stream->SeekI( m_fo_entry_offset + fo_pos
                                  ) == wxInvalidOffset
     )
   {  return( wxInvalidOffset ) ; }

   /*---------------------------------------------------------------------*/
   m_fo_entry_offset = fo_pos ;
   /*---------------------------------------------------------------------*/
   return( m_fo_entry_offset ) ;
   /*---------------------------------------------------------------------*/
}

/*------------------------------------------------------------------------*/



/*==========================================================================+
!                      End of file CArInputStream.cpp                       !
+==========================================================================*/
