/*====================================+=====================================+
! File CFSWatcher.cpp                 ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <wx/wx.h>
#include "CApp.h"
#include "CFileInit.h"
#include "CFSWatcher.h"
/*-------------------------------------------------------------------------*/



/*--( Comment the following line to disable trace creation in this file )--*/
//#define FSWATCHER_TRACE
/*-------------------------------------------------------------------------*/
#ifdef FSWATCHER_TRACE
#define FSWTRC1(a)     g_trace(a)
#define FSWTRC2(a,b)   g_trace(a,b)
#define FSWTRC3(a,b,c) g_trace(a,b,c)
#else
#define FSWTRC1(a)
#define FSWTRC2(a,b)
#define FSWTRC3(a,b,c)
#endif
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE( CFSWatcher, wxFileSystemWatcher )
   /*----------------------------------------------------------------------*/
   EVT_FSWATCHER( wxID_ANY, CFSWatcher::OnChange )
   EVT_TIMER( wxID_ANY, CFSWatcher::OnApplyChanges )
   /*----------------------------------------------------------------------*/
END_EVENT_TABLE()
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
void CFSWatcher::OnChange( wxFileSystemWatcherEvent &event )
{
   /*----------------------------------------------------------------------*/
   FSWTRC1( wxString::Format( "Id : %d %s\n<%s>\n<%s>\n",
                              event.GetId(), event.ToString(),
                              event.GetPath().GetFullPath(),
                              event.GetNewPath().GetFullPath()
                            )
          ) ;
   /*----------------------------------------------------------------------*/
   bool boo_add_event = true ;
   CFSWchange fsw_change( event.GetChangeType(), event.GetId(),
                          event.GetPath().GetFullPath(),
                          event.GetNewPath().GetFullPath()
                        ) ;

   /*----------------------------------------------------------------------*/
   switch( fsw_change.m_i_type )
   {
      /*--------------------------------------------------------------------+
      ! Some filtering has to be done because :                             !
      ! - bunches of "modify" events can be sent for one modification       !
      ! - "rename" is preceded by a "modify" of the new name => skipped     !
      ! - "create" is followed by a "modify" of the new name => skipped     !
      +--------------------------------------------------------------------*/
      case wxFSW_EVENT_MODIFY :
      {
         /*----------------------------------------------------------------*/
         t_vec_change_cit cit ;
         /*----------------------------------------------------------------*/
         for( cit = m_vec_change.begin() ;
              boo_add_event && cit != m_vec_change.end() ;
              ++cit
            )
         {  /*-------------------------------------------------------------*/
            if(    (    cit->m_i_id != FSW_WID_REN_DIFF_PATH
                     && (    cit->m_s_name == fsw_change.m_s_name
                          || cit->m_s_new_name == fsw_change.m_s_name
                        )
                   )
                || (    cit->m_i_id == FSW_WID_REN_DIFF_PATH
                     && (       to_fullpath( cit->m_s_new_name )
                             == fsw_change.m_s_new_name
                          ||    to_fullpath( cit->m_s_name )
                             == fsw_change.m_s_name
                        )
                   )
              )
            {  boo_add_event = false ; }
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*--------------------------------------------------------------------+
      ! After a renaming from one directory to another done with Siren      !
      ! useless events can be generated :                                   !
      ! - "create" in the destination directory                             !
      ! - "delete" in the source directory                                  !
      +---------------------------------------------------------------------+
      ! A "manual" delete under MSW/Siren generates a "modify" then a       !
      ! "delete". It seems that the "modify" is useless.                    !
      +--------------------------------------------------------------------*/
      case wxFSW_EVENT_CREATE :
      case wxFSW_EVENT_DELETE :
      {
         /*----------------------------------------------------------------*/
         t_vec_change_it it ;
         /*----------------------------------------------------------------*/
         for( it = m_vec_change.begin() ;
              boo_add_event && it != m_vec_change.end() ;
              ++it
            )
         {  /*-------------------------------------------------------------*/
            if(    it->m_i_id == FSW_WID_REN_DIFF_PATH
                && (    (    fsw_change.m_i_type == wxFSW_EVENT_CREATE
                          &&    to_fullpath( it->m_s_new_name )
                             == fsw_change.m_s_new_name
                        )
                     || (    fsw_change.m_i_type == wxFSW_EVENT_DELETE
                          &&    to_fullpath( it->m_s_name )
                             == fsw_change.m_s_name
                        )
                   )
              )
            {  /*----------------------------------------------------------*/
               boo_add_event = false ;
               FSWTRC1( wxString::Format( "==> Element found : %s\n",
                                          it->to_string()
                                        )
                      ) ;
               /*----------------------------------------------------------*/
            }
#ifdef __WXMSW__
            else
            if(    fsw_change.m_i_type == wxFSW_EVENT_DELETE
                && it->m_i_type == wxFSW_EVENT_MODIFY
                && it->m_s_name == fsw_change.m_s_name
                && it->m_s_new_name == fsw_change.m_s_new_name
              )
            {  /*----------------------------------------------------------*/
               FSWTRC1( wxString::Format( "==> Repl mod with del evt (%s)\n",
                                          it->to_string()
                                        )
                      ) ;
               /*----------------------------------------------------------*/
               *it = fsw_change ;
               boo_add_event = false ;
               /*----------------------------------------------------------*/
            }
#endif // __WXMSW__
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
#ifdef __WXGTK__
      /*--------------------------------------------------------------------+
      ! Some changes have to be done because under GTK a move can generate  !
      ! a rename event with same curent and new paths.                      !
      ! This happens during cut/paste using the system file explorer.       !
      ! Of course this happens too when Siren renames a file from one       !
      ! directory to another. In this case the previous event has to be     !
      ! replaced. Based on tests ... the FSW event appears before Siren's   !
      ! one.                                                                !
      +--------------------------------------------------------------------*/
      case wxFSW_EVENT_RENAME :
         /*----------------------------------------------------------------*/
         if( event.GetId() != FSW_WID_REN_DIFF_PATH )
         {
            /*-------------------------------------------------------------*/
            if( fsw_change.m_s_name == fsw_change.m_s_new_name )
            {
               /*----------------------------------------------------------*/
               fsw_change.m_i_type = ( event.GetPath().FileExists()
                                       ? wxFSW_EVENT_CREATE
                                       : wxFSW_EVENT_DELETE
                                     ) ;
               FSWTRC2( "==> Change type to %s\n",
                        fsw_change.m_i_type == wxFSW_EVENT_CREATE
                        ? "create" : "delete"
                      ) ;
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
         }
         else /*--( event.GetId() == FSW_WID_REN_DIFF_PATH )---------------*/
         {
            /*--------------------------------------------------------------+
            ! The search is done on absolute paths because Siren sets the   !
            ! event paths as it has been chosen by the user.                !
            +--------------------------------------------------------------*/
            wxFileName fn_name( event.GetPath().GetFullPath() ) ;
            wxFileName fn_new_name( event.GetNewPath().GetFullPath() ) ;
            t_vec_change_it it ;
            /*-------------------------------------------------------------*/
            if( !fn_name.MakeAbsolute() || !fn_new_name.MakeAbsolute() )
            {  break ; }
            /*--------------------------------------------------------------+
            ! Comparing on wxFileName is very heavy therefore it is done on !
            ! the equivalent strings.                                       !
            +--------------------------------------------------------------*/
            wxString s_name( fn_name.GetFullPath() ) ;
            wxString s_new_name( fn_new_name.GetFullPath() ) ;
            /*-------------------------------------------------------------*/
            for( it = m_vec_change.begin() ;
                 boo_add_event && it != m_vec_change.end() ;
                 ++it
               )
            {  /*----------------------------------------------------------*/
               if(    (    it->m_i_type == wxFSW_EVENT_CREATE
                        && it->m_s_new_name == s_new_name
                      )
                   || (    it->m_i_type == wxFSW_EVENT_DELETE
                        && it->m_s_name == s_name
                      )
                 )
               {
                  /*-------------------------------------------------------*/
                  FSWTRC1( wxString::Format( "==> Replace event (%s)\n",
                                            it->to_string()
                                           )
                         ) ;
                  /*-------------------------------------------------------*/
                  *it = fsw_change ;
                  boo_add_event = false ;
                  /*-------------------------------------------------------*/
               }
               /*----------------------------------------------------------*/
            }
            /*-------------------------------------------------------------*/
         }
         /*----------------------------------------------------------------*/
         break ;
         /*----------------------------------------------------------------*/
#endif // __WXGTK__
      /*-------------------------------------------------------------------*/
   }

   /*--( Keep this event to "apply" it a bit later )-----------------------*/
   if( boo_add_event )
   {  /*-------------------------------------------------------------------*/
      m_vec_change.push_back( fsw_change ) ;
      m_timer_to_apply.StartOnce( 50 ) ;
      FSWTRC1( "==> Accepted\n\n" ) ;
      /*-------------------------------------------------------------------*/
   }
   else
   {  FSWTRC1( "==> Skipped\n\n" ) ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFSWatcher::OnApplyChanges( wxTimerEvent & WXUNUSED( event ) )
{
   /*----------------------------------------------------------------------*/
   FSWTRC2( "------------------------------\n%s\n", __func__ ) ;

   /*-----------------------------------------------------------------------+
   ! Reuse of the frame's loaddir member because it contains all the        !
   ! parameters used during the loading of the directory.                   !
   ! Note that no information about this loading should be displayed.       !
   +-----------------------------------------------------------------------*/
   CLoadDir &ld( wxGetApp().m_frame->m_loaddir ) ;
   /*--( Nothing done if a loading is active )-----------------------------*/
   if( ld.thread_running() )
   {  return ; }

   /*----------------------------------------------------------------------*/
   CFileList  *fl( wxGetApp().m_frame->m_fl ) ;
   wxFileName fn_name ;
   bool       boo_recompute = false ;
   bool       boo_refresh_tv = false ;
   int        i_num_file ;
   long       l_num_focus = fl->get_item_focus() ;
   t_vec_change_cit it_change ;
   CFileListDisplayState display_state ;
   bool       boo_apply_display_state = false ;

   /*--( Necessary to set back correctly the display after deletes )-------*/
   fl->extract_display_state( display_state ) ;

   /*----------------------------------------------------------------------*/
   for( it_change = m_vec_change.begin() ; it_change != m_vec_change.end() ;
        ++it_change
      )
   {
      /*-------------------------------------------------------------------*/
      FSWTRC1( wxString::Format( "%s\n" , it_change->to_string() ) ) ;

      /*-------------------------------------------------------------------*/
      fn_name.Assign( it_change->m_s_name ) ;
      /*--------------------------------------------------------------------+
      ! If the event has been generated by Siren the names are kept intact. !
      +--------------------------------------------------------------------*/
      if( it_change->m_i_id != FSW_WID_REN_DIFF_PATH )
      {  fn_name.MakeRelativeTo( wxGetApp().M_s_dir.get() ) ; }

      /*-------------------------------------------------------------------*/
      switch( it_change->m_i_type )
      {
         /*----------------------------------------------------------------*/
         case wxFSW_EVENT_CREATE :
         {
            /*--( Show hidden files ? )------------------------------------*/
            if(    !ld.get_list_hidden()
                && sr::is_file_attr_hidden( fn_name.GetFullPath() )
              )
            {  continue ; }

            /*--------------------------------------------------------------+
            ! In some cases, for example "undoing" a rename of a file       !
            ! from another directory than the current one, the file may     !
            ! already be present in the file list.                          !
            +--------------------------------------------------------------*/
            if( fl->m_dir.file_with_name( fn_name ) >= 0 )
            {  continue ; }
            /*-------------------------------------------------------------*/
            FSWTRC1( "==> Create\n" ) ;
            /*--( Add name to internal vector )----------------------------*/
            if( ld.add_name( fn_name.GetFullPath(),
                             wxFileName::DirExists( fn_name.GetFullPath() )
                           ) != 0
              )
            {  continue ; }
            /*--( Transfer the file data to the main vector )--------------*/
            fl->m_dir.push_back( ld.m_dir.back() ) ;
            fl->SetItemCount( fl->m_dir.size() ) ;
            /*--( Parent link )--------------------------------------------*/
            fl->m_dir.back().set_cdir( &fl->m_dir ) ;
            /*--( File transferred. Useless data can be removed )----------*/
            ld.m_dir.zero() ;
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }

         /*----------------------------------------------------------------*/
         case wxFSW_EVENT_DELETE :
         {
            /*--( Known file ? )-------------------------------------------*/
            if( ( i_num_file = fl->m_dir.file_with_name( fn_name ) ) < 0 )
            {  continue ; }
            /*-------------------------------------------------------------*/
            FSWTRC1( "==> Delete\n" ) ;
            /*-------------------------------------------------------------*/
            CFile &file( fl->m_dir[ i_num_file ] ) ;
            /*-------------------------------------------------------------*/
            if( file.is_selected() )
            {  boo_recompute = true ; }
            if( i_num_file == l_num_focus )
            {  boo_refresh_tv = true ; }
            /*-------------------------------------------------------------*/
            if( fl->m_dir.remove_element( i_num_file ) != 0 )
            {  continue ; }
            fl->SetItemCount( fl->m_dir.size() ) ;
            /*------------------------------------------------------------*/
            boo_apply_display_state = true ;
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }

         /*----------------------------------------------------------------*/
         case wxFSW_EVENT_MODIFY :
         {
            /*--( File exists ? )------------------------------------------*/
            if( ( i_num_file = fl->m_dir.file_with_name( fn_name ) ) < 0 )
            {  continue ; }
            /*-------------------------------------------------------------*/
            FSWTRC1( "==> Modify\n" ) ;
            /*-------------------------------------------------------------*/
            CFile &file( fl->m_dir[ i_num_file ] ) ;
            /*-------------------------------------------------------------*/
            if( file.is_selected() )
            {  boo_recompute = true ; }
            if( i_num_file == l_num_focus )
            {  boo_refresh_tv = true ; }
            /*-------------------------------------------------------------*/
            if( file.reload_info( true ) == 0 )
            {  fl->RefreshItem( i_num_file ) ; }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }

         /*----------------------------------------------------------------*/
         case wxFSW_EVENT_RENAME :
         {
            /*--( File exists ? )------------------------------------------*/
            FSWTRC1( wxString::Format( "search for <%s>\n",
                                       fn_name.GetFullPath()
                                     )
                   ) ;
            if( ( i_num_file = fl->m_dir.file_with_name( fn_name ) ) < 0 )
            {  continue ; }
            /*-------------------------------------------------------------*/
            FSWTRC1( "==> Rename\n" ) ;
            /*-------------------------------------------------------------*/
            CFile &file( fl->m_dir[ i_num_file ] ) ;

            /*-------------------------------------------------------------*/
            file.set_name( it_change->m_s_new_name ) ;
            /*--------------------------------------------------------------+
            ! If the event has been generated by Siren the names are kept   !
            ! intact.                                                       !
            +--------------------------------------------------------------*/
            if( it_change->m_i_id != FSW_WID_REN_DIFF_PATH )
            {  file.get_name().MakeRelativeTo( wxGetApp().M_s_dir.get() ) ; }
            /*-------------------------------------------------------------*/
            if( file.is_selected() )
            {  boo_recompute = true ; }
            if( i_num_file == l_num_focus )
            {  boo_refresh_tv = true ; }
            /*--( A full reload is done if the extension has changed )-----*/
            bool boo_full_reload
               = (    it_change->m_s_name.AfterLast( '.' )
                   != it_change->m_s_new_name.AfterLast( '.' )
                 ) ;
            if( file.reload_info( boo_full_reload ) == 0 )
            {  fl->RefreshItem( i_num_file ) ; }
            /*-------------------------------------------------------------*/
            break ;
            /*-------------------------------------------------------------*/
         }

         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( List has changed so the "use" needs to be reinitialized )---------*/
   fl->init_col_and_grp_used() ;
   fl->create_all_col() ;
   /*----------------------------------------------------------------------*/
   if( boo_recompute )
   {  fl->compute_new_names() ; }
   if( boo_refresh_tv )
   {  wxGetApp().m_frame->visu_file() ; }
   if( boo_apply_display_state )
   {  fl->Refresh() ; fl->apply_display_state( display_state ) ; }

   /*--( Selected files "disappeared" ? Undo has to be enabled ? ... )-----*/
   wxGetApp().m_frame->gui_update_oper() ;

   /*--( All the changes have been applied )-------------------------------*/
   m_vec_change.clear() ;
   /*----------------------------------------------------------------------*/
   FSWTRC2( "------------------------------\n\n", __func__ ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFSWatcher::start()
{
   /*----------------------------------------------------------------------*/
   if( wxGetApp().M_s_dir.get().empty() )
   {  return ; }
   /*----------------------------------------------------------------------*/
   if( GetWatchedPathsCount() > 0 )
   {  return ; }

   /*-----------------------------------------------------------------------+
   ! Under wx2.9.5 MSW, modify event are received in bunches when a file    !
   ! content is changed. A modify is sent too when a file is renamed.       !
   +-----------------------------------------------------------------------*/
   const int  co_i_flags =   wxFSW_EVENT_CREATE | wxFSW_EVENT_DELETE
                           | wxFSW_EVENT_MODIFY | wxFSW_EVENT_RENAME ;
   wxFileName fn_path( wxFileName::DirName( wxGetApp().M_s_dir.get() ) ) ;

   /*--( Returns information about the link, not the target )--------------*/
   fn_path.DontFollowLink() ;
   /*----------------------------------------------------------------------*/
   {
      /*--------------------------------------------------------------------+
      ! "Add..." can generate logs. For example, under MSW some directories !
      ! may not be "watched" : mounted drives under VirtualBox.             !
      +--------------------------------------------------------------------*/
      wxLogNull no_log ;
      /*-------------------------------------------------------------------*/
      if( wxGetApp().m_frame->get_recurse() )
      {  m_boo_active = AddTree( fn_path, co_i_flags ) ; }
      else
      {  m_boo_active = Add( fn_path, co_i_flags ) ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   wxGetApp().m_frame->gui_update_oper() ;
   /*----------------------------------------------------------------------*/
   FSWTRC1( wxString::Format( "START%s : %s (%d)\n\n",
                              wxGetApp().m_frame->get_recurse()
                              ? " TREE" : "",
                              m_boo_active ? "OK" : "KO !!!!!!!",
                              m_boo_active
                            )
          ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CFSWatcher::stop()
{
   /*----------------------------------------------------------------------*/
   wxArrayString as_path ;
   int           i_nb_path ;
   int           i_num ;

   /*-----------------------------------------------------------------------+
   ! It seems there is an issue with the "RemoveAll" function at least      !
   ! under msw. Therefore the removal is done one path after another.       !
   +-----------------------------------------------------------------------*/
   i_nb_path = GetWatchedPaths( &as_path ) ;
   FSWTRC1( wxString::Format( "STOP : %d path(s) to remove\n", i_nb_path ) );
   /*----------------------------------------------------------------------*/
   for( i_num = 0 ; i_num < i_nb_path ; ++i_num )
   {
      /*-------------------------------------------------------------------*/
      if( !Remove( wxFileName::DirName( as_path[ i_num ] ) ) )
      {  FSWTRC1( wxString::Format( "error removing path <%s>\n",
                                    as_path[ i_num ]
                                  )
                ) ;
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   FSWTRC1( "\n" ) ;
   /*----------------------------------------------------------------------*/
   m_boo_active = ( GetWatchedPathsCount() > 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                       End of file CFSWatcher.cpp                          !
+==========================================================================*/
