/*====================================+=====================================+
! File CDir.cpp                       ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <map>
#include <algorithm>
#include <wx/longlong.h>
#include <wx/datetime.h>
#include <wx/filename.h>
#include "common/sr_lib.h"
#include "siren.h"
#include "CApp.h"
#include "CDir.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
wxString CDir::get_dir() const
{
   /*----------------------------------------------------------------------*/
   if( m_s_fake_dir.empty() )
   {  return( wxGetApp().M_s_dir.get() ) ; }
   /*----------------------------------------------------------------------*/
   return( m_s_fake_dir ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! The selection/unselection operations imply the management of the selected !
! files vector.                                                             !
! And the selection size/duration too ...                                   !
+--------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
void CDir::add_sel_cumul_info( const CFile &file )
{
   /*----------------------------------------------------------------------*/
   m_ull_sel_size   += file.get_size() ;
   m_i_sel_duration += file.get_duration() ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CDir::sub_sel_cumul_info( const CFile &file )
{
   /*----------------------------------------------------------------------*/
   m_ull_sel_size   -= file.get_size() ;
   m_i_sel_duration -= file.get_duration() ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CDir::sel( int i_item )
{
   /*----------------------------------------------------------------------*/
   if( i_item < 0 || i_item >= ( int )m_vec_file.size() )
   {  wxFAIL ; return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   if( !m_vec_file[ i_item ].is_selectable() )
   {  return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   if( m_vec_file[ i_item ].is_selected() )
   {  return( -3 ) ; }
   /*----------------------------------------------------------------------*/
   m_vec_file[ i_item ].set_sel_num( m_vec_i_sel.size() )  ;
   add_sel_cumul_info( m_vec_file[ i_item ] ) ;
   m_vec_i_sel.push_back( i_item ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CDir::unsel( int i_item )
{
   /*----------------------------------------------------------------------*/
   if( i_item < 0 || i_item >= ( int )m_vec_file.size() )
   {  wxFAIL ; return( -1 ) ; }
   /*----------------------------------------------------------------------*/
   if( !m_vec_file[ i_item ].is_selected() )
   {  return( -2 ) ; }
   /*----------------------------------------------------------------------*/
   int i_sel_num = m_vec_file[ i_item ].get_sel_num() ;
   /*----------------------------------------------------------------------*/
   m_vec_file[ i_item ].unsel() ;
   sub_sel_cumul_info( m_vec_file[ i_item ] ) ;

   /*--( Remove the selected item )----------------------------------------*/
   m_vec_i_sel.erase( m_vec_i_sel.begin() + i_sel_num ) ;

   /*--( Renumbering of the rest )-----------------------------------------*/
   for( ; i_sel_num < get_nb_sel() ; ++i_sel_num )
   {
      /*-------------------------------------------------------------------*/
      i_item = get_sel_file_num( i_sel_num ) ;
      m_vec_file[ i_item ].set_sel_num( i_sel_num ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CDir::sort( bool boo_natural_comp, bool boo_dir_top, int i_subitem,
                 int i_sort_type, bool boo_asc,
                 CRunInfoMessage &runinfomessage
               )
{
   /*----------------------------------------------------------------------*/
   std::sort( m_vec_file.begin(), m_vec_file.end(),
              CFileComp( boo_natural_comp, boo_dir_top, i_subitem,
                         i_sort_type, boo_asc, runinfomessage
                       )
            ) ;

   /*--( Restore selection data )------------------------------------------*/
   if( m_vec_i_sel.size() > 0 )
   {
      /*-------------------------------------------------------------------*/
      size_t sz_num     = 0 ;
      size_t sz_sel_num = m_vec_i_sel.size() ;
      /*-------------------------------------------------------------------*/
      for( ; sz_num < m_vec_file.size() && sz_sel_num > 0 ; ++sz_num )
      {
         /*----------------------------------------------------------------*/
         if( m_vec_file[ sz_num ].is_selected() )
         {  m_vec_i_sel[ m_vec_file[ sz_num ].get_sel_num() ] = sz_num ;
            --sz_sel_num ;
         }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CDir::file_with_name( const wxString &s_name ) const
{
   /*----------------------------------------------------------------------*/
   t_vec_file_cit it ;
   /*----------------------------------------------------------------------*/
   it = std::find_if( m_vec_file.begin(), m_vec_file.end(),
                      CFileWithName( s_name )
                    ) ;
   /*----------------------------------------------------------------------*/
   return( it == m_vec_file.end() ? -1 : it - m_vec_file.begin() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CDir::file_with_prefix_circle( int i_start, const wxString &s_prefix
                                 ) const
{
   /*--( By default the searched start at the top )------------------------*/
   if( i_start < 0 || i_start >= ( int )m_vec_file.size() )
   {  i_start = 0 ; }

   /*----------------------------------------------------------------------*/
   t_vec_file_cit cit_begin = m_vec_file.begin() + i_start ;
   t_vec_file_cit cit_end   = m_vec_file.end() ;
   t_vec_file_cit cit_file ;

   /*----------------------------------------------------------------------*/
   cit_file = std::find_if( cit_begin, cit_end,
                            CFileWithPrefix( s_prefix )
                          ) ;
   /*--( Circular search )-------------------------------------------------*/
   if( i_start > 0 && cit_file == cit_end )
   {  /*-------------------------------------------------------------------*/
      cit_end   = cit_begin ;
      cit_begin = m_vec_file.begin() ;
      /*-------------------------------------------------------------------*/
      cit_file  = std::find_if( cit_begin, cit_end,
                                CFileWithPrefix( s_prefix )
                              ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( cit_file == cit_end ? -1 : cit_file - m_vec_file.begin() ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Remove entries from the vectors )------------------------------------*/
int CDir::remove_element( int i_num )
{
   /*----------------------------------------------------------------------*/
   if( i_num < 0 || i_num >= ( int )m_vec_file.size() ) { return( -1 ) ; }
   /*--( First it may have to be unselected )------------------------------*/
   unsel( i_num ) ;
   /*-----------------------------------------------------------------------+
   ! The selection vector has to be updated to reflect the coming removal   !
   ! of the file from the file list : the positions of the files following  !
   ! it have to be updated.                                                 !
   +-----------------------------------------------------------------------*/
   int i_sel_num ;
   /*----------------------------------------------------------------------*/
   for( i_sel_num = 0 ; i_sel_num < get_nb_sel() ; ++i_sel_num )
   {
      /*-------------------------------------------------------------------*/
      if( m_vec_i_sel[ i_sel_num ] > i_num )
      {  --m_vec_i_sel[ i_sel_num ] ; }
      /*-------------------------------------------------------------------*/
   }
   /*--( Now it can be removed from the vector )---------------------------*/
   m_vec_file.erase( m_vec_file.begin() + i_num ) ;
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! For "inter file" expression evaluation the parent has to be known         !
+--------------------------------------------------------------------------*/
void CDir::init_file_cdir_pointer()
{
   /*----------------------------------------------------------------------*/
   size_t sz_num ;
   /*----------------------------------------------------------------------*/
   for( sz_num = 0 ; sz_num < m_vec_file.size() ; ++sz_num )
   {  m_vec_file[ sz_num ].set_cdir( this ) ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CDir::swap( CDir &dir )
{
   /*----------------------------------------------------------------------*/
   std::swap( m_boo_recurse, dir.m_boo_recurse ) ;
   m_vec_file.swap( dir.m_vec_file ) ;
   m_vec_i_sel.swap( dir.m_vec_i_sel ) ;
   std::swap( m_ull_sel_size, dir.m_ull_sel_size ) ;
   std::swap( m_i_sel_duration, dir.m_i_sel_duration ) ;
   m_s_fake_dir.swap( dir.m_s_fake_dir ) ;

   /*--( The "parent" pointers must be reinitialized )---------------------*/
   init_file_cdir_pointer() ;
   /*--( Useless, isnt'it ? )----------------------------------------------*/
   //dir.init_file_cdir_pointer() ;
   /*----------------------------------------------------------------------*/
}

/*--( Directory-Reset selection number )-----------------------------------*/
int CDir::get_s_sel_num_rel_dir( const CFile &file ) const
{
   /*----------------------------------------------------------------------*/
   int      i_sel_num                      ;
   int      i_num                          ;
   int      i_sel_num_rel = 0              ;
   wxString s_file_path( file.get_path() ) ;
   /*----------------------------------------------------------------------*/
   for( i_sel_num = 0 ; i_sel_num < file.get_sel_num() ; ++i_sel_num )
   {
      /*-------------------------------------------------------------------*/
      i_num = get_sel_file_num( i_sel_num ) ;
      /*-------------------------------------------------------------------*/
      if( s_file_path.IsSameAs( m_vec_file[ i_num ].get_path(),
                                wxFileName::IsCaseSensitive()
                              )
        )
      {  ++i_sel_num_rel ; }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   return( i_sel_num_rel ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! Selection number incremented on "collision"                               !
! "s_beg_name" is passed by value because it is modified in the function    !
+--------------------------------------------------------------------------*/
int CDir::get_s_sel_num_rel_collision( const CFile &file,
                                       wxString    s_beg_name,
                                       bool        boo_without_path
                                     ) const
{
   /*----------------------------------------------------------------------*/
   int       i_sel_num                             ;
   const int co_i_sel_num_max = file.get_sel_num() ;
   int       i_num                                 ;
   int       i_sel_num_rel = 0                     ;

   /*--( Global modifications already done on already treated files )------*/
   //### file.modif_globale_nom( s_deb_nom ) ;

   /*--( Comparison take the path into account ? )-------------------------*/
   if( boo_without_path )
   {
      /*-------------------------------------------------------------------*/
      for( i_sel_num = 0 ; i_sel_num < co_i_sel_num_max ; ++i_sel_num )
      {
         /*----------------------------------------------------------------*/
         i_num = get_sel_file_num( i_sel_num ) ;
         /*----------------------------------------------------------------*/
         if( s_beg_name.IsSameAs(
                       m_vec_file[ i_num ].get_new_full_name().Left(
                                                            s_beg_name.size()
                                                                   ),
                       wxFileName::IsCaseSensitive()
                                )
           )
         {  ++i_sel_num_rel ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   else
   {
      /*-------------------------------------------------------------------*/
      for( i_sel_num = 0 ; i_sel_num < co_i_sel_num_max ; ++i_sel_num )
      {
         /*----------------------------------------------------------------*/
         i_num = get_sel_file_num( i_sel_num ) ;
         /*----------------------------------------------------------------*/
         if( s_beg_name.IsSameAs(
                       m_vec_file[ i_num ].get_new_full_path().Left(
                                                            s_beg_name.size()
                                                                   ),
                       wxFileName::IsCaseSensitive()
                                )
           )
         {  ++i_sel_num_rel ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( i_sel_num_rel ) ;
   /*----------------------------------------------------------------------*/
}

/*--------------------------------------------------------------------------+
! This function is used mainly for the expression testbed                   !
+--------------------------------------------------------------------------*/
void CDir::compute_new_names( const wxString &s_exp_src )
{
   /*----------------------------------------------------------------------*/
   if( get_nb_sel() == 0 ) { return ; }

   /*----------------------------------------------------------------------*/
   t_exp_comp_info  eci              ;
   CFile            *p_file          ;
   wxString         s_exp            ;
   sr::wxString_cit it_sub_exp_start ;
   int              i_sel_num        ;

   /*--( Computation initialisations )-------------------------------------*/
   if( CFileList::st_compute_init( s_exp_src, s_exp ) != 0 )
   {  return ; }
   eci.boo_first_sub_exp = true ;
   eci.it_exp = s_exp.begin() ;
   eci.it_exp_end = s_exp.end() ;

   /*-----------------------------------------------------------------------+
   ! Even if the expression is empty, the recomputation is done to zero the !
   ! result.                                                                !
   ! It is made one sub-expression after another on all selected files.     !
   +-----------------------------------------------------------------------*/
   do
   {
      /*-------------------------------------------------------------------*/
      it_sub_exp_start = eci.it_exp ;
      /*--------------------------------------------------------------------+
      ! It is done following the selection number to manage the             !
      ! dependencies like the ones for "%nc"                                !
      +--------------------------------------------------------------------*/
      for( i_sel_num = 0 ; i_sel_num < get_nb_sel() ; ++i_sel_num )
      {
         /*----------------------------------------------------------------*/
         p_file = get_p_file( get_sel_file_num( i_sel_num ) ) ;
         if( p_file == NULL ) { continue ; }
         /*----------------------------------------------------------------*/
         eci.it_exp = it_sub_exp_start ;
         /*----------------------------------------------------------------*/
         p_file->compute_sub_exp( eci ) ;
         /*----------------------------------------------------------------*/
         p_file->set_new_name( eci.s_current_value ) ;

         /*--( The "global" modif are only applied on the last exp )-------*/
         if( eci.it_exp == s_exp.end() )
         {  p_file->new_name_global_modification() ; }

         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      eci.boo_first_sub_exp = false ;
      /*-------------------------------------------------------------------*/
   } while( eci.it_exp < eci.it_exp_end ) ;

   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                           End of File CDir.cpp                            !
+==========================================================================*/
