/*====================================+=====================================+
! File CCheckVersion.cpp              ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <wx/url.h>
#include <wx/txtstrm.h>
#include "CApp.h"
#include "CCheckVersion.h"
#include "siren_info.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
CCheckVersion::CCheckVersion()
              : m_boo_thread_stopping( false )
{
   /*----------------------------------------------------------------------*/
   wxCriticalSectionLocker enter_csec( m_csec_p_thread ) ;
   m_p_thread = NULL ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
CCheckVersion::~CCheckVersion()
{  ; }

/*-------------------------------------------------------------------------*/
CCheckVersion::e_ret_code CCheckVersion::exec( e_exec_mode exec_mode,
                                               wxString &s_info
                                             )
{
   /*--( Don't want any error message during the version check ... )-------*/
   wxLogNull no_log ;

   /*----------------------------------------------------------------------*/
   wxHTTP        http ;
   int           i_nb_retry = 0 ;
   wxInputStream *p_input ;

   /*--( Default error message )-------------------------------------------*/
   s_info = _( "Unable to get information. Please, try again later." ) ;
   /*--( Lower limit ... )-------------------------------------------------*/
   http.SetTimeout( 1 ) ;

   /*--( The real connection will be established by the input stream )-----*/
   if( !http.Connect( SIREN_HOST ) )
   {  return( CVC_ERROR ) ; }

   /*--( Polling connection/stop thread )----------------------------------*/
   do
   {  /*--( Check if stop requested )--------------------------------------*/
      if(    exec_mode == EM_THREADED
          && m_p_thread != NULL && m_p_thread->TestDestroy()
        )
      {  s_info.clear() ;
         return( CVC_ERROR ) ;
      }
      /*--( Maximum number of retries )------------------------------------*/
      if( ++i_nb_retry > 60 )
      {  return( CVC_ERROR ) ; }
      /*--( Establishing connection. Timeout errors only accepted )--------*/
      p_input = http.GetInputStream( "/download/siren_version.txt" ) ;
      if(    p_input == NULL
          && http.Error()
          && http.LastError() != wxSOCKET_TIMEDOUT
        )
      {  return( CVC_ERROR ) ; }
      /*-------------------------------------------------------------------*/
   } while( p_input == NULL ) ;

   /*--( Read file content: one line with the current version )------------*/
   wxTextInputStream text_input( *p_input ) ;
   wxString          s_site_version = text_input.ReadLine() ;

   /*----------------------------------------------------------------------*/
   delete p_input ;

   /*----------------------------------------------------------------------*/
   if( s_site_version.empty() )
   {  return( CVC_ERROR ) ; }

   /*--( No new version available )----------------------------------------*/
   if( s_site_version == SIREN_VERSION )
   {  s_info = _( "You've got the last version" ) ;
      return( CVC_LAST_VERSION ) ;
   }

   /*--( New version available )-------------------------------------------*/
   s_info = s_site_version ;
   return( CVC_NEW_VERSION ) ;
   /*----------------------------------------------------------------------*/
}

/*--( This function is called from outside of this module )----------------*/
CCheckVersion::e_ret_code CCheckVersion::non_threaded_exec( wxString &s_info)
{
   /*----------------------------------------------------------------------*/
   return( exec( EM_NON_THREADED, s_info ) ) ;
   /*----------------------------------------------------------------------*/
}

/*--( This function is called from outside of this module )----------------*/
void CCheckVersion::thread_start()
{
   /*----------------------------------------------------------------------*/
   wxCriticalSectionLocker enter_csec( m_csec_p_thread ) ;
   /*----------------------------------------------------------------------*/
   if( m_p_thread != NULL ) { return ; }
   /*----------------------------------------------------------------------*/
   m_p_thread = new CThreadCheckVersion( this ) ;
   /*----------------------------------------------------------------------*/
   if( m_p_thread->Create() != wxTHREAD_NO_ERROR )
   {  wxFAIL ; return ; }
   /*----------------------------------------------------------------------*/
   if( m_p_thread->Run() != wxTHREAD_NO_ERROR )
   {  wxFAIL ; return ; }
   /*----------------------------------------------------------------------*/
}

/*--( This function is called from outside of this module )----------------*/
void CCheckVersion::thread_stop()
{
   /*----------------------------------------------------------------------*/
   m_boo_thread_stopping = true ;
   /*----------------------------------------------------------------------*/
   {  wxCriticalSectionLocker enter_csec( m_csec_p_thread ) ;
      /*-------------------------------------------------------------------*/
      if( m_p_thread != NULL )
      {  /*--( Stop asked means that no end event will be sent )-----------*/
         m_p_thread->set_send_end_event( SEEVT_DONT_SEND_END_EVENT ) ;
         /*----------------------------------------------------------------*/
         if( m_p_thread->Delete() != wxTHREAD_NO_ERROR )
         {  wxFAIL ; return ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*--( Wait until thread real death ! )----------------------------------*/
   while( true )
   {  /*-------------------------------------------------------------------*/
      {  wxCriticalSectionLocker enter_csec( m_csec_p_thread ) ;
         if( m_p_thread == NULL ) { break ; }
      }
      /*-------------------------------------------------------------------*/
      wxThread::This()->Sleep( 1 ) ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
   m_boo_thread_stopping = false ;
   /*----------------------------------------------------------------------*/
}

/*--( This function is called from outside of this module )----------------*/
bool CCheckVersion::thread_running()
{
   /*----------------------------------------------------------------------*/
   wxCriticalSectionLocker enter_csec( m_csec_p_thread ) ;
   /*----------------------------------------------------------------------*/
   return( m_p_thread != NULL && !m_boo_thread_stopping ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
CThreadCheckVersion::CThreadCheckVersion( CCheckVersion *p_cv )
                    : wxThread( wxTHREAD_DETACHED ),
                      m_p_cv( p_cv ),
                      m_seevt( SEEVT_DO_SEND_END_EVENT )
{  ; }

/*-------------------------------------------------------------------------*/
CThreadCheckVersion::~CThreadCheckVersion()
{
   /*----------------------------------------------------------------------*/
   wxCriticalSectionLocker enter_csec( m_p_cv->m_csec_p_thread ) ;
   m_p_cv->m_p_thread = NULL ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
wxThread::ExitCode CThreadCheckVersion::Entry()
{
   /*----------------------------------------------------------------------*/
   int      i_ret  ;
   wxString s_info ;

   /*----------------------------------------------------------------------*/
   i_ret = m_p_cv->exec( CCheckVersion::EM_THREADED, s_info ) ;

   /*--( The event may not have to be sent. For example if stopping )------*/
   if( get_send_end_event() != SEEVT_DO_SEND_END_EVENT )
   {  return( ( wxThread::ExitCode )0 ) ; }

   /*----------------------------------------------------------------------*/
   wxCommandEvent event( EVT_SR_CHECKVERSION_AUTO_END, wxID_ANY ) ;
   /*----------------------------------------------------------------------*/
   event.SetInt( i_ret ) ;
   event.SetString( s_info ) ;
   /*----------------------------------------------------------------------*/
   wxQueueEvent( wxGetApp().m_frame, new wxCommandEvent( event ) ) ;
   /*----------------------------------------------------------------------*/
   return( ( wxThread::ExitCode )0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                     End of file CCheckVersion.cpp                         !
+==========================================================================*/
