/*====================================+=====================================+
! File CApp.cpp                       ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#include <wx/fs_zip.h>
#include <wx/fs_inet.h>
#include <wx/stdpaths.h>
#include <wx/image.h>
#include <wx/stopwatch.h>
/*-------------------------------------------------------------------------*/
#include "siren_info.h"
#include "CApp.h"
#include "CDirTree.h"
#include "dynmenu/CFavourite.h"
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
bool CApp::OnInit()
{
   /*----------------------------------------------------------------------*/
   m_boo_init_done = false ;
   m_boo_cmdline_dir_loaded = false ;

   /*--( The locale object is necessary since wx 2.9.5 )-------------------*/
   m_locale.Init( wxLANGUAGE_DEFAULT, wxLOCALE_DONT_LOAD_DEFAULT ) ;
     m_s_system_canonical_name
   = wxLocale::GetLanguageCanonicalName( wxLocale::GetSystemLanguage() ) ;

   /*--( The command line analyse and parsing is done via this call )------*/
   if( !wxApp::OnInit() ) { return( false ) ; }

   /*--( By default, the application name is set to argv[0] )--------------*/
   SetAppName( SIREN_NAME ) ;
   /*----------------------------------------------------------------------*/
   wxFileSystem::AddHandler( new wxArchiveFSHandler() ) ;

   /*----------------------------------------------------------------------*/
   m_frame = new CFrame() ;

   /*--( Useless to load images if siren is used in "batch" mode )---------*/
   m_frame->set_disp_image( !m_boo_cmdline_quit ) ;
   /*--( Load handlers only if something is going to be displayed )--------*/
   if( m_frame->get_disp_image() ) { wxInitAllImageHandlers() ; }

   /*----------------------------------------------------------------------*/
   m_frame->set_recurse( m_boo_cmdline_recurse ) ;
   m_frame->m_loaddir_param.set_sel_filter( m_s_cmdline_sel_filter ) ;

   /*--( Initialize main window layout )-----------------------------------*/
   m_frame->init() ;

   /*--( First file given as command line parameter will have the focus )--*/
   if( m_as_cmdline_file.GetCount() > 0 )
   {  m_frame->m_loaddir_param.set_repos_name( m_as_cmdline_file[ 0 ] ) ; }

   /*--( Load directory and select the ones specified on command line )----*/
   if(    startup_load_directory() == 0
       && m_as_cmdline_file.GetCount() > 0
     )
   {  /*-------------------------------------------------------------------*/
      get_fl()->sel_name_array( m_as_cmdline_file ) ;
      /*-------------------------------------------------------------------*/
#ifdef __WXMSW__
      /*--( Under MSW this doesn't work if done in the IDLE slot ... )----*/
      get_fl()->init_view_sel() ;
      /*-------------------------------------------------------------------*/
#endif // __WXMSW__
      /*-------------------------------------------------------------------*/
   }

   /*--( Don't need these data any more )----------------------------------*/
   m_s_cmdline_sel_filter.clear() ;
   m_s_cmdline_dir.clear() ;
   m_as_cmdline_file.clear() ;

   /*--( Immediately quit ? )----------------------------------------------*/
   if( m_boo_cmdline_quit )
   {
      /*--------------------------------------------------------------------+
      ! This operation can be done at two different places because of the   !
      ! fact that a wxFileSystemWatcher instance can only be created once   !
      ! the event loop exists :                                             !
      ! - Here if Siren exits immediately                                   !
      ! - In "CFrame::OnFirstIdle" if no immediate exits is asked           !
      +--------------------------------------------------------------------*/
      apply_cmdline_oper() ;
      /*-------------------------------------------------------------------*/
      return( false ) ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   m_frame->Show() ;
   SetTopWindow( m_frame ) ;
   /*----------------------------------------------------------------------*/
   m_favourite.update_menu() ;
   m_tool.update_menu() ;
   /*----------------------------------------------------------------------*/
   m_frame->m_fl->SetFocus() ;

   /*--( Some work has to be done once all initialisations have been done )*/
   m_frame->Bind( wxEVT_IDLE, &CFrame::OnFirstIdle, m_frame ) ;

   /*----------------------------------------------------------------------*/
   m_boo_init_done = true ;
   /*----------------------------------------------------------------------*/
   return( true ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CApp::OnInitCmdLine( wxCmdLineParser &parser )
{
   /*----------------------------------------------------------------------*/
   static const wxCmdLineEntryDesc st_co_tb_cmd_line_desc[] =
   {
      { wxCMD_LINE_SWITCH, "v", "version",
        "print version information"
      },
      { wxCMD_LINE_SWITCH, "h", "help",
        "show this help message",
        wxCMD_LINE_VAL_NONE, wxCMD_LINE_OPTION_HELP
      },
      { wxCMD_LINE_OPTION, "d", "dir",
        "directory to load"
      },
      { wxCMD_LINE_OPTION, "f", "filter",
        "loading filter"
      },
      { wxCMD_LINE_OPTION, "s", "select",
        "selection filter"
      },
      { wxCMD_LINE_SWITCH, "x", "listhidden",
        "list hidden directories and files",
        wxCMD_LINE_VAL_NONE, wxCMD_LINE_SWITCH_NEGATABLE
      },
      { wxCMD_LINE_SWITCH, "t", "subdir",
        "force the loading of sub directories"
      },
      { wxCMD_LINE_OPTION, "e", "expression",
        "renaming expression"
      },
      { wxCMD_LINE_OPTION, "F", "favname",
        "name of a favourite expression"
      },
      { wxCMD_LINE_SWITCH, "r", "rename",
        "execute a rename operation"
      },
      { wxCMD_LINE_SWITCH, "c", "copy",
        "execute a copy operation"
      },
      { wxCMD_LINE_SWITCH, "q", "quit",
        "quit after rename/copy execution"
      },
      { wxCMD_LINE_OPTION, "l", "language",
        "language (en, en_US, fr_FR ...)"
      },
      { wxCMD_LINE_OPTION, "p", "paramfile",
        "configuration file"
      },
      { wxCMD_LINE_SWITCH, "P", "nosaveparam",
        "don't save configuration parameters"
      },
      { wxCMD_LINE_PARAM,  NULL, NULL, "file/directory names",
        wxCMD_LINE_VAL_STRING,
        wxCMD_LINE_PARAM_MULTIPLE | wxCMD_LINE_PARAM_OPTIONAL
      },
      { wxCMD_LINE_NONE }
   } ;

   /*----------------------------------------------------------------------*/
   parser.SetLogo( SIREN_DISPLAY_NAME " " SIREN_VERSION "\n"
                   SIREN_COPYRIGHT_ASCII " (" SIREN_HOST ")\n"
                   SIREN_DESCRIPTION "\n"
                   SIREN_LICENSE "\n"
                 ) ;
   parser.SetDesc( st_co_tb_cmd_line_desc ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
bool CApp::OnCmdLineParsed( wxCmdLineParser &parser )
{
   /*--( Version information => quit )-------------------------------------*/
   if( parser.Found( "v" ) )
   {  wxMessageOutput::Get()->Printf( "%s %s",
                                      SIREN_DISPLAY_NAME, SIREN_VERSION
                                    ) ;
      return( false ) ;
   }

   /*--( Executable directory )--------------------------------------------*/
   m_s_exe_path = wxFileName( wxStandardPaths::Get().GetExecutablePath()
                            ).GetPath( wxPATH_GET_VOLUME ) ;

   /*-----------------------------------------------------------------------+
   ! Setting up the directories where localized files will be searched for. !
   +-----------------------------------------------------------------------*/

   /*--( The executable's directory )--------------------------------------*/
   m_as_locale_path.Add( m_s_exe_path ) ;
   /*--( The "locale" sub directory )--------------------------------------*/
   wxFileName fn_locale_path( m_s_exe_path, "locale" ) ;
   m_as_locale_path.Add( fn_locale_path.GetFullPath() ) ;

   /*--( Or some levels up, in the main dev directory )--------------------*/
#ifdef __WXDEBUG__
   if( fn_locale_path.GetDirCount() >= 3 )
   {  /*-------------------------------------------------------------------*/
      fn_locale_path.RemoveLastDir() ;
      fn_locale_path.RemoveLastDir() ;
      fn_locale_path.RemoveLastDir() ;
      m_as_locale_path.Add( fn_locale_path.GetFullPath() ) ;
      /*-------------------------------------------------------------------*/
   }
#endif // __WXDEBUG__

   /*--( Add some specific "system" paths )--------------------------------*/
#ifdef __LINUX__
   /*--( Debian/Ubuntu paths after installation )--------------------------*/
   m_as_locale_path.Add( "/usr/share" ) ; // locale
   m_as_locale_path.Add( "/usr/share/" SIREN_NAME "/locale" ) ; // help
#elif !defined( __WXMSW__ )
#error Unknown platform to: determine paths to catalog file
#endif // __LINUX__

   /*--( Add all the paths to look into )----------------------------------*/
   for( size_t sz_num = 0 ; sz_num < m_as_locale_path.GetCount() ; ++sz_num )
   {  m_locale.AddCatalogLookupPathPrefix( m_as_locale_path[ sz_num ] ) ; }

   /*-----------------------------------------------------------------------+
   ! The name of the config file can be defined as command line parameter   !
   +-----------------------------------------------------------------------*/
     m_boo_default_config_file
   = (    !parser.Found( "p", &m_s_config_filename )
       || m_s_config_filename.empty()
     ) ;

   /*-----------------------------------------------------------------------+
   ! Under MSW the config file will be in the executable's directory        !
   ! Under GTK (and others) the default is used: ~/.appname                 !
   +-----------------------------------------------------------------------*/
#ifdef __WXMSW__
   /*----------------------------------------------------------------------*/
   if( m_boo_default_config_file )
   {  m_s_config_filename = SIREN_NAME ".ini" ; }
   /*----------------------------------------------------------------------*/
   wxFileName fn_config_file( m_s_config_filename ) ;
   if( fn_config_file.MakeAbsolute( m_s_exe_path ) )
   {  m_s_config_filename = fn_config_file.GetFullPath() ; }
   /*----------------------------------------------------------------------*/
#elif defined( __WXGTK__ )
     m_s_config_filename
   = wxFileConfig::GetLocalFile( SIREN_NAME ).GetFullPath() ;
#else
#error Unknown platform to: determine the config file full name
#endif // __WXGTK__

   /*-----------------------------------------------------------------------+
   ! Initialization of the configuration file.                              !
   ! If it doesn't exist, it means that the program has never been started  !
   +-----------------------------------------------------------------------*/
   m_boo_first_start_ever = !wxFileExists( m_s_config_filename ) ;

   /*----------------------------------------------------------------------*/
   bool boo_conv_from_v2 ;
   CConfig_init( m_s_config_filename, g_co_i_config_version,
                 boo_conv_from_v2
               ) ;

   /*--( Load/init info contained in the configuration file )--------------*/
   init_config_info() ;
   /*--( Change some data based on command line content )------------------*/
   init_vars_from_command_line( parser ) ;

   /*-----------------------------------------------------------------------+
   ! Language specification precedence:                                     !
   ! 1) command line, 2) config, 3) system                                  !
   +-----------------------------------------------------------------------*/
   wxString s_canonical_name ;
   if( !parser.Found( "l", &s_canonical_name ) )
   {  s_canonical_name = get_canonical_name() ; }
   /*--( The language change will only change the texts, not the locales )-*/
   wxTranslations *p_translations = new wxTranslations() ;
   p_translations->SetLanguage( s_canonical_name ) ;
   wxTranslations::Set( p_translations ) ;

   /*-----------------------------------------------------------------------+
   ! The standard catalog (wxstd.mo) is used for the help window.           !
   ! Why adding the standard catalog before siren's one ?                   !
   ! Unfortunately in the French "wxstd.po" "shift" is translated to        !
   ! "majuscule" which is not the right translation for "shift" in          !
   ! menu accelerators.                                                     !
   ! As the menu strings are stored translated in the "po" file and "maj"   !
   ! is used in them, the accelerators with "Maj+" are not correctly set    !
   ! via the "wxAcceleratorEntry::CompareAccelString".                      !
   ! "shift" is present in siren's and standard "po" files. As it seems     !
   ! that the last loaded translations are used siren's catalog is loaded   !
   ! last.                                                                  !
   ! Same thing for "OK", translated in French "wxstd" as "Appliquer".      !
   ! Therefore "OK" is present in siren's "po" translated to "OK".          !
   +-----------------------------------------------------------------------*/
   m_locale.AddCatalog( "wxstd" ) ;
   m_locale.AddCatalog( SIREN_NAME ) ;

   /*--( The group names are in the message catalog )----------------------*/
   m_colgroup.init_names() ;

   /*--( In config data, empty language means default one )----------------*/
   if( s_canonical_name != m_s_system_canonical_name )
   {  M_s_lang.set( s_canonical_name ) ; }

   /*--( Some examples are added once translation initialization is done )-*/
   if( m_boo_first_start_ever || boo_conv_from_v2 )
   {  /*-------------------------------------------------------------------*/
      m_favourite.create_default_elmts() ;
      m_tool.create_default_elmts() ;
      /*-------------------------------------------------------------------*/
   }

   /*----------------------------------------------------------------------*/
   return( true ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CApp::OnEventLoopEnter( wxEventLoopBase *loop )
{
   /*--( wxFileSystemWatcher needs an active loop to be created )----------*/
   if( m_frame != NULL )
   {  /*-------------------------------------------------------------------*/
      m_frame->create_fswatcher() ;
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CApp::init_config_info()
{
   /*--( Don't forget to add "groups" to keep in "CConfig_init" )----------*/
   m_colgroup.init_var() ;
   m_favourite.init_var( "/favourite/" ) ;
   m_tool.init_var( "/tool/" ) ;
   /*----------------------------------------------------------------------*/
   M_boo_toolbar_grp.init( "/current/toolbar_grp", false ) ;
   M_boo_three_toolbars.init( "/current/three_toolbars", true ) ;
   M_i_layout.init( "current/layout", LAYOUT_VISU_LEFT ) ;
   M_boo_show_statusbar.init( "/current/show_statusbar", true ) ;
   M_boo_thumbnail.init( "/current/thumbnail", true ) ;
   /*----------------------------------------------------------------------*/
   M_i_spl_ver_pos.init( "/current/spl_ver_pos", -1 ) ;
   M_i_spl_hor_pos.init( "/current/spl_hor_pos", -1 ) ;
   /*----------------------------------------------------------------------*/
   M_rec_main.init( "/current/rec_main" ) ;
   M_rec_completion.init( "/current/rec_completion" ) ;
   M_rec_completion_tb.init( "/current/rec_completion_tb" ) ;
   M_rec_prefs.init( "/current/rec_prefs" ) ;
   M_rec_wizard.init( "/current/rec_wizard" ) ;
   M_rec_testbed.init( "/current/rec_testbed" ) ;
   M_rec_filter.init( "/current/rec_filter" ) ;
   M_rec_filter_search.init( "/current/rec_filter_search" ) ;
   M_rec_manren.init( "/current/rec_manren" ) ;
   /*----------------------------------------------------------------------*/
   M_boo_main_maxi.init( "/current/main_maxi", false ) ;
   M_nb_sel_page.init( "/current/nb_sel_page" ) ;
   M_i_sort_subitem.init( "/current/sort_subitem", COL_NONE_NAME ) ;
   M_boo_sort_asc.init( "/current/sort_asc", true ) ;
   M_s_exp.init( "/current/exp" ) ;
   M_s_dir.init( "/current/dir" ) ;
   M_s_fil.init( "/current/fil" ) ;
   M_s_opfil.init( "/current/opfil_fil" ) ;
   M_as_exp.init( "/combo/exp_%02d", 15 ) ;
   M_as_dir.init( "/combo/dir_%02d", 15 ) ;
   M_as_fil.init( "/combo/fil_%02d", 15 ) ;
   M_as_opfil.init( "/combo/opfil_%02d", 15 ) ;
   M_boo_opfil_dir.init( "/current/opfil_inv", false ) ;
   M_boo_opfil_match.init( "/current/opfil_match", true ) ;
   /*----------------------------------------------------------------------*/
   M_s_lang.init( "/general/lang", wxEmptyString ) ;
   M_boo_chkver_auto.init( "/general/chkver_auto", false ) ;
   M_s_last_chkver_auto.init( "/general/last_chkver_auto" ) ;
   M_boo_copy_file_tms.init( "/general/copy_file_tms", false ) ;
   M_boo_extr_meta.init( "/general/extr_meta", true ) ;
   M_boo_file_type_detect.init( "/general/file_type_detect", true ) ;
   M_boo_loose_mp3_check.init( "/general/loose_mp3_check", true ) ;
   /*----------------------------------------------------------------------*/
   M_boo_list_alt_attr.init( "/display/list_alt_attr", true ) ;
   M_boo_grid_fl.init( "/display/grid_fl", false ) ;
   M_boo_list_hidden.init( "/display/list_hidden", false ) ;
   M_boo_natural_sort.init( "/display/natural_sort", true ) ;
   M_boo_fsizeb_fl.init( "/display/fsizeb_fl", false ) ;
   M_boo_list_dir.init( "/display/list_dir", true ) ;
   M_boo_list_dir_top.init( "/display/list_dir_top", true ) ;
   M_boo_filter_dir.init( "/display/filter_dir", true ) ;
   M_boo_dir_size_fl.init( "/display/dir_size_fl", false ) ;
   M_boo_img_rotation.init( "/display/img_rotation", true  ) ;
   M_boo_img_hq_manip.init( "/display/img_hq_manip", false ) ;
   M_boo_img_stretch.init( "/display/img_stretch", false ) ;
   M_boo_exp_fix.init( "/display/exp_fix", true ) ;
   M_boo_fil_fix.init( "/display/fil_fix", false ) ;
   M_s_fmt_date_fl.init( "/display/fmt_date_fl" ) ;
   /*----------------------------------------------------------------------*/
   M_tb_col.init( "/column/col_%03d", COL_NB, wxEmptyString ) ;
   M_boo_grp_auto.init( "/column/grp_auto", true ) ;
   M_boo_col_used.init( "/column/col_used", true ) ;
   /*----------------------------------------------------------------------*/
   M_boo_cbc_sep_tab.init( "/clipboard/col_sep_tab", true ) ;
   M_s_cbc_sep_char.init( "/clipboard/col_sep_char", ";" ) ;
   M_s_cbc_repl_sep_char.init( "/clipboard/repl_sep_char", "_" ) ;
   /*----------------------------------------------------------------------*/
   M_boo_imm_recompute.init( "/comp/imm_recompute", false ) ;
   M_boo_exp_dir_ext.init( "/comp/exp_dir_ext", true ) ;
   M_boo_id3v1aud_nvl.init( "/comp/id3v1aud_nvl" , true ) ;
   M_s_chars_tab_sep.init( "/comp/chars_tab_sep", "-" ) ;
   M_s_chars_word_sep.init( "/comp/chars_word_sep", " _()" ) ;
   /*----------------------------------------------------------------------*/
   M_i_max_name_len.init( "/comp/max_name_len", g_co_i_undef_max_name_len ) ;
   if(    M_i_max_name_len.get() != g_co_i_undef_max_name_len
       && M_i_max_name_len.get() <  g_co_i_min_name_len
     )
   {  M_i_max_name_len.set( g_co_i_undef_max_name_len ) ; }
   /*----------------------------------------------------------------------*/
   M_s_del_chars.init( "/comp/del_chars" ) ;
   M_s_del_repeat_chars.init( "/comp/del_repeat_chars" ) ;
   M_s_rep_chars_from.init( "/comp/rep_chars_from" ) ;
   M_s_rep_chars_to.init( "/comp/rep_chars_to" ) ;
   /*----------------------------------------------------------------------*/
   M_s_rep_char_inv.init( "/comp/rep_char_inv", "_" ) ;
   if(    M_s_rep_char_inv.empty()
       || wxFileName::GetForbiddenChars().find( M_s_rep_char_inv.get()[ 0 ]
                                              ) != wxString::npos
     )
   {  M_s_rep_char_inv.set( "_" ) ; }
   /*----------------------------------------------------------------------*/
   M_boo_force_case_word.init( "/comp/force_case_word", false ) ;
   M_as_force_case_word.init( "/comp/force_case_word/word_%03d", -1 ) ;
   /*----------------------------------------------------------------------*/
   M_i_sel_num_pad.init( "/comp/sel_num_pad", 3 )  ;
   M_i_sel_num_beg.init( "/comp/sel_num_beg", 1 )  ;
   M_i_sel_num_inc.init( "/comp/sel_num_inc", 1 )  ;
   M_i_num_in_name_pad.init( "/comp/num_in_name_pad", 3 ) ;
   M_i_num_in_name_add.init( "/comp/num_in_name_add", 0 ) ;
   M_i_track_pad.init( "/comp/track_pad", 2 ) ;
   M_i_rand_len.init( "/comp/rand_len", 8 ) ;
   /*----------------------------------------------------------------------*/
   M_s_fmt_date.init_mandatory( "/comp/fmt_d", "%Y%m%d" ) ;
   M_s_fmt_time.init_mandatory( "/comp/fmt_t", "%H%M%S" ) ;
   M_s_fmt_date_time.init_mandatory( "/comp/fmt_dt", "%Y%m%d_%H%M%S" ) ;
   M_i_fmt_date_add.init( "/comp/fmt_t_add", 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CApp::save_config_info()
{
   /*----------------------------------------------------------------------*/
   m_colgroup.save_var() ;
   m_favourite.save_var() ;
   m_tool.save_var() ;
   /*----------------------------------------------------------------------*/
   M_boo_toolbar_grp.save() ;
   M_boo_three_toolbars.save() ;
   M_i_layout.save() ;
   M_boo_show_statusbar.save() ;
   M_boo_thumbnail.save() ;
   /*----------------------------------------------------------------------*/
   M_i_spl_ver_pos.save() ;
   M_i_spl_hor_pos.save() ;
   /*----------------------------------------------------------------------*/
   M_rec_main.save() ;
   M_rec_completion.save() ;
   M_rec_completion_tb.save() ;
   M_rec_prefs.save() ;
   M_rec_wizard.save() ;
   M_rec_testbed.save() ;
   M_rec_filter.save() ;
   M_rec_filter_search.save() ;
   M_rec_manren.save() ;
   /*----------------------------------------------------------------------*/
   M_boo_main_maxi.save() ;
   M_nb_sel_page.save() ;
   M_i_sort_subitem.save() ;
   M_boo_sort_asc.save() ;
   M_s_exp.save() ;
   M_s_dir.save() ;
   M_s_fil.save() ;
   M_s_opfil.save() ;
   M_as_exp.save() ;
   M_as_dir.save() ;
   M_as_fil.save() ;
   M_as_opfil.save() ;
   M_boo_opfil_dir.save() ;
   M_boo_opfil_match.save() ;
   /*----------------------------------------------------------------------*/
   M_s_lang.save() ;
   M_boo_chkver_auto.save() ;
   M_s_last_chkver_auto.save() ;
   M_boo_copy_file_tms.save() ;
   M_boo_extr_meta.save() ;
   M_boo_file_type_detect.save() ;
   M_boo_loose_mp3_check.save() ;
   /*----------------------------------------------------------------------*/
   M_boo_list_alt_attr.save() ;
   M_boo_grid_fl.save() ;
   M_boo_list_hidden.save() ;
   M_boo_natural_sort.save() ;
   M_boo_fsizeb_fl.save() ;
   M_boo_list_dir.save() ;
   M_boo_list_dir_top.save() ;
   M_boo_filter_dir.save() ;
   M_boo_dir_size_fl.save() ;
   M_boo_img_rotation.save() ;
   M_boo_img_hq_manip.save() ;
   M_boo_img_stretch.save() ;
   M_boo_exp_fix.save() ;
   M_boo_fil_fix.save() ;
   M_s_fmt_date_fl.save() ;
   /*----------------------------------------------------------------------*/
   M_tb_col.save() ;
   M_boo_grp_auto.save() ;
   M_boo_col_used.save() ;
   /*----------------------------------------------------------------------*/
   M_boo_cbc_sep_tab.save() ;
   M_s_cbc_sep_char.save() ;
   M_s_cbc_repl_sep_char.save() ;
   /*----------------------------------------------------------------------*/
   M_boo_imm_recompute.save() ;
   M_boo_exp_dir_ext.save() ;
   M_boo_id3v1aud_nvl.save() ;
   M_s_chars_tab_sep.save() ;
   M_s_chars_word_sep.save() ;
   M_i_max_name_len.save() ;
   /*----------------------------------------------------------------------*/
   M_s_del_chars.save() ;
   M_s_del_repeat_chars.save() ;
   M_s_rep_chars_from.save() ;
   M_s_rep_chars_to.save() ;
   /*----------------------------------------------------------------------*/
   M_s_rep_char_inv.save() ;
   /*----------------------------------------------------------------------*/
   M_boo_force_case_word.save() ;
   M_as_force_case_word.save() ;
   /*----------------------------------------------------------------------*/
   M_i_sel_num_pad.save() ;
   M_i_sel_num_beg.save() ;
   M_i_sel_num_inc.save() ;
   M_i_num_in_name_pad.save() ;
   M_i_num_in_name_add.save() ;
   M_i_track_pad.save() ;
   M_i_rand_len.save() ;
   /*----------------------------------------------------------------------*/
   M_s_fmt_date.save() ;
   M_s_fmt_time.save() ;
   M_s_fmt_date_time.save() ;
   M_i_fmt_date_add.save() ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
bool CApp::is_path_ok_to_read( const wxString &s_path )
{
   /*----------------------------------------------------------------------*/
   return(    wxFileName::IsDirReadable( s_path )
           && (    !sr::is_file_attr_hidden( s_path )
                || M_boo_list_hidden.get()
              )
         ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CApp::init_vars_from_command_line( wxCmdLineParser &parser )
{
   /*----------------------------------------------------------------------*/
   wxString s_option ;

   /*--( Expression )------------------------------------------------------*/
   if( parser.Found( "e", &s_option ) )
   {  M_s_exp.set( s_option ) ; }
   /*--( Favourite name )--------------------------------------------------*/
   if( parser.Found( "F", &s_option ) )
   {  M_s_exp.set( m_favourite.get_exp( s_option ) ) ; }
   /*--( Filter )----------------------------------------------------------*/
   if( parser.Found( "f", &s_option ) )
   {  M_s_fil.set( s_option ) ; }
   parser.Found( "s", &m_s_cmdline_sel_filter ) ;
   m_boo_cmdline_recurse = parser.Found( "t" ) ;
   /*-----------------------------------------------------------------------+
   ! Switches can be "negated": followed by a '-'                           !
   ! If not specified on the command line, the default/config value is used !
   +-----------------------------------------------------------------------*/
   switch( parser.FoundSwitch( "x" ) )
   {  case wxCMD_SWITCH_OFF : M_boo_list_hidden.set( false ) ; break ;
      case wxCMD_SWITCH_ON  : M_boo_list_hidden.set( true )  ; break ;
      case wxCMD_SWITCH_NOT_FOUND : break ;
   }
   /*----------------------------------------------------------------------*/
   m_cmdline_otfile = OFILE_NB ;
   if( parser.Found( "r" ) )
   {  m_cmdline_otfile = OFILE_RENAME ; }
   if( parser.Found( "c" ) )
   {  m_cmdline_otfile = OFILE_COPY ; }
   /*----------------------------------------------------------------------*/
   m_boo_cmdline_quit = parser.Found( "q" ) ;
   m_boo_cmdline_save_config = !parser.Found( "P" ) ;

   /*--( The directory and possibly files to select (cf below for %xx) )---*/
   if( parser.Found( "d", &m_s_cmdline_dir ) )
   {  m_s_cmdline_dir = sr::replace_percent_xx( m_s_cmdline_dir ) ; }
   /*----------------------------------------------------------------------*/
   const int co_i_nb = parser.GetParamCount() ;
   /*----------------------------------------------------------------------*/
   if( co_i_nb > 0 )
   {
      /*-------------------------------------------------------------------*/
      wxFileName filename ;
      int        i_num    ;
      /*-------------------------------------------------------------------*/
      for( i_num = 0 ; i_num < co_i_nb ; ++i_num )
      {
         /*-----------------------------------------------------------------+
         ! The OS may have replaced special characters with "%xx"           !
         ! sequences. For example under Ubuntu 11.04 (natty) using          !
         ! Nautilus-Actions.                                                !
         +-----------------------------------------------------------------*/
         filename.Assign( sr::replace_percent_xx( parser.GetParam( i_num )));
         /*----------------------------------------------------------------*/
         if( m_s_cmdline_dir.empty() )
         {  m_s_cmdline_dir = filename.GetPath() ; }
         else /*--( Only one path can be used )----------------------------*/
         if(    filename.GetDirCount() > 0
             && m_s_cmdline_dir != filename.GetPath()
           )
         {  continue ; }
         /*----------------------------------------------------------------*/
         m_as_cmdline_file.Add( filename.GetFullName() ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CApp::startup_load_directory()
{
   /*----------------------------------------------------------------------*/
   e_loading_mode loading_mode ;

   /*--( For now the expected directory has not been loaded )--------------*/
   m_boo_cmdline_dir_loaded = false ;

   /*-----------------------------------------------------------------------+
   ! If a work (selection/rename/...) has to be done, the loading mode used !
   ! is non threaded because these tasks need the file list loading to be   !
   ! achieved. A non threaded loading ensures that.                         !
   +-----------------------------------------------------------------------*/
   if( m_cmdline_otfile != OFILE_NB || m_as_cmdline_file.GetCount() != 0 )
   {  loading_mode = LM_NON_THREADED ; }
   else
   {  loading_mode = LM_THREADED ; }

   /*-----------------------------------------------------------------------+
   ! A path given through the command line ?                                !
   +-----------------------------------------------------------------------*/
   if( !m_s_cmdline_dir.empty() )
   {
      /*--------------------------------------------------------------------+
      ! The user may have specified vars, dots ... dir that have to be      !
      ! expanded.                                                           !
      +--------------------------------------------------------------------*/
      wxString s_path( sr::expand_path( m_s_cmdline_dir ) ) ;
      /*-------------------------------------------------------------------*/
      if(    is_path_ok_to_read( s_path )
          && m_frame->change_directory( s_path, LM_NON_THREADED ) == 0
        )
      {  m_boo_cmdline_dir_loaded = true ;
         return( 0 ) ;
      }
      /*-------------------------------------------------------------------*/
   }

   /*-----------------------------------------------------------------------+
   ! Else load the last used directory                                      !
   +-----------------------------------------------------------------------*/
   if( !M_s_dir.empty() )
   {
      /*--( The last used one is ok ? )------------------------------------*/
      if(    is_path_ok_to_read( M_s_dir.get() )
          && m_frame->change_directory( M_s_dir.get(), loading_mode ) == 0
        )
      {  m_boo_cmdline_dir_loaded = true ;
         return( 0 ) ;
      }

      /*-------------------------------------------------------------------*/
      wxFileName fn( M_s_dir.get(), wxEmptyString ) ;

      /*--( If dir is not available the first parent ok will be used )-----*/
      if( fn.GetDirCount() > 0 )
      {
         /*----------------------------------------------------------------*/
         do
         {  fn.RemoveLastDir() ;
         } while(    !is_path_ok_to_read( fn.GetPath() )
                  && fn.GetDirCount() > 0
                ) ;
         /*----------------------------------------------------------------*/
         if(    fn.GetDirCount() > 0
             && m_frame->change_directory( fn.GetPath(), loading_mode ) == 0
           )
         {  return( 0 ) ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
   }

   /*--( Finally, the current directory is loaded )------------------------*/
   return( m_frame->change_directory( wxFileName::GetCwd(), loading_mode ) );
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
int CApp::apply_cmdline_oper()
{
   /*-----------------------------------------------------------------------+
   ! An operation requested is only applied if the directory loaded is the  !
   ! expected one (asked by the user)                                       !
   +-----------------------------------------------------------------------*/
   if( m_boo_cmdline_dir_loaded && m_cmdline_otfile != OFILE_NB )
   {  return( get_fl()->oper_do( m_cmdline_otfile ) ) ; }
   /*----------------------------------------------------------------------*/
   return( 0 ) ;
   /*----------------------------------------------------------------------*/
}

/*--( Search in the paths already set for the requested file name )--------*/
wxString CApp::translated_file_path( const wxString &s_filename )
{
   /*----------------------------------------------------------------------*/
   const wxArrayString *p_as_locale_path = &wxGetApp().get_locale_path() ;
   wxString            s_path_name      ;
   wxString            s_path_lang      ;
   wxString            s_path_lang_base ;
   wxString            s_fullpath       ;
   size_t              sz_num           ;

   /*----------------------------------------------------------------------*/
   s_path_name = wxFileName::GetPathSeparator() + s_filename ;
   /*----------------------------------------------------------------------*/
     s_path_lang
   = wxFileName::GetPathSeparator() + get_canonical_name() + s_path_name ;
   /*--( The base part of the language: fr_FR => fr )----------------------*/
   s_path_lang_base = get_canonical_name().BeforeFirst( '_' ) ;
   if( !s_path_lang_base.empty() )
   {  s_path_lang_base.insert( 0, wxFileName::GetPathSeparator() ) ;
      s_path_lang_base.append( s_path_name ) ;
   }

   /*--( Looking in the already set paths list )---------------------------*/
   for( sz_num = 0 ; sz_num < p_as_locale_path->GetCount() ; ++sz_num )
   {
      /*--( First, look in the full language path ? )----------------------*/
      s_fullpath = p_as_locale_path->Item( sz_num ) + s_path_lang ;
      if( wxFileExists( s_fullpath ) ) { return( s_fullpath ) ; }
      /*--( Then, if any, in the base language path )----------------------*/
      if( !s_path_lang_base.empty() )
      {  /*----------------------------------------------------------------*/
         s_fullpath = p_as_locale_path->Item( sz_num ) + s_path_lang_base ;
         if( wxFileExists( s_fullpath ) ) { return( s_fullpath ) ; }
         /*----------------------------------------------------------------*/
      }
      /*--( Then in the directory itself )---------------------------------*/
      s_fullpath = p_as_locale_path->Item( sz_num ) + s_path_name ;
      if( wxFileExists( s_fullpath ) ) { return( s_fullpath ) ; }
      /*-------------------------------------------------------------------*/
   }

   /*--( By default, a search is done in the executable directory )--------*/
   s_fullpath = wxFileName( m_s_exe_path, s_filename ).GetFullPath() ;
   if( wxFileExists( s_fullpath ) )
   {  return( s_fullpath ) ; }

   /*----------------------------------------------------------------------*/
   return( wxEmptyString ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CApp::tlw_set_rect( wxWindow *p_win, const wxRect &rect )
{
   /*----------------------------------------------------------------------*/
   if( rect.GetSize() == wxDefaultSize )
   {  p_win->SetSize( p_win->GetMinSize() ) ; }
   else
   {  p_win->SetSize( rect.GetSize() ) ; }
   /*----------------------------------------------------------------------*/
   if( rect.GetPosition() == wxDefaultPosition )
   {  p_win->CenterOnParent() ; }
   else
   {  p_win->SetPosition( rect.GetPosition() ) ; }
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CApp::init_tlw_rect( wxWindow *p_win, const CConfigRect &cfg_rect )
{
   /*----------------------------------------------------------------------*/
   p_win->SetMinSize( p_win->GetEffectiveMinSize() ) ;
   /*----------------------------------------------------------------------*/
   tlw_set_rect( p_win, cfg_rect.get() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/
void CApp::save_tlw_rect( wxWindow *p_win, CConfigRect &cfg_rect )
{
   /*----------------------------------------------------------------------*/
   cfg_rect.set( p_win->GetRect() ) ;
   /*----------------------------------------------------------------------*/
}

/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                         End of file CApp.cpp                              !
+==========================================================================*/
