/*====================================+=====================================+
! File CFileInit.h                    ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+---------------------------------------------------------------------------+
!                                                                           !
!                    Extract data associated to a file                      !
!                                                                           !
+-------------------------------------+-------------------------------------+
! Principle of the column reservation !                                     !
+-------------------------------------+                                     !
! Reserving a column means flagging it as used.                             !
! Checking for a column reservation can't simply be a test of emptiness.    !
!                                                                           !
! No format specification clearly explains what to do in case of metadata   !
! repetition (with different values, of course ...)                         !
! I've chosen as "base": "The first to speak is right"                      !
! Unfortunately this rule has limits !                                      !
! Here is a list that I hope as exhaustive as possible of the special       !
! cases:                                                                    !
!                                                                           !
! - MP3: COMM (comment tag)                                                 !
!   This tag has an associated description (tag subinfo) and some softs     !
!   like "Music Match" use it to store some "personal" data without         !
!   touching the "real" comment.                                            !
!   What I've decided to do in this case is to keep the last non-empty      !
!   value (value and not description).                                      !
!                                                                           !
! - Exif: data associated to thumbnails.                                    !
!   I've added this one after adding Exif in TIFFs and some raw formats (in !
!   particularly NEF).                                                      !
!   From one "SubIfd" to another (different image formats) some tags may    !
!   not be repeated.                                                        !
!   The chosen solution: if it's a thumbnail unreserved info, I store it    !
!   without reservation. If it'a "pure image" (not thumbnail) I reserve     !
!   and store it.                                                           !
!                                                                           !
! - CIFF: string data type                                                  !
!   The nesting "heaps" system and apparently systematic repetition of some !
!   tags oblige me to reserve/store only for non empty strings.             !
!                                                                           !
+-------------+-------------------------------------------------------------+
! Date & Time !                                                             !
+-------------+                                                             !
! When clearly indicated in the specifications the date & time is converted !
! to local time zone. These cases are rare: pdf, iso.                       !
!                                                                           !
+------------+--------------------------------------------------------------+
! Image info !                                                              !
+------------+                                                              !
! Some image types define their own metadata format.                        !
! Even if they are not Exif their data will be stored in the Exif group.    !
! For example: tga, crw, webp, x3f                                          !
!                                                                           !
+--------------+------------------------------------------------------------+
! Translations !                                                            !
+--------------+                                                            !
! I've decided to keep in English/original language all extracted           !
! information. For example, all the "constants" associated to EXIF data.    !
! To be coherent even the "built" ones are not translated ("%d icons" ...)  !
!                                                                           !
! Only a very few explanatory messages are translated: pdf encryption ...   !
!                                                                           !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#ifndef CFILEINIT_H
#define CFILEINIT_H
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
#include <set>
#include <wx/zipstrm.h>
#include "siren.h"
#include "CFileAccess.h"
#include "common/sr_string.h"
/*-------------------------------------------------------------------------*/
class CFile ;
class CFileInit ;
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
typedef int ( CFileInit::*t_p_init_func )() ;

/*-------------------------------------------------------------------------*/
enum e_bitrate_type
{
   /*----------------------------------------------------------------------*/
   TBR_CBR      , // Constant bitrate
   TBR_ABR      , // Average bitrage
   TBR_VBR      , // Variable bitrate
   TBR_VBR_XING ,
   TBR_VBR_FH
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_mp3_frame_info
{
   /*----------------------------------------------------------------------*/
   enum e_mpeg_type { MPEG_TYPE_1, MPEG_TYPE_2, MPEG_TYPE_NB } ;
   /*----------------------------------------------------------------------*/
   int         i_mpeg_version ;
   e_mpeg_type mpeg_type      ;
   int         i_layer_ind    ;
   int         i_bitrate      ;
   int         i_samprate     ;
   int         i_padding      ;
   int         i_channel      ;
   int         i_frame_len    ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
class CFileInit
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit( CFile &file_treated ) ;
      ~CFileInit() ;
      /*-------------------------------------------------------------------*/

   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFile       &m_f ;
      CFileAccess m_fa ;
      /*-------------------------------------------------------------------*/
      wxString        m_s_type_det ; // Currently detected file type
      sr::e_conv_type m_conv_type  ;

      /*--------------------------------------------------------------------+
      ! Column "reservation" information.                                   !
      ! Cf to the top of this file for more info.                           !
      +--------------------------------------------------------------------*/
      std::set< int > m_set_reserved_col ;

   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      void reset_col_set()
      {  m_set_reserved_col.clear() ; }
      /*--------------------------------------------------------------------+
      ! If the value is present the insert returned second pair element is  !
      ! false                                                               !
      +--------------------------------------------------------------------*/
      bool reserve_col( int i_col )
      {  return( i_col >= 0 && i_col < COL_NB
                 ? m_set_reserved_col.insert( i_col ).second : false
               ) ;
      }
      bool is_col_reserved( int i_col ) const
      {  return( m_set_reserved_col.count( i_col ) != 0 ) ; }
      /*-------------------------------------------------------------------*/
      int prepare_string( const char *p_c_val, int i_len, wxString &s
                        ) const
      {  return( sr::prepare_string( p_c_val, i_len, s, m_conv_type ) ) ; }
      int prepare_string( wxString &s )
      {  return( sr::clean_string( s ) ) ; }

      /*--( String conversion type )---------------------------------------*/
      sr::e_conv_type get_conv_type() const { return( m_conv_type ) ; }
      void set_conv_type( sr::e_conv_type conv_type )
      {  m_conv_type = conv_type ; }
      void set_conv_from_8bit() { m_conv_type = sr::CONV_FROM_8BIT ; }
      void set_conv_from_utf8() { m_conv_type = sr::CONV_FROM_UTF8 ; }
      void set_conv_from_mac()  { m_conv_type = sr::CONV_FROM_MAC  ; }
      void set_conv_from_utf16le()
      {  m_conv_type = sr::CONV_FROM_UTF16LE ; }
      void set_conv_from_utf16be()
      {  m_conv_type = sr::CONV_FROM_UTF16BE ; }
      void set_conv_from_bom_utf16lebe()
      {  m_conv_type = sr::CONV_FROM_UTF16LEBE ; }

      /*--------------------------------------------------------------------+
      ! CFileInit_Base.cpp                                                  !
      +--------------------------------------------------------------------*/
      void init_mini_name( const wxString &s_name ) ;
      void init_full( const wxString &s_path,
                      const wxString &s_name,
                      bool boo_is_dir
                    ) ;
      /*-------------------------------------------------------------------*/
      void init_date( int i_col, const wxDateTime &dt )                     ;
      void init_attr()                                                      ;
      void init_img_x_y( int i_x, int i_y )                                 ;
      void init_img_bpp( int i_bpp )                                        ;
      void init_col_mp3_channel( int i_col, int i_channel )                 ;
      void init_audio_channel( int i_channel )                              ;
      void init_audio_samprate( int i_samprate )                            ;
      void init_audio_duration( int i_nb_sec )                              ;
      void init_audio_bitrate( e_bitrate_type bitrate_type, int i_bitrate ) ;
      void init_video_x_y( int i_x, int i_y )                               ;
      void init_video_duration( int i_nb_sec )                              ;
      void init_audio_num_and_nb_track( const wxString &s_track )           ;
      void init_video_fps( double do_fps )                                  ;
      void init_video_channel( int i_channel )                              ;
      void init_video_samprate( int i_samprate )                            ;
      void init_exif_shutterspeedvalue( double do_val )                     ;
      void init_exif_focallength( double do_val )                           ;
      void init_exif_focallength35mmfilm( long l_val )                      ;
      void init_exif_exposuretime( unsigned long ul_nomin,
                                   unsigned long ul_denomin
                                 ) ;
      void init_exif_rotation( e_iv_rotation iv_rotation )                  ;
      void init_doc_nb_files( int i_nb )                                    ;
      void init_doc_zip_info( wxZipInputStream &zip_input )                 ;
      int  file_read_tb_c( size_t sz_size, wxString &s ) ;
      int  file_read_tb_wc_le( size_t sz_size, wxString &s ) ;
      int  riff_read_serie_info( wxUint32 &dw_size_list,
                                 int ( *p_f_conv_tag_col )( wxUint32 dw_id )
                               ) ;
      int  read_embedded_image_info( wxFileOffset fo_offset,
                                     wxULongLong ull_size,
                                     const wxString &s_mime
                                   ) ;
      void add_codec_info( const wxString &s_info, wxString &s_result
                         ) const ;
      wxString concat_av_codecs( const wxString &s_audio,
                                 const wxString &s_video
                               ) const ;

      /*--------------------------------------------------------------------+
      ! CFileInit_Type.cpp                                                  !
      +--------------------------------------------------------------------*/
      wxString get_type_desc() ;
      t_p_init_func get_init_func_ext() const ;
      t_p_init_func get_init_func_detect() ;
      bool info_fcc_video( const char * const p_c_fcc, wxString &s_info
                         ) const ;

      /*--------------------------------------------------------------------+
      ! The following functions are defined in the CFileInit_....cpp files  !
      +--------------------------------------------------------------------*/
      bool mp3_genre_id_ok( int i_genre_id ) const ;
      wxString mp3_get_genre( int i_genre_id ) const ;
      int mp3_extr_frame_info( wxUint32 dw_val, str_mp3_frame_info &s_fi ) ;
      int mp3_search_next_frame_header( str_mp3_frame_info &s_fi,
                                        int i_max_read_len
                                      ) ;
      wxUint32 mp3_conv_id3v2_synch_size( wxUint32 dw_size ) ;
      int mp3_read_id3v1( wxFileOffset &fo_offset_data_end ) ;
      int mp3_read_id3v2() ;
      /*-------------------------------------------------------------------*/
      int ogg_read_vorbis_comment( bool boo_video ) ;
      int ape_init_apetag( wxFileOffset &fo_offset_end ) ;
      int tiff_read_exif( wxFileOffset fo_offset, wxULongLong ull_size ) ;
      int tiff_read_tiff( wxFileOffset fo_offset, wxULongLong ull_size ) ;
      int read_iptc( wxFileOffset fo_offset, size_t sz_size,
                     bool boo_photoshop
                   ) ;

      /*-------------------------------------------------------------------*/
      int init_mp3() ;
      int init_kar() ;
      int init_jpg() ;
      int init_gif() ;
      int init_pcx() ;
      int init_bmp() ;
      int init_ico() ;
      int init_png() ;
      int init_url() ;
      int init_au()  ;
      int init_ogg() ;
      int init_avi() ;
      int init_mpg() ;
      int init_qtf() ;
      int init_asf() ;
      int init_wav() ;
      int init_ape() ;
      int init_mpc() ;
      int init_torrent() ;
      int init_pdf() ;
      int init_x3f() ;
      int init_mrw() ;
      int init_raf() ;
      int init_crw() ;
      int init_tiff() ;
      /*-------------------------------------------------------------------*/
      int  png_read( wxFileOffset fo_image_offset,
                     wxULongLong  ull_image_size
                   ) ;
      int  bmp_read( wxFileOffset fo_image_offset,
                     wxULongLong  ull_image_size
                   ) ;
      int  ico_read( wxFileOffset fo_image_offset,
                     wxULongLong  ull_image_size
                   ) ;
      int  pcx_read( wxFileOffset fo_image_offset,
                     wxULongLong  ull_image_size
                   ) ;
      int  gif_read( wxFileOffset fo_image_offset,
                     wxULongLong  ull_image_size
                   ) ;
      int  jpg_read( wxFileOffset fo_image_offset,
                     wxULongLong  ull_image_size
                   ) ;
      int  tga_read( wxFileOffset WXUNUSED( fo_offset_image ),
                     wxULongLong  ull_image_size
                   ) ;
      int  xpm_read( wxFileOffset WXUNUSED( fo_image_offset ),
                     wxULongLong WXUNUSED( ull_image_size )
                   ) ;
      int  pnm_read( wxFileOffset WXUNUSED( fo_image_offset ),
                     wxULongLong WXUNUSED( ull_image_size )
                   ) ;
      int  webp_read( wxFileOffset WXUNUSED( fo_image_offset ),
                      wxULongLong WXUNUSED( ull_image_size )
                    ) ;
      /*-------------------------------------------------------------------*/
      int init_tga() ;
      int init_xpm() ;
      int init_pnm() ;
      int init_webp() ;
      int init_svg() ;
      int init_html() ;
      int init_zip() ;
      int init_deb() ;
      int init_rpm() ;
      int init_epub() ;
      int init_odf() ;
      int init_flv() ;
      int init_swf() ;
      int init_mkv() ;
      int init_real() ;
      int init_flac() ;
      int init_aac() ;
      int init_ttf() ;
      int init_eml() ;
      int init_desktop() ;
      int init_iso() ;
      int init_xpi() ;
      int init_crx() ;
      int init_apk() ;
      int init_mo() ;
      /*-------------------------------------------------------------------*/
#ifdef __WXMSW__
      /*-------------------------------------------------------------------*/
      int init_lnk() ;
      int init_exe() ;
      int init_off() ;
      /*-------------------------------------------------------------------*/
#endif // __WXMSW__

   /*----------------------------------------------------------------------*/
} ;

/*--------------------------------------------------------------------------+
! This class is used as base in each module specifically oriented to a file !
! type (avi, pdf ...)                                                       !
+--------------------------------------------------------------------------*/
class CFileInit_type_base
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileInit_type_base( CFileInit &parent ) :
                       m_fi( parent ), m_fa( parent.m_fa ), m_f( parent.m_f )
      {  ; }
      /*-------------------------------------------------------------------*/
      CFileInit   &m_fi ;
      CFileAccess &m_fa ;
      CFile       &m_f  ;

   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
#endif // CFILEINIT_H
/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                         End of file CFileInit.h                           !
+==========================================================================*/
