/*====================================+=====================================+
! File CFileAccess.h                  ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#ifndef CFILEACCESS_H
#define CFILEACCESS_H
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
#include <wx/file.h>
#include <wx/filename.h>
/*-------------------------------------------------------------------------*/



/*--------------------------------------------------------------------------+
! Data structures used to read files in the "CFileInit" module              !
! The elements names beginning with "___" are not supposed to be used by    !
! the application. That's why they are not "swapped".                       !
+--------------------------------------------------------------------------*/
#pragma pack( push, 1 )

/*-------------------------------------------------------------------------*/
struct str_chunk
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_id   ;
   wxUint32 dw_size ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_atom
{
   /*----------------------------------------------------------------------*/
   wxUint32 dw_size ;
   wxUint32 dw_id   ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_atom_short
{
   /*----------------------------------------------------------------------*/
   wxUint16 w_size ;
   wxUint16 w_id   ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_riff_header
{
   /*----------------------------------------------------------------------*/
   str_chunk main_chunk ; // 'RIFF' & size
   wxUint32  dw_id      ; // Id of the file: AVI, WEBP ...
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_WAVEFORMATEX
{
   /*----------------------------------------------------------------------*/
   wxUint16 wFormatTag      ; // format type
   wxUint16 nChannels       ; // nb channels (i.e. mono, stereo...)
   wxUint32 nSamplesPerSec  ; // sample rate
   wxUint32 nAvgBytesPerSec ; // Average number of bytes per second
   wxUint16 nBlockAlign     ; // block size of data
   wxUint16 wBitsPerSample  ; // Number of bits per sample of mono data
   wxUint16 cbSize          ; // The count in bytes of the size of
                              // extra information (after cbSize)
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
struct str_BITMAPINFOHEADER
{
   /*----------------------------------------------------------------------*/
   wxUint32 biSize          ;
   wxInt32  biWidth         ;
   wxInt32  biHeight        ;
   wxUint16 biPlanes        ;
   wxUint16 biBitCount      ;
   wxUint32 biCompression   ;
   wxUint32 biSizeImage     ;
   wxInt32  biXPelsPerMeter ;
   wxInt32  biYPelsPerMeter ;
   wxUint32 biClrUsed       ;
   wxUint32 biClrImportant  ;
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
#pragma pack( pop )



/*-------------------------------------------------------------------------*/
class CFileAccess : public wxFile
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      int open_to_read( const wxString &s_name )
      {  return( Open( s_name, wxFile::read ) ? 0 : -1 ) ; }
      int open_to_read( const wxFileName &name )
      {  return( open_to_read( name.GetFullName() ) ) ; }
      /*-------------------------------------------------------------------*/
      int open_to_write( const wxString &s_name )
      {  return( Open( s_name, wxFile::write ) ? 0 : -1 ) ; }
      int open_to_write( const wxFileName &name )
      {  return( open_to_write( name.GetFullName() ) ) ; }
      /*-------------------------------------------------------------------*/
      int create_zero( const wxString &s_name )
      {  return( Create( s_name, false ) ? 0 : -1 ) ; }
      int create_zero( const wxFileName &name )
      {  return( create_zero( name.GetFullName() ) ) ; }
      /*-------------------------------------------------------------------*/
      wxFileOffset get_offset() const
      {  return( Tell() ) ; }
      /*-------------------------------------------------------------------*/
      void close()
      {  Close() ; }
      /*-------------------------------------------------------------------*/
      int set_offset( wxFileOffset val )
      {  return( val >= 0 && Seek( val ) != wxInvalidOffset ? 0 : -1 ) ; }
      int set_offset_from_end( wxFileOffset val )
      {  return( val >= 0 && SeekEnd( - val ) != wxInvalidOffset ? 0 : -1 );}
      int skip_nb_byte( wxFileOffset val )
      {  return( Seek( val, wxFromCurrent ) != wxInvalidOffset ? 0 : -1  ) ;}
      /*-------------------------------------------------------------------*/
      int write_buffer( const void *p_buffer, size_t sz )
      {  return( Write( p_buffer, sz ) == sz ? 0 : -1 ) ; }
      int write_string( const void *p_c )
      {  return( write_buffer( p_c, strlen( ( char * )p_c ) ) ) ; }
      int write_string( const wxString &s )
      {  return( Write( s ) ? 0 : -1 ) ; }

      /*-------------------------------------------------------------------*/
      wxFileOffset get_end_offset() ;
      int read_buffer( size_t sz, void *p_buffer ) ;
      int read_buffer( size_t sz, wxMemoryBuffer &mb ) ;
      int read_buffer_max( size_t &sz, void *p_buffer ) ;
      int search_next_seq( const wxUint8 b, size_t sz_size,
                           wxFileOffset &fo_offset
                         ) ;
      int search_prev_seq( const wxUint32 dw, size_t sz_size,
                           wxFileOffset &fo_offset
                         ) ;
      int search_last_seq( const wxUint32 dw, size_t sz_size,
                           wxFileOffset &fo_offset
                         ) ;
      /*-------------------------------------------------------------------*/
      int  read_data( wxUint8 &b ) ;
      int  read_data( wxUint32 &dw ) ;
      int  read_be_data( size_t sz, wxUint32 &dw ) ;
      int  read_be_data( size_t sz, wxUint64 &ddw ) ;
      int  read_le_data( wxUint16 &w ) ;
      int  read_be_data( wxUint16 &w ) ;
      int  read_le_data( wxUint32 &dw ) ;
      int  read_be_data( wxUint32 &dw ) ;
      int  read_le_data( wxUint64 &ddw ) ;
      int  read_be_data( wxUint64 &ddw ) ;
      int  read_le_data( str_chunk &chunk ) ;
      int  read_be_data( str_chunk &chunk ) ;
      int  read_be_data( str_atom &atom ) ;
      int  read_be_data( str_atom_short &atom_short ) ;
      int  read_le_data( str_riff_header &rh ) ;
      int  read_le_data( str_WAVEFORMATEX &wfx,
                         size_t sz_size = sizeof( str_WAVEFORMATEX )
                       ) ;
      int  read_le_data( str_BITMAPINFOHEADER &bih ) ;

   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
#endif // CFILEACCESS_H
/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                       End of file CFileAccess.h                           !
+==========================================================================*/
