/*====================================+=====================================+
! File CFile.h                        ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#ifndef CFILE_H
#define CFILE_H
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
#include <map>
#include <wx/log.h>
#include <wx/filename.h>
#include "common/sr_lib.h"
#include "common/sr_string.h"
#include "common/CExternalData.h"
#include "siren.h"
#include "CRunInfoMessage.h"
/*-------------------------------------------------------------------------*/
class CFile ;
class CDir ;
/*-------------------------------------------------------------------------*/



/*--------------------------------------------------------------------------+
! Information used during the computation of new name                       !
+--------------------------------------------------------------------------*/
typedef struct
{
   /*----------------------------------------------------------------------*/
   bool             boo_first_sub_exp ;
   /*----------------------------------------------------------------------*/
   sr::wxString_cit it_exp            ;
   sr::wxString_cit it_exp_end        ;
   wxString         s_current_value   ;
   /*----------------------------------------------------------------------*/
} t_exp_comp_info ;
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
class CFile
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFile() : m_i_sel_num( -1 )
      {  ; }
      /*--------------------------------------------------------------------+
      ! Infos associated to the file:                                       !
      ! The maps are indexed on the column codes "COL_..."                  !
      ! "m_s_val" is used for display and compute, the others are used for  !
      ! the sorts.                                                          !
      +--------------------------------------------------------------------*/
      typedef std::map< int, wxString >   t_s_val      ;
      typedef t_s_val::iterator           t_s_val_it   ;
      typedef t_s_val::const_iterator     t_s_val_cit  ;
      /*-------------------------------------------------------------------*/
      typedef std::map< int, wxLongLong > t_ll_val     ;
      typedef t_ll_val::iterator          t_ll_val_it  ;
      typedef t_ll_val::const_iterator    t_ll_val_cit ;
      /*-------------------------------------------------------------------*/
      typedef std::map< int, double >     t_do_val     ;
      typedef t_do_val::iterator          t_do_val_it  ;
      typedef t_do_val::const_iterator    t_do_val_cit ;
      /*-------------------------------------------------------------------*/
      typedef std::map< int, wxDateTime > t_dt_val     ;
      typedef t_dt_val::iterator          t_dt_val_it  ;
      typedef t_dt_val::const_iterator    t_dt_val_cit ;
   /*----------------------------------------------------------------------*/
   private :
      /*--( Columns information )------------------------------------------*/
      t_s_val  m_s_val  ;
      t_ll_val m_ll_val ;
      t_do_val m_do_val ;
      t_dt_val m_dt_val ;

      /*--( > 0 => selected )----------------------------------------------*/
      int        m_i_sel_num ;
      /*-------------------------------------------------------------------*/
      wxFileName m_name  ;
      int        m_i_ico ;
      /*-------------------------------------------------------------------*/
      bool        m_boo_is_dir ;
      wxULongLong m_ull_size   ;
      wxFileName  m_new_name   ;

      /*--------------------------------------------------------------------+
      ! Parent container. Useful when :                                     !
      ! - computing information relative to other files: %np, %nc ...       !
      ! - add / remove file info from selection sums (size, duration ...)   !
      +--------------------------------------------------------------------*/
      CDir *m_p_cdir ;

      /*--( Keep name history for undo operations )------------------------*/
      struct str_oper
      {  wxFileName from ;
         wxFileName to   ;
      } ;
      typedef std::map< int , str_oper > t_undo     ;
      typedef t_undo::iterator           t_undo_it  ;
      typedef t_undo::const_iterator     t_undo_cit ;
      t_undo m_undo ;

      /*--( System error message of the last rename/copy ... operations )--*/
      wxString     m_s_oper_err ;
      /*--( Size and offset should be same size ... )----------------------*/
      wxFileOffset m_fo_image_offset ;
      wxULongLong  m_ull_image_size  ;
      /*-------------------------------------------------------------------*/
      wxFileOffset m_fo_thumbnail_offset ;
      wxULongLong  m_ull_thumbnail_size  ;

      /*--------------------------------------------------------------------+
      ! Expressed in seconds. Used to indicate the total "selection"        !
      ! duration. No matter of the file type.                               !
      +--------------------------------------------------------------------*/
      int          m_i_duration ;

      /*-------------------------------------------------------------------*/
      e_iv_rotation m_img_rotation ;

      /*--------------------------------------------------------------------+
      ! Checksum information with their computation date                    !
      ! These variables are declared "mutable" because their computation    !
      ! can take a while and their values are stored in the object to       !
      ! "improve" performance.                                              !
      ! This allows to mark the functions as const ...                      !
      +--------------------------------------------------------------------*/
      mutable wxDateTime m_dt_md5   ;
      mutable wxString   m_s_md5    ;
      mutable wxDateTime m_dt_sha1  ;
      mutable wxString   m_s_sha1   ;
      mutable wxDateTime m_dt_crc32 ;
      mutable wxString   m_s_crc32  ;

   /*----------------------------------------------------------------------*/
   public :
      /*--------------------------------------------------------------------+
      ! External data used to handle "global" information for expression    !
      ! evaluation.                                                         !
      +--------------------------------------------------------------------*/
      static CExternalData_Systime   st_m_ed_st ;
      static CExternalData_Clipboard st_m_ed_cp ;
      static CExternalData_File      st_m_ed_fi ;

   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      wxString &val_s( int i_col ) { return( m_s_val[ i_col ] ) ; }
      wxLongLong &val_ll( int i_col ) { return( m_ll_val[ i_col ] ) ; }
      double &val_do( int i_col ) { return( m_do_val[ i_col ] ) ; }
      wxDateTime &val_dt( int i_col ) { return( m_dt_val[ i_col ] ) ; }

      /*-------------------------------------------------------------------*/
      const t_s_val &get_map_s_val() const
      {  return( m_s_val ) ; }
      t_s_val_it find_map_s_val( int i_col )
      {  return( m_s_val.find( i_col ) ) ; }
      /*-------------------------------------------------------------------*/
      wxFileOffset get_image_offset() const
      {  return( m_fo_image_offset ) ; }
      void set_image_offset( const wxFileOffset &fo )
      {  m_fo_image_offset = fo ; }
      wxULongLong get_image_size() const
      {  return( m_ull_image_size ) ; }
      void set_image_size( const wxULongLong &ull )
      {  m_ull_image_size = ull ; }
      /*-------------------------------------------------------------------*/
      wxFileOffset get_thumbnail_offset() const
      {  return( m_fo_thumbnail_offset ) ; }
      void set_thumbnail_offset( const wxFileOffset &fo )
      {  m_fo_thumbnail_offset = fo ; }
      wxULongLong get_thumbnail_size() const
      {  return( m_ull_thumbnail_size ) ; }
      void set_thumbnail_size( const wxULongLong &ull_size )
      {  m_ull_thumbnail_size = ull_size ; }
      /*-------------------------------------------------------------------*/
      bool has_image() const
      {  return( m_fo_image_offset != -1 ) ; }
      bool has_thumbnail() const
      {  return( m_fo_thumbnail_offset != -1 ) ; }
      /*-------------------------------------------------------------------*/
      e_iv_rotation get_img_rotation() const
      {  return( m_img_rotation ) ; }
      void set_img_rotation( e_iv_rotation iv_rotation )
      {  m_img_rotation = iv_rotation ; }
      /*-------------------------------------------------------------------*/
      size_t get_nb_undo() const
      {  return( m_undo.size() ) ; }
      t_undo &get_undo()
      {  return( m_undo ) ; }
      void swap_undo( CFile &f )
      {  m_undo.swap( f.get_undo() ) ; }
      void clear_undo()
      {  m_undo.clear() ; }
      /*-------------------------------------------------------------------*/
      int get_i_ico() const { return( m_i_ico ) ; }
      void set_i_ico( int i_ico ) { m_i_ico = i_ico ; }
      bool is_ico_initialized() { return( m_i_ico != -1 ) ; }
      /*-------------------------------------------------------------------*/
      wxFileName &get_name() { return( m_name ) ; }
      void set_name( const wxFileName &name ) { m_name = name ; }
      const wxFileName &get_name() const { return( m_name ) ; }
      wxFileName &get_new_name() { return( m_new_name ) ; }
      const wxFileName &get_new_name() const { return( m_new_name ) ; }

      wxString get_full_name() const { return( m_name.GetFullName() ) ; }
      wxString get_new_full_name() const
      {  return( m_new_name.GetFullName() ) ; }

      wxString get_full_path() const { return( m_name.GetFullPath() ) ; }
      wxString get_new_full_path() const
      {  return( m_new_name.GetFullPath() ) ; }

      wxString get_path() const { return( m_name.GetPath() ) ; }
      wxString get_new_path() const { return( m_new_name.GetPath() ) ; }
      /*-------------------------------------------------------------------*/
      void set_new_name( const wxFileName &name )
      {  m_new_name = name ; }
      void set_new_name( const wxString &s_name )
      {  m_new_name.Assign( s_name ) ; }
      /*-------------------------------------------------------------------*/
      wxULongLong get_size() const { return( m_ull_size ) ; }
      void set_size( const wxULongLong &ull ) { m_ull_size = ull ; }
      int get_duration() const { return( m_i_duration ) ; }
      void set_duration( int i_duration ) { m_i_duration = i_duration ; }
      /*-------------------------------------------------------------------*/
      const CDir *get_cdir() const { return( m_p_cdir ) ; }
      void set_cdir( CDir *p ) { m_p_cdir = p ; }
      /*-------------------------------------------------------------------*/
      bool is_selected() const { return( m_i_sel_num >= 0 ) ; }
      /*-------------------------------------------------------------------*/
      bool is_oper_err() const { return( !m_s_oper_err.empty() ) ; }
      wxString get_oper_err() const { return( m_s_oper_err ) ; }
      void set_oper_err( const wxString &s )
      {  m_s_oper_err = s ;
         if( !m_s_oper_err.empty() && m_s_oper_err.Last() == '\n' )
         {  m_s_oper_err.RemoveLast() ; }
      }
      void set_oper_err( unsigned long ul_code )
      {  m_s_oper_err = wxSysErrorMsg( ul_code ) ; }
      void reset_oper_err() { m_s_oper_err.clear() ; }

      /*-------------------------------------------------------------------*/
      int  get_sel_num() const { return( m_i_sel_num ) ; }
      /*-------------------------------------------------------------------*/
      bool is_dir() const { return( m_boo_is_dir ) ; }

      /*--------------------------------------------------------------------+
      ! CFile_Base.cpp                                                      !
      +--------------------------------------------------------------------*/
      void reset( bool boo_is_dir ) ;
      bool is_fake() const ;
      bool checksum_info_empty() const ;
      void checksum_info_copy( const CFile &f ) ;
      void init_s_size() ;
      void exec_system_open() const ;
      void init_info_s_track() ;
      void init_s_sel_num() ;
      void set_sel_num( int i_num ) ;
      void unsel() ;
      bool is_selectable() const ;
      wxString compute_abs_path( const wxString &s_path ) const ;
      wxString get_abs_path() const
      { return( compute_abs_path( get_path() ) ) ; }
      wxString get_abs_new_path() const
      { return( compute_abs_path( get_new_path() ) ) ; }
      wxString get_abs_full_path() const ;
      wxString get_abs_new_full_path() const ;
      /*-------------------------------------------------------------------*/
      wxString get_s_val( int i_col ) const ;
      wxLongLong get_ll_val( int i_col ) const ;
      double get_do_val( int i_col ) const ;
      wxDateTime get_dt_val( int i_col ) const ;
      /*-------------------------------------------------------------------*/
      wxString id3v1aud_equiv( int i_col_1, int i_col_2 ) const ;
      wxString get_s_size( const wxUniChar &c_unit ) const ;
      wxString get_s_duration( int i_col, const wxUniChar &c_fmt ) const ;

      /*--( Marked as "const" but they modify the local vars ... )---------*/
      wxString get_md5()   const ;
      wxString get_sha1()  const ;
      wxString get_crc32() const ;
      /*-------------------------------------------------------------------*/
      void names_undo_redo( int i_oper, wxString &s_undo, wxString &s_redo
                          ) const ;
      int reload_info( bool boo_full ) ;
      int do_rename( wxFileName &new_name ) ;
      int do_copy( wxFileName &new_name ) ;
      int do_delete( wxFileName &name ) ;
      int do_delete() { return( do_delete( m_name ) ) ; }
      int oper_do( int i_num, e_otfile ofile, wxFileName &new_name ) ;
      int oper_undo( int i_num, e_otfile ofile ) ;
      int oper_redo( int i_num, e_otfile ofile ) ;

      /*--------------------------------------------------------------------+
      ! CFile_ExpExtr.cpp                                                   !
      +--------------------------------------------------------------------*/
      sr::wxString_cit extr_compute_variable( t_exp_comp_info &eci,
                                              wxString &s_val
                                            ) const ;
      sr::wxString_cit extr_integer( t_exp_comp_info &eci,
                                     int &i_val
                                   ) const ;
      sr::wxString_cit extr_quoted_string( t_exp_comp_info &eci,
                                           wxString &s_val
                                         ) const ;
      sr::wxString_cit extr_param_quoted_string( t_exp_comp_info &eci,
                                                 wxString &s_val
                                               ) const ;
      sr::wxString_cit extr_param_string( t_exp_comp_info &eci,
                                           wxString &s_val
                                        ) const ;
      sr::wxString_cit extr_time( t_exp_comp_info &eci,
                                  int &i_nb_sec
                                ) const ;

      /*--------------------------------------------------------------------+
      ! CFile_ExpMod1.cpp                                                   !
      +--------------------------------------------------------------------*/
      int exp_sel_num( t_exp_comp_info &eci, int i_num, wxString &s_val
                     ) const ;
      int exp_non_num_in_string( t_exp_comp_info &eci,
                                 const wxString  &s_source,
                                 int             i_num,
                                 wxString        &s_val
                               ) const ;
      int exp_num_in_string( t_exp_comp_info &eci,
                             const wxString  &s_source,
                             int             i_num_num,
                             wxString        &s_val
                           ) const ;
      int exp_clipboard_line( t_exp_comp_info &eci,
                              int             i_num,
                              wxString        &s_val
                            ) const ;
      int exp_file_line( t_exp_comp_info &eci,
                         int             i_num,
                         wxString        &s_val
                       ) const ;
      int exp_num_pad( t_exp_comp_info &eci,
                       int             i_num,
                       int             i_pad,
                       wxString        &s_val
                     ) const ;
      int exp_track_pad( t_exp_comp_info &eci,
                         int             i_col_1,
                         int             i_col_2,
                         wxString        &s_val
                       ) const ;
      int exp_date( t_exp_comp_info &eci,
                    wxDateTime      dt,
                    wxString        &s_val
                  ) const ;
      int exp_date( t_exp_comp_info &eci,
                    int             i_col,
                    wxString        &s_val
                  ) const ;
      /*-------------------------------------------------------------------*/
      enum e_srg_rand { SRG_RAND_DIGIT = 1 << 0,
                        SRG_RAND_ALPHA = 1 << 1
                      } ;
      int exp_rand( t_exp_comp_info &eci,
                    int             i_srg_rand,
                    wxString        &s_val
                  ) const ;

      /*--------------------------------------------------------------------+
      ! CFile_ExpMod2.cpp                                                   !
      +--------------------------------------------------------------------*/
      int exp_string_tab_elmt( t_exp_comp_info &eci,
                               wxString        &s_val
                             ) const ;
      int exp_replace( t_exp_comp_info &eci,
                       const wxString  &s_param,
                       wxString        &s_val
                     ) const ;
      int exp_substr( t_exp_comp_info &eci,
                      const wxString  &s_param,
                      wxString        &s_val
                    ) const ;
      int exp_regex( t_exp_comp_info &eci, wxString &s_val ) const ;
      int exp_tr( t_exp_comp_info &eci, wxString &s_val ) const ;

      /*--------------------------------------------------------------------+
      ! CFile_CompVar.cpp                                                   !
      +--------------------------------------------------------------------*/
   private :
      sr::wxString_cit compute_exp_variable_base( t_exp_comp_info &eci,
                                                  wxString &s_val
                                                ) const ;
   public :
      sr::wxString_cit compute_exp_variable( t_exp_comp_info &eci,
                                             wxString &s_val
                                           ) const ;

      /*--------------------------------------------------------------------+
      ! CFile_Comp.cpp                                                      !
      +--------------------------------------------------------------------*/
      sr::wxString_cit compute_sub_exp( t_exp_comp_info &eci ) const ;
      void name_global_modification( wxFileName &name ) const ;
      void new_name_global_modification()
      {  name_global_modification( m_new_name ) ; }
      /*-------------------------------------------------------------------*/
      wxString get_comp_basename( const wxFileName &name ) const ;
      wxString get_comp_extension( const wxFileName &name ) const ;
      /*-------------------------------------------------------------------*/
      wxFileName compute_exp( const wxString &s_exp_src ) ;
      wxString compute_exp_tool( const wxString &s_exp_src ) ;

   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
class CFileWithName
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileWithName( const wxString &s_name )
                   : m_s_name( s_name )
      {  ; }
   /*----------------------------------------------------------------------*/
   private :
      /*-------------------------------------------------------------------*/
      wxString m_s_name ;
   /*----------------------------------------------------------------------*/
   public :
      /*--( Comparing wxFileNames is very heavy )--------------------------*/
      bool operator() ( const CFile &fic ) const
      {  return( fic.get_name().GetFullPath() == m_s_name ) ; }
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
class CFileWithPrefix
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileWithPrefix( const wxString &s_prefix )
                     : m_s_prefix( s_prefix ),
                       m_i_len( s_prefix.size() )
      {  ; }
   /*----------------------------------------------------------------------*/
   private :
      /*-------------------------------------------------------------------*/
      wxString m_s_prefix ;
      int      m_i_len ;
   /*----------------------------------------------------------------------*/
   public :
      /*--( On all files explorers the search seems case insensitive )-----*/
      bool operator() ( const CFile &fic ) const
      {  return( m_s_prefix.CmpNoCase(
                                 fic.get_name().GetFullPath().Left( m_i_len )
                                     ) == 0
               ) ;
      }
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
class CFileComp
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CFileComp( bool boo_natural_comp,
                 bool boo_dir_top, int i_subitem,
                 int i_sort_type, bool boo_asc,
                 CRunInfoMessage &runinfomessage
               ) : m_boo_natural_comp( boo_natural_comp ),
                   m_boo_dir_top( boo_dir_top ),
                   m_i_subitem( i_subitem ), m_i_sort_type( i_sort_type ),
                   m_boo_asc( boo_asc ), m_runinfomessage( runinfomessage )
      {  init_fnc_comp() ; }
   /*----------------------------------------------------------------------*/
   private :
      /*-------------------------------------------------------------------*/
      bool            m_boo_natural_comp ;
      bool            m_boo_dir_top      ;
      int             m_i_subitem        ;
      int             m_i_sort_type      ;
      bool            m_boo_asc          ;
      CRunInfoMessage &m_runinfomessage  ;

      /*--------------------------------------------------------------------+
      ! A specific sort function is done for the name because it is surely  !
      ! the most used sort item. This function avoids useless calls ...     !
      +--------------------------------------------------------------------*/
      int ( CFileComp::*m_p_fnc_comp_name )( const CFile &f1, const CFile &f2
                                           ) const ;
      int ( CFileComp::*m_p_fnc_comp_s )( const CFile &f1, const CFile &f2
                                        ) const ;
      int ( CFileComp::*m_p_fnc_comp )( const CFile &f1, const CFile &f2
                                      ) const ;

      /*-------------------------------------------------------------------*/
      int _natural_comp_name( const CFile &f1, const CFile &f2 ) const
      {  return( sr::natural_comp( f1.get_name().GetFullPath(),
                                   f2.get_name().GetFullPath()
                                 )
               ) ;
      }
      /*-------------------------------------------------------------------*/
      int _comp_name( const CFile &f1, const CFile &f2 ) const
      {  return( sr::ic_comp( f1.get_name().GetFullPath(),
                              f2.get_name().GetFullPath()
                            )
               ) ;
      }
      /*-------------------------------------------------------------------*/
      int _natural_comp_s( const CFile &f1, const CFile &f2 ) const
      {  /*----------------------------------------------------------------*/
         return( sr::natural_comp( f1.get_s_val( m_i_subitem ),
                                   f2.get_s_val( m_i_subitem )
                                 )
               ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      int _comp_s( const CFile &f1, const CFile &f2 ) const
      {  /*----------------------------------------------------------------*/
         return( sr::ic_comp( f1.get_s_val( m_i_subitem ),
                              f2.get_s_val( m_i_subitem )
                            )
               ) ;
         /*----------------------------------------------------------------*/
      }

      /*-------------------------------------------------------------------*/
      int comp_sel_num( const CFile &f1, const CFile &f2 ) const
      {  /*----------------------------------------------------------------*/
         if(  f1.is_selected() && !f2.is_selected() ) { return( +1 ) ; }
         if( !f1.is_selected() &&  f2.is_selected() ) { return( -1 ) ; }
         if(  f1.is_selected() &&  f2.is_selected() )
         {  return( f1.get_sel_num() - f2.get_sel_num() ) ; }
         /*----------------------------------------------------------------*/
         return( 0 ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      int comp_size( const CFile &f1, const CFile &f2 ) const
      {  /*----------------------------------------------------------------*/
         if( f1.get_size() < f2.get_size() ) { return( -1 ) ; }
         if( f1.get_size() > f2.get_size() ) { return( +1 ) ; }
         /*----------------------------------------------------------------*/
         return( 0 ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      int comp_dt( const CFile &f1, const CFile &f2 ) const
      {
         /*----------------------------------------------------------------*/
         const wxDateTime dt1 = f1.get_dt_val( m_i_subitem ) ;
         const wxDateTime dt2 = f2.get_dt_val( m_i_subitem ) ;
         /*----------------------------------------------------------------*/
         if( dt1.IsValid() )
         {  if( !dt2.IsValid() || dt1 < dt2 ) { return( -1 ) ; }
            if(  dt1 > dt2 ) { return( +1 ) ; }
         }
         else
         if( dt2.IsValid() ) { return( +1 ) ; }

         /*--( By default, compare the strings )---------------------------*/
         return( ( this->*m_p_fnc_comp_s )( f1, f2 ) ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      int comp_ll( const CFile &f1, const CFile &f2 ) const
      {  /*----------------------------------------------------------------*/
         wxLongLong ll_1 = f1.get_ll_val( m_i_subitem ) ;
         wxLongLong ll_2 = f2.get_ll_val( m_i_subitem ) ;
         /*----------------------------------------------------------------*/
         if( ll_1 < ll_2 ) { return( -1 ) ; }
         if( ll_1 > ll_2 ) { return( +1 ) ; }
         /*----------------------------------------------------------------*/
         return( 0 ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      int comp_do( const CFile &f1, const CFile &f2 ) const
      {  /*----------------------------------------------------------------*/
         double do_1 = f1.get_do_val( m_i_subitem ) ;
         double do_2 = f2.get_do_val( m_i_subitem ) ;
         /*----------------------------------------------------------------*/
         if( do_1 < do_2 ) { return( -1 ) ; }
         if( do_1 > do_2 ) { return( +1 ) ; }
         /*----------------------------------------------------------------*/
         return( 0 ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      void init_fnc_comp()
      {
         /*----------------------------------------------------------------*/
         if( m_boo_natural_comp )
         {  m_p_fnc_comp_name = &CFileComp::_natural_comp_name ;
            m_p_fnc_comp_s    = &CFileComp::_natural_comp_s ;
         }
         else
         {  m_p_fnc_comp_name = &CFileComp::_comp_name ;
            m_p_fnc_comp_s    = &CFileComp::_comp_s ;
         }
         /*--( For some columns the sort types are specific )--------------*/
         if( m_i_subitem == COL_NONE_NAME )
         {  m_p_fnc_comp = m_p_fnc_comp_name ; }
         else
         if( m_i_subitem == COL_NONE_SEL_NUM )
         {  m_p_fnc_comp = &CFileComp::comp_sel_num ; }
         else
         if( m_i_subitem == COL_BASE_SIZE )
         {  m_p_fnc_comp = &CFileComp::comp_size ; }
         else
         if( m_i_sort_type == SORT_INTEGER )
         {  m_p_fnc_comp = &CFileComp::comp_ll ; }
         else
         if( m_i_sort_type == SORT_DOUBLE )
         {  m_p_fnc_comp = &CFileComp::comp_do ; }
         else
         if( m_i_sort_type == SORT_DATETIME )
         {  m_p_fnc_comp = &CFileComp::comp_dt ; }
         else /*--( By default, the sort is done on the string value )-----*/
         {  m_p_fnc_comp = m_p_fnc_comp_s ; }
         /*----------------------------------------------------------------*/
      }

   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      bool operator() ( const CFile &f1, const CFile &f2 ) const
      {
         /*----------------------------------------------------------------*/
         m_runinfomessage.display() ;

         /*--( Directories first ? )---------------------------------------*/
         if( m_boo_dir_top )
         {
            /*-------------------------------------------------------------*/
            if( f1.is_dir() && !f2.is_dir() ) { return( true ) ; }
            /*-------------------------------------------------------------*/
            if( f2.is_dir() && !f1.is_dir() ) { return( false ) ; }
            /*-------------------------------------------------------------*/
         }

         /*----------------------------------------------------------------*/
         int i_res = ( this->*m_p_fnc_comp )( f1, f2 ) ;
         /*--( By default or same value, the sort is done on the name )----*/
         if( i_res == 0 ) { i_res = ( this->*m_p_fnc_comp_name )( f1, f2 ) ;}

         /*----------------------------------------------------------------*/
         return(    (  m_boo_asc && i_res < 0 )
                 || ( !m_boo_asc && i_res > 0 )
               ) ;
         /*----------------------------------------------------------------*/
      }
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
#endif // CFILE_H
/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                          End of file CFile.h                              !
+==========================================================================*/
