/*====================================+=====================================+
! File CCombo.h                       ! Copyright (C) 2002-2013 Remi PASCAL !
+-------------------------------------+-------------------------------------+
! This file is part of Siren.                                               !
! Siren is free software: you can redistribute it and/or modify it under    !
! the terms of the GNU General Public License as published by the Free      !
! Software Foundation, either version 3 of the License, or any later        !
! version.                                                                  !
! Siren is distributed in the hope that it will be useful, but WITHOUT ANY  !
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS !
! FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more    !
! details.                                                                  !
! You should have received a copy of the GNU General Public License along   !
! with Siren. If not, see <http://www.gnu.org/licenses/>.                   !
+==========================================================================*/



/*-------------------------------------------------------------------------*/
#ifndef CCOMBO_H
#define CCOMBO_H
/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
#include <wx/frame.h>
#include <wx/notebook.h>
#include <wx/listbox.h>
#include <wx/combo.h>
#include <wx/filename.h>
#include <wx/textctrl.h>
#include "CDragAndDrop.h"
#include "common/sr_lib.h"
/*-------------------------------------------------------------------------*/



/*--------------------------------------------------------------------------+
!                                                                           !
! Why using a wxComboPopup/wxComboCtrl couple rather than a standard        !
! wxComboBox ?                                                              !
!                                                                           !
! I want the user to be able to delete elements from the combo list.        !
! Under GTK with a standard wxCombo, the events used when the list has      !
! the focus are not available to wx events.                                 !
!                                                                           !
+--------------------------------------------------------------------------*/
class CListBoxComboPopup : public wxListBox, public wxComboPopup
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CListBoxComboPopup() : m_i_val( 0 ),            // for cppcheck
                             m_boo_case_sens( false ) // for cppcheck

      {  ; }
   /*----------------------------------------------------------------------*/
   private :
      /*-------------------------------------------------------------------*/
      void choose_current_list_element() ;
      void delete_current_list_element() ;
      /*-------------------------------------------------------------------*/
      int  m_i_val ;
      bool m_boo_case_sens ; // Content case sensitive ? for path & filter

   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      void OnMouseMove( wxMouseEvent &event ) ;
      void OnMouseLeftClick( wxMouseEvent &event ) ;
      void OnMouseRightClick( wxMouseEvent &event ) ;
      void OnKeyDown( wxKeyEvent &event ) ;
      /*-------------------------------------------------------------------*/
      virtual void Init()
      {  m_i_val = -1 ; m_boo_case_sens = true ; }
      /*--( To manage different keys: delete, space ... )------------------*/
      virtual bool Create( wxWindow *parent )
      {  return( wxListBox::Create( parent, wxID_ANY, wxPoint( 0, 0 ),
                                    wxDefaultSize, 0, NULL, wxWANTS_CHARS
                                  )
               ) ;
      }
      /*-------------------------------------------------------------------*/
      virtual wxWindow *GetControl()
      {  return( this ) ; }
      /*-------------------------------------------------------------------*/
      virtual void SetStringValue( const wxString &s )
      {  /*----------------------------------------------------------------*/
         int i_sel = wxListBox::FindString( s, true ) ;
         if( i_sel != wxNOT_FOUND ) { wxListBox::Select( i_sel ) ; }
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      virtual wxString GetStringValue() const
      {  /*----------------------------------------------------------------*/
         if( m_i_val >= 0 ) { return( wxListBox::GetString( m_i_val ) ) ; }
         return( wxEmptyString ) ;
         /*----------------------------------------------------------------*/
      }
      /*-------------------------------------------------------------------*/
      void add_value_to_list( const wxString &s ) ;
      void move_current_value_to_top() ;
      /*-------------------------------------------------------------------*/
      void set_case_sens( bool boo_case_sens )
      {  m_boo_case_sens = boo_case_sens ; }

   /*----------------------------------------------------------------------*/
   private :
      /*-------------------------------------------------------------------*/
      DECLARE_EVENT_TABLE()
   /*----------------------------------------------------------------------*/
} ;

/*--------------------------------------------------------------------------+
!                                                                           !
! Why defining the code around the "focus" set/kill ?                       !
!                                                                           !
! In the main window typing Enter in one of the three combos (exp, dir &    !
! fil) implies actions. The same actions have to be executed when the combo !
! looses the focus (this was present in Siren's versions < 3).              !
! To achieve this:                                                          !
! - on set focus, the current value is saved                                !
! - on kill focus if the value changed and the combo has the flag           !
!   "wxTE_PROCESS_ENTER" an event "wxEVT_COMMAND_TEXT_ENTER" is sent.       !
!                                                                           !
+--------------------------------------------------------------------------*/
class CCombo : public wxComboCtrl
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CCombo( wxWindow *parent, wxWindowID id,
              const wxString &value = wxEmptyString,
              const wxPoint &pos = wxDefaultPosition,
              const wxSize &size = wxDefaultSize,
              int WXUNUSED( n ) = 0,
              const wxString WXUNUSED( choices )[] = NULL,
              long style = 0,
              bool boo_case_sens = false
            ) ;
      /*-------------------------------------------------------------------*/
      ~CCombo()
      {  ; }
   /*----------------------------------------------------------------------*/
   protected :
      /*-------------------------------------------------------------------*/
      CListBoxComboPopup *m_popup ;
      /*--------------------------------------------------------------------+
      ! In some combos, after a value has been chosen or when the combo     !
      ! looses the focus, an action associated to the Enter key has to be   !
      ! executed.                                                           !
      ! For example for the main combo expression the new names have to be  !
      ! recomputed when :                                                   !
      ! - an expression is chosen in the list                               !
      ! - the expression is edited without validation and the combo looses  !
      !   the focus.                                                        !
      ! On the contrary for the combo used in the search/selection dialogs  !
      ! when don't want this behaviour.                                     !
      ! By default this feature will be on. Use the "set" to deactivate it  !
      +--------------------------------------------------------------------*/
      bool     m_boo_simulate_enter_key ;
      /*--( Value of the combo when the combo received focus )-------------*/
      wxString m_s_setfocus_val ;
      /*-------------------------------------------------------------------*/
      wxString m_s_help_page ;
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      void OnKeyDown( wxKeyEvent &event ) ;
      void OnHelp( wxHelpEvent &event ) ;
      void OnFocusSet( wxFocusEvent &event ) ;
      void OnFocusKill( wxFocusEvent &event ) ;
      /*-------------------------------------------------------------------*/
      void Append( const wxArrayString &as )
      {  if( as.GetCount() > 0 )
         {  m_popup->InsertItems( as, m_popup->GetCount() ) ; }
      }
      void Append( const wxString &s )
      {  m_popup->Insert( s, m_popup->GetCount() ) ; }
      wxArrayString GetStrings() const
      {  return( m_popup->GetStrings() ) ; }
      /*-------------------------------------------------------------------*/
      void add_value_to_list( const wxString &s )
      {  m_popup->add_value_to_list( s ) ; }
      void add_current_value_to_list()
      {  m_popup->add_value_to_list( GetValue() ) ; }
      void move_current_value_to_top()
      {  m_popup->move_current_value_to_top() ; }
      /*-------------------------------------------------------------------*/
      bool set_simulate_enter_key() const
      {  return( m_boo_simulate_enter_key ) ; }
      void set_simulate_enter_key( bool boo_val )
      {  m_boo_simulate_enter_key = boo_val ; }
      /*-------------------------------------------------------------------*/
      void simulate_enter_key_pressed() ;
      /*-------------------------------------------------------------------*/
      wxString get_help_page() const
      {  return( m_s_help_page ) ; }

   /*----------------------------------------------------------------------*/
   private :
      /*-------------------------------------------------------------------*/
      DECLARE_EVENT_TABLE()
   /*----------------------------------------------------------------------*/
} ;

/*--( An empty value is kept at the end of the list to clear curr value )--*/
class CComboWithCleaner : public CCombo
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CComboWithCleaner( wxWindow *parent, wxWindowID id,
                         const wxString &value = wxEmptyString,
                         const wxPoint &pos = wxDefaultPosition,
                         const wxSize &size = wxDefaultSize,
                         int n = 0, const wxString choices[] = NULL,
                         long style = 0,
                         bool boo_case_sens = false
                       )
                     : CCombo( parent, id, value, pos, size, n, choices,
                               style, boo_case_sens
                             )
      {  CCombo::Append( wxEmptyString ) ; }
      /*-------------------------------------------------------------------*/
      ~CComboWithCleaner()
      {  ; }

      /*-------------------------------------------------------------------*/
   public :
      /*--( Delete the empty string, append array and readd the empty one )*/
      void Append( const wxArrayString &strings )
      {
         /*----------------------------------------------------------------*/
         if( strings.IsEmpty() ) { return ; }
         /*----------------------------------------------------------------*/
         m_popup->Delete( m_popup->GetCount() - 1 ) ;
         m_popup->Append( strings ) ;
         m_popup->Append( wxEmptyString ) ;
         /*----------------------------------------------------------------*/
      }
      /*--( Remove the empty string before returning the array ... )-------*/
      wxArrayString GetStrings() const
      {
         /*----------------------------------------------------------------*/
         wxArrayString strings( m_popup->GetStrings() ) ;
         /*----------------------------------------------------------------*/
         if( !strings.IsEmpty() )
         {  strings.RemoveAt( strings.GetCount() - 1 ) ; }
         return( strings ) ;
         /*----------------------------------------------------------------*/
      }
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
class CComboFil : public CComboWithCleaner
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CComboFil( wxWindow *parent, wxWindowID id,
                 const wxString &value = wxEmptyString,
                 const wxPoint &pos = wxDefaultPosition,
                 const wxSize &size = wxDefaultSize,
                 int n = 0, const wxString choices[] = NULL,
                 long style = 0
               ) ;

   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
class CComboDir : public CCombo
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CComboDir( wxWindow *parent, wxWindowID id,
                 const wxString &value = wxEmptyString,
                 const wxPoint &pos = wxDefaultPosition,
                 const wxSize &size = wxDefaultSize,
                 int n = 0, const wxString choices[] = NULL,
                 long style = 0
               ) ;

   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      void init_drop_target()
      {  GetTextCtrl()->SetDropTarget( new CDropDir() ) ; }
      void reset_drop_target()
      {  GetTextCtrl()->SetDropTarget( NULL ) ; }

   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/
class CComboExp : public CComboWithCleaner
{
   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      CComboExp( wxWindow *parent, wxWindowID id,
                 const wxString &value = wxEmptyString,
                 const wxPoint &pos = wxDefaultPosition,
                 const wxSize &size = wxDefaultSize,
                 int n = 0, const wxString choices[] = NULL,
                 long style = 0
               ) ;

   /*----------------------------------------------------------------------*/
   public :
      /*-------------------------------------------------------------------*/
      void OnKeyDown( wxKeyEvent &event ) ;

   /*----------------------------------------------------------------------*/
   private :
      /*-------------------------------------------------------------------*/
      DECLARE_EVENT_TABLE()
   /*----------------------------------------------------------------------*/
} ;

/*-------------------------------------------------------------------------*/



/*-------------------------------------------------------------------------*/
#endif // CCOMBO_H
/*-------------------------------------------------------------------------*/



/*==========================================================================+
!                         End of file CCombo.h                              !
+==========================================================================*/
